"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: 'Creates transliterated assemblies from jsii-enabled npm packages',
            environment,
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(15),
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.ISOLATED },
        });
        const repository = props.vpcEndpoints
            ? (_c = props.codeArtifact) === null || _c === void 0 ? void 0 : _c.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(lambda);
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        // The handler reads & writes to this bucket.
        bucket.grantRead(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.DOCS_KEY_SUFFIX_ANY}`);
        // Creating the event chaining
        lambda.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED],
            filters: [{ prefix: constants.STORAGE_KEY_PREFIX, suffix: constants.ASSEMBLY_KEY_SUFFIX }],
        }));
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
        this.alarmDeadLetterQueueNotEmpty = lambda.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The transliteration function failed for one or more packages',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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