"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const clients = new Map();
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return withFakeHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        const created = new Array();
        for (const record of event.Records) {
            // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
            // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
            // that would not replace `@` in the position where it is in the keys... So we have to work on
            // the URI components instead.
            const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
            const [, packageName, packageVersion] = (_a = inputKey.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
            if (packageName == null) {
                throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
            }
            const packageFqn = `${packageName}@${packageVersion}`;
            const client = (clients.has(record.awsRegion)
                ? clients
                : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
            console.log(`Source Bucket:  ${record.s3.bucket.name}`);
            console.log(`Source Key:     ${inputKey}`);
            console.log(`Source Version: ${record.s3.object.versionId}`);
            console.log(`Fetching assembly: ${inputKey}`);
            const assemblyResponse = await client.getObject({ Bucket: record.s3.bucket.name, Key: inputKey }).promise();
            if (!assemblyResponse.Body) {
                throw new Error(`Response body for assembly at key ${inputKey} is empty`);
            }
            const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
            const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
            const targetLanguages = Object.keys(assembly.targets);
            async function generateDocs(lang) {
                const uploads = new Map();
                const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
                function renderAndDispatch(submodule) {
                    var _a;
                    console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                    const page = docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render();
                    const key = inputKey.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                    console.log(`Uploading ${key}`);
                    const upload = client.putObject({
                        Bucket: record.s3.bucket.name,
                        Key: key,
                        Body: page,
                        ContentType: 'text/html',
                        Metadata: {
                            'Origin-Version-Id': (_a = record.s3.object.versionId) !== null && _a !== void 0 ? _a : 'N/A',
                            'Lambda-Log-Group': context.logGroupName,
                            'Lambda-Log-Stream': context.logStreamName,
                            'Lambda-Run-Id': context.awsRequestId,
                        },
                    }).promise();
                    uploads.set(key, upload);
                }
                renderAndDispatch();
                for (const submodule of submodules) {
                    renderAndDispatch(submodule);
                }
                for (const [key, upload] of uploads.entries()) {
                    const response = await upload;
                    created.push({ bucket: record.s3.bucket.name, key: key, versionId: response.VersionId });
                    console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
                }
            }
            for (const language of [...targetLanguages, language_1.DocumentationLanguage.TYPESCRIPT.toString()]) {
                try {
                    // we await per language here because every language will eventually be moved to
                    // a separate function.
                    await generateDocs(language);
                }
                catch (e) {
                    if (e instanceof docgen.UnsupportedLanguageError) {
                        console.warn(`Skipping '${language}' for ${packageFqn} since it is not yet supported by jsii-docgen`);
                    }
                    else if (e instanceof language_1.UnsupportedLanguageError) {
                        console.warn(`Skipping '${language}' for ${packageFqn} since it is not yet supported by construct-hub`);
                    }
                    else {
                        console.error(`Failed generating ${language} documentation for ${packageFqn}: ${e}`);
                    }
                }
            }
        }
        return created;
    });
}
exports.handler = handler;
async function withFakeHome(cb) {
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
    }
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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