"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const process = require("process");
const tls = require("tls");
const AWS = require("aws-sdk");
/**
 * Obtains the TLS certificate used by the HTTPS endpoint designated by the
 * `HTTPS_ENDPOINT` environment variable, checks how many validity days remain
 * for this endpoint, and produces a CloudWatch metric under the namespace
 * designated by `METRIC_NAMESPACE`, named as specified by `METRIC_NAME`.
 *
 * If the certificate is past expiration, the metric will be trimmed to `0`
 * instead of turning into negative values.
 */
async function handler(_event, _context) {
    const endpoint = requireEnv('HTTPS_ENDPOINT');
    const metricNamespace = requireEnv('METRIC_NAMESPACE');
    const metricName = requireEnv('METRIC_NAME');
    const now = new Date();
    const daysRemaining = await tlsValidDaysRemaining(endpoint, now);
    console.log(`The certificate has ${daysRemaining} remaining validity days`);
    return new AWS.CloudWatch()
        .putMetricData({
        Namespace: metricNamespace,
        MetricData: [
            // One metric entry with the DomainName dimension set
            {
                Dimensions: [{ Name: 'DomainName', Value: endpoint }],
                MetricName: metricName,
                Timestamp: now,
                Unit: 'Count',
                Value: daysRemaining,
            },
        ],
    })
        .promise();
}
exports.handler = handler;
function requireEnv(name) {
    const result = process.env[name];
    if (!result) {
        throw new Error(`Missing required environment variable "${name}"`);
    }
    return result;
}
/**
 * Obtains the remaining validity days from the provided `endpoint`. This
 * establishes a TLS connection to the `endpoint` on port `443`, and assesses
 * the `valid_to` value therein. This function also works with endpoints that
 * require SNI in order to present a certificate (as is the case for API Gateway
 * and CloudFront, for example). It does however not bother with ALPN protocols.
 *
 * @param endpoint the domain name for which a certificate is needed.
 * @param now      the date to use as the `now` reference point.
 *
 * @returns the amount of days the certificate is still valid for, or `0` if the
 *          certificate has already expired.
 */
function tlsValidDaysRemaining(endpoint, now) {
    console.log(`Checking remaining validity time for certificate on ${endpoint}:443`);
    const sock = tls.connect({
        host: endpoint,
        port: 443,
        servername: endpoint,
    });
    // "Cleanly" terminates the socket connection, so it does not leak
    const closeThen = (cb) => sock.end(() => {
        sock.destroy();
        sock.unref();
        cb();
    });
    return new Promise((resolve, reject) => {
        const ok = (val) => closeThen(() => resolve(val));
        const ko = (reason) => closeThen(() => reject(reason));
        try {
            sock.once('error', (err) => err.code === 'CERT_HAS_EXPIRED' ? ok(0) : ko(err));
            sock.once('secureConnect', () => {
                const cert = sock.getPeerCertificate();
                console.log(`Secure connection established: ${cert.fingerprint256} is valid from ${cert.valid_from} until ${cert.valid_to}`);
                const remainingMillis = new Date(cert.valid_to).getTime() - now.getTime();
                if (remainingMillis < 0) {
                    console.error(`The certificate expired ${remainingMillis}ms ago!`);
                    return ok(0);
                }
                // Converting to days (there are 86,400,000 milliseconds in a day)
                ok(remainingMillis / 86400000);
            });
        }
        catch (e) {
            ko(e);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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