"""
Contains the functions that will download the market stack data.
"""
import asyncio
import json
from typing import List, Tuple
from itertools import chain

import aiohttp
from aiohttp import ClientSession
from pymarket.responses import Response, Pagination, Data
from pymarket.helper import chunk
from pymarket.exceptions import *


async def _download_batch(url: str, session: ClientSession) -> Response | None:
    """
    Downloads a single batch [1, 100] symbols.

    :param url: The already constructed request url
    :param session: The client session provided by aiohttp
    :return: A Response object on success (or partial success), None on error, raises on fatal error.
    """

    async with session.get(url) as response:
        res = await response.content.read()
        content = res.decode()

        match ApiCodes(response.status):
            # Ok
            case ApiCodes.OK:
                return Response.from_json(content)

            # Non-fatal error
            case ApiCodes.BATCH:
                return None

            # Fatal errors:
            case ApiCodes.TOKEN:
                raise TokenException(Error.from_dict(json.loads(content)["error"]))

            case ApiCodes.ACCESS:
                raise AccessException(Error.from_dict(json.loads(content)["error"]))

            case ApiCodes.ENDPOINT:
                raise EndpointException(Error.from_dict(json.loads(content)["error"]))

            case ApiCodes.LIMIT:
                raise LimitException(Error.from_dict(json.loads(content)["error"]))

            case ApiCodes.INTERNAL:
                raise InternalException(Error.from_dict(json.loads(content)["error"]))

            case _:
                raise UnspecifiedException("An unknown error occurred downloading data.")


async def _spawner(url: str, symbols: List[str]):
    PAGINATION_LIMIT = 1000

    # Split into batches with max length of 100 each (api requirement)
    batches = list(chunk(symbols, PAGINATION_LIMIT))
    batch_count = len(batches)

    connector = aiohttp.TCPConnector(limit=batch_count)
    async with aiohttp.ClientSession(connector=connector) as session:
        tasks = []
        for batch in batches:
            url = f"{url}&symbols={','.join(batch)}"
            tasks.append(asyncio.ensure_future(_download_batch(url, session)))

        return await asyncio.gather(*tasks, return_exceptions=False)


def download(url: str, *symbols: str) -> Tuple[Response, List[str]]:
    """
    Spawns asyncio workers.
    :param url: The request url generated by one of the features.
    :param symbols: A list of all symbols to download
    :return: A tuple containing the response and a list of possibly failed symbols.
    """

    # Await for all responses
    res = asyncio.run(_spawner(url, list(symbols)))
    responses = [response for response in res if response is not None]

    if len(responses) == 0:
        raise NoDataException(f"No valid data downloaded, a total of {res.count(None)} batches failed.")

    # Merge data
    data: List[Data] = list(chain(*[response.data for response in responses]))

    # Merge all paginations into one
    pagination = Pagination(
        limit=responses[0].pagination.limit,
        offset=responses[0].pagination.offset,
        count=responses[0].pagination.count,
        total=len(data)
    )

    # Succeeded symbols
    succeeded = [d.symbol for d in data]
    failed = [symbol for symbol in symbols if symbol not in succeeded]

    return Response(pagination, data), failed
