# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_slugify_processor', 'django_slugify_processor.templatetags']

package_data = \
{'': ['*']}

install_requires = \
['Django>=1.11']

extras_require = \
{'test:python_version < "3"': ['django-extensions<2.2.6'],
 'test:python_version >= "3"': ['django-extensions']}

setup_kwargs = {
    'name': 'django-slugify-processor',
    'version': '0.9.0b2',
    'description': 'pipeline for slugification edgecases in django',
    'long_description': 'pipeline for handling slugification edgecases\n\n|pypi| |build-status| |docs| |coverage| |license|\n\nWhat are slugs?\n===============\n\n*Slugs* are URL\'s, typically generated from post titles, that you want to\nbe both human readable and a valid URL. They are SEO friendly.\n\nDjango provides a `slugify function <https://docs.djangoproject.com/en/1.11/ref/utils/#django.utils.text.slugify>`__\nin ``django.utils.text.slugify`` which is also made available as a\n`default filter <https://github.com/django/django/blob/1.11.4/django/template/defaultfilters.py#L232>`__.\n\nDjango slugs can be automatically generated in django models via packages\nsuch as:\n\n- `django-autoslug <https://pypi.python.org/pypi/django-autoslug>`__\n  (`docs <https://pythonhosted.org/django-autoslug/>`__)\n  (`github <https://github.com/neithere/django-autoslug>`__)\n- `django-extensions <https://pypi.python.org/pypi/django-extensions>`__\n  via `AutoSlugField <https://django-extensions.readthedocs.io/en/latest/field_extensions.html>`__\n  (`docs <https://django-extensions.readthedocs.io/en/latest/>`__)\n  (`github <https://github.com/django-extensions/django-extensions>`__)\n\nThe problem\n===========\n\nThis project is based on an article from `devel.tech <https://devel.tech>`__\ncovering `django\'s import strings\n<https://devel.tech/tips/n/djms3tTe/demystifying-djangos-import-strings/>`__.\n\nCorner cases exist with slugification. For instance:\n\n================  =============================  =============\nTerm              ``django.utils.text.slugify``  What you want\n================  =============================  =============\nC                 c (correct)                    n/a\nC++               c                              cpp\nC#                c                              c-sharp\n================  =============================  =============\n\nTo make matters worse, if using a specialized model field like\n``AutoSlugField`` from django-autoslug or django-extensions, the default\nbehavior may be to name the slugs for C++ and C# to "c-1", "c-2" after "c" is\ntaken.\n\nHere\'s another case, acronyms / shorthands:\n\n==================  =============================  ===================\nTerm                ``django.utils.text.slugify``  What you (may) want\n==================  =============================  ===================\nNew York City       new-york-city                  nyc\nY Combinator        y-combinator                   yc \nPortland            portland                       pdx\nTexas               texas                          tx\n\\$                  \'\' (empty)                     usd, aud, etc?\nUS$                 us                             usd\nA$                  a                              aud\nbitcoin             bitcoin                        btc\nUnited States       united-states                  usa\nLeague of Legends   league-of-legends              league\nApple® iPod Touch   apple-ipod-touch               ipod-touch\n==================  =============================  ===================\n\nEach website and niche has its own edge cases for slugs. So we need a solution\nthat can scale, where you can craft your own functions.\n\nHow django-slugify-processor helps\n==================================\n\nThis builds on top of `django.utils.text.slugify`_ to handle your django\nproject\'s edgecases. By default, django-slugify-processor will be a pass\nthrough to django\'s default behavior. Adding slugification functions via\nyour Django project\'s settings file allows you to adjust.\n\n.. _django.utils.text.slugify: https://docs.djangoproject.com/en/1.11/ref/utils/#django.utils.text.slugify\n\nInstallation\n============\n\n.. code-block:: sh\n\n   $ pip install django-slugify-processor\n\nConfigure\n=========\n\nTo create a processor, create a function that accepts a string, and\nreturns a string. Assume this is *project/app/slugify_processors.py*:\n\n.. code-block:: python\n\n   def my_processor(value):\n      value = value.replace(\'++\', \'pp\')\n      return value\n\nInside of your settings, add a ``SLUGIFY_PROCESSORS`` list of strings\nthat points to the function. Anything that\'s compatible with\n`import_string <https://docs.djangoproject.com/en/1.11/ref/utils/#django.utils.module_loading.import_string>`__,\nin your settings file:\n\n.. code-block:: python\n\n   SLUGIFY_PROCESSORS = [\n      \'project.app.slugify_processors.my_processor\'\n   ]\n\nUsage\n=====\n\nIn normal django code\n---------------------\n\nImport ``slugify`` from ``django_slugify_processor.text``:\n\n.. code-block:: python\n\n   from django_slugify_processor.text import slugify\n\n   print(slugify(\'C++\'))\n   > \'cpp\'\n\nTemplate code\n-------------\n\ndjango-slugify-processor is designed to override the built-in``slugify``\nfilter.\n\nvia load\n""""""""\n\nYou can load by default via ``{% load django_slugify_processor %}``\nin your template.\n\nIn your settings ``INSTALLED_APPS``:\n\n.. code-block:: python\n\n   INSTALLED_APPS = [\n       \'django_slugify_processor\'\n   ]\n\nIn your template:\n\n.. code-block:: django\n\n   {% load slugify_processor %}\n   {{"C++"|slugify}}\n\nvia built-in\n""""""""""""\n\nTo make this available in all templates, in the ``OPTIONS`` of your\ntemplate engine, add ``django_slugify_processor.template_tags``:\n\n.. code-block:: python\n\n   TEMPLATES = [{\n       \'BACKEND\': \'django.template.backends.django.DjangoTemplates\',\n       \'OPTIONS\': {\n           \'builtins\': [\n               \'django_slugify_processor.templatetags.slugify_processor\',\n           ],\n       },\n   }]\n\nFrom within the template file:\n\n.. code-block:: django\n\n   {{"C++"|slugify}}\n\nOutput should be: cpp\n\nModels\n------\n\nFor the most up to date documentation, view the documetation for the\nplugin you\'re using (e.g. django-autoslug or django-extensions).\n\nTo use django-slugify-processor\'s ``slugify`` instead of django\'s default,\nthere will be a field option to use the function.\n\ndjango-extensions\n"""""""""""""""""\n\nTested with 1.9.7 (2017-11-26):\n\n.. code-block:: python\n\n   from django.db import models\n\n   from django_extensions.db.fields import AutoSlugField\n   from django_slugify_processors.text import slugify\n\n   class MyModel(models.Model):\n       title = models.CharField(max_length=255)\n       slug = AutoSlugField(\n           populate_from=\'title\',\n           slugify_function=slugify\n       )\n\ndjango-autoslug\n"""""""""""""""\n\nTested with 1.9.3 (2017-11-26):\n\n.. code-block:: python\n\n   from django.db import models\n\n   from autoslug import AutoSlugField\n   from django_slugify_processors.text import slugify\n\n   class MyModel(models.Model):\n       title = models.CharField(max_length=255)\n       slug = AutoSlugField(\n           populate_from=\'title\',\n           slugify=slugify\n       )\n\nCredits\n=======\n\n- tox.ini based off DRF\'s (BSD 2-clause licensed)\n- yapf configuration based off RTD / devel.tech\'s (MIT-licensed)\n\nProject details\n===============\n\n==============  ============================================================================\npython support  2.7, >= 3.6, pypy3\ndjango support  3.0, >2, 1.11\nSource          https://github.com/develtech/django-slugify-processor\nDocs            https://django-slugify-processor.git-pull.com\nAPI             https://django-slugify-processor.git-pull.com/api.html\nChangelog       https://django-slugify-processor.git-pull.com/history.html\nIssues          https://github.com/develtech/django-slugify-processor/issues\nTest Coverage   https://codecov.io/gh/develtech/django-slugify-processor\npypi            https://pypi.python.org/pypi/django-slugify-processor\nOpen Hub        https://www.openhub.net/p/django-slugify-processor\nLicense         MIT\ngit repo        .. code-block:: bash\n\n                   $ git clone https://github.com/develtech/django-slugify-processor.git\ninstall stable  .. code-block:: bash\n\n                   $ pip install django-slugify-processor\ninstall dev     .. code-block:: bash\n\n                   $ git clone https://github.com/develtech/django-slugify-processor.git\n                   $ cd ./django-slugify-processor\n                   $ pipenv install --dev --skip-lock\n                   $ pipenv shell\n\ntests           .. code-block:: bash\n\n                   $ make test\n==============  ============================================================================\n\n.. |pypi| image:: https://img.shields.io/pypi/v/django-slugify-processor.svg\n    :alt: Python Package\n    :target: http://badge.fury.io/py/django-slugify-processor\n\n.. |docs| image:: https://github.com/develtech/django-slugify-processor/workflows/docs/badge.svg\n   :alt: Docs\n   :target: https://github.com/develtech/django-slugify-processor/actions?query=workflow%3Adocs\n\n.. |build-status| image:: https://github.com/develtech/django-slugify-processor/workflows/tests/badge.svg\n   :alt: Build Status\n   :target: https://github.com/develtech/django-slugify-processor/actions?query=workflow%3Atests\n\n.. |coverage| image:: https://codecov.io/gh/develtech/django-slugify-processor/branch/master/graph/badge.svg\n    :alt: Code Coverage\n    :target: https://codecov.io/gh/develtech/django-slugify-processor\n\n.. |license| image:: https://img.shields.io/github/license/develtech/django-slugify-processor.svg\n    :alt: License \n',
    'author': 'Tony Narlock',
    'author_email': 'tony@git-pull.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://django-slugify-processor.git-pull.com',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
