"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const validations = require("../lib/validation");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
function boundsValidationResult(numberOfArtifacts, min, max) {
    const artifacts = [];
    for (let i = 0; i < numberOfArtifacts; i++) {
        artifacts.push(new codepipeline.Artifact(`TestArtifact${i}`));
    }
    return validations.validateArtifactBounds('output', artifacts, min, max, 'testCategory', 'testProvider');
}
module.exports = {
    'artifact bounds validation': {
        'artifacts count exceed maximum'(test) {
            const result = boundsValidationResult(1, 0, 0);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/cannot have more than 0/), 'the validation should have failed');
            test.done();
        },
        'artifacts count below minimum'(test) {
            const result = boundsValidationResult(1, 2, 2);
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/must have at least 2/), 'the validation should have failed');
            test.done();
        },
        'artifacts count within bounds'(test) {
            const result = boundsValidationResult(1, 0, 2);
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action type validation': {
        'must be source and is source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
        'must be source and is not source'(test) {
            const result = validations.validateSourceAction(true, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only contain Source actions/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.SOURCE, 'test action', 'test stage');
            test.deepEqual(result.length, 1);
            test.ok(result[0].match(/may only occur in first stage/), 'the validation should have failed');
            test.done();
        },
        'cannot be source and is not source'(test) {
            const result = validations.validateSourceAction(false, codepipeline.ActionCategory.DEPLOY, 'test action', 'test stage');
            test.deepEqual(result.length, 0);
            test.done();
        },
    },
    'action name validation': {
        'throws an exception when adding an Action with an empty name to the Pipeline'(test) {
            const stack = new cdk.Stack();
            const action = new fake_source_action_1.FakeSourceAction({
                actionName: '',
                output: new codepipeline.Artifact(),
            });
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Source' });
            test.throws(() => {
                stage.addAction(action);
            }, /Action name must match regular expression:/);
            test.done();
        },
    },
    'action Artifacts validation': {
        'validates that input Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                extraInputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                ],
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Build\/Fake cannot have more than 3 input artifacts/);
            test.done();
        },
        'validates that output Artifacts are within bounds'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const extraOutput1 = new codepipeline.Artifact('A1');
            const extraOutput2 = new codepipeline.Artifact('A2');
            const extraOutput3 = new codepipeline.Artifact('A3');
            const extraOutput4 = new codepipeline.Artifact('A4');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                                extraOutputs: [
                                    extraOutput1,
                                    extraOutput2,
                                    extraOutput3,
                                    extraOutput4,
                                ],
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            test.throws(() => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {}));
            }, /Source\/Fake cannot have more than 4 output artifacts/);
            test.done();
        },
    },
    'automatically assigns artifact names to the Actions'(test) {
        const stack = new cdk.Stack();
        const pipeline = new codepipeline.Pipeline(stack, 'pipeline');
        const sourceOutput = new codepipeline.Artifact();
        const sourceAction = new fake_source_action_1.FakeSourceAction({
            actionName: 'CodeCommit',
            output: sourceOutput,
        });
        pipeline.addStage({
            stageName: 'Source',
            actions: [sourceAction],
        });
        pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: sourceOutput,
                    output: new codepipeline.Artifact(),
                }),
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            "Stages": [
                {
                    "Name": "Source",
                    "Actions": [
                        {
                            "Name": "CodeCommit",
                            "OutputArtifacts": [
                                {
                                    "Name": "Artifact_Source_CodeCommit",
                                },
                            ],
                        }
                    ],
                },
                {
                    "Name": "Build",
                    "Actions": [
                        {
                            "Name": "CodeBuild",
                            "InputArtifacts": [
                                {
                                    "Name": "Artifact_Source_CodeCommit",
                                }
                            ],
                            "OutputArtifacts": [
                                {
                                    "Name": "Artifact_Build_CodeBuild",
                                },
                            ],
                        }
                    ],
                },
            ],
        }));
        test.done();
    },
    'the same Action can be safely added to 2 different Stages'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'Source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const action = new fake_build_action_1.FakeBuildAction({ actionName: 'FakeAction', input: sourceOutput });
        const stage2 = {
            stageName: 'Stage2',
            actions: [action],
        };
        const stage3 = {
            stageName: 'Stage3',
            actions: [action],
        };
        pipeline.addStage(stage2);
        test.doesNotThrow(() => {
            pipeline.addStage(stage3);
        }, /FakeAction/);
        test.done();
    },
    'input Artifacts': {
        'can be added multiple times to an Action safely'(test) {
            const artifact = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact,
                    extraInputs: [artifact],
                });
            });
            test.done();
        },
        'can have duplicate names'(test) {
            const artifact1 = new codepipeline.Artifact('SomeArtifact');
            const artifact2 = new codepipeline.Artifact('SomeArtifact');
            test.doesNotThrow(() => {
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'CodeBuild',
                    input: artifact1,
                    extraInputs: [artifact2],
                });
            });
            test.done();
        },
    },
    'output Artifacts': {
        'accept multiple Artifacts with the same name safely'(test) {
            test.doesNotThrow(() => {
                new fake_source_action_1.FakeSourceAction({
                    actionName: 'CodeBuild',
                    output: new codepipeline.Artifact('Artifact1'),
                    extraOutputs: [
                        new codepipeline.Artifact('Artifact1'),
                        new codepipeline.Artifact('Artifact1'),
                    ],
                });
            });
            test.done();
        },
    },
    'an Action with a non-AWS owner cannot have a Role passed for it'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const buildStage = pipeline.addStage({ stageName: 'Build' });
        // constructing it is fine
        const buildAction = new fake_build_action_1.FakeBuildAction({
            actionName: 'build',
            input: sourceOutput,
            owner: 'ThirdParty',
            role: new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            }),
        });
        // an attempt to add it to the Pipeline is where things blow up
        test.throws(() => {
            buildStage.addAction(buildAction);
        }, /Role is not supported for actions with an owner different than 'AWS' - got 'ThirdParty' \(Action: 'build' in Stage: 'Build'\)/);
        test.done();
    },
    'actions can be retrieved from stages they have been added to'(test) {
        const stack = new cdk.Stack();
        const sourceOutput = new codepipeline.Artifact();
        const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
            stages: [
                {
                    stageName: 'Source',
                    actions: [
                        new fake_source_action_1.FakeSourceAction({
                            actionName: 'source',
                            output: sourceOutput,
                        }),
                    ],
                },
            ],
        });
        const sourceStage = pipeline.stages[0];
        const buildStage = pipeline.addStage({
            stageName: 'Build',
            actions: [
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build1',
                    input: sourceOutput,
                    runOrder: 11,
                }),
                new fake_build_action_1.FakeBuildAction({
                    actionName: 'build2',
                    input: sourceOutput,
                    runOrder: 2,
                }),
            ],
        });
        test.equal(sourceStage.actions.length, 1);
        test.equal(sourceStage.actions[0].actionProperties.actionName, 'source');
        test.equal(buildStage.actions.length, 2);
        test.equal(buildStage.actions[0].actionProperties.actionName, 'build1');
        test.equal(buildStage.actions[1].actionProperties.actionName, 'build2');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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