# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flask_timeloop']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'flask-timeloop',
    'version': '0.2.4',
    'description': 'Timeloop module that integrates with Flask',
    'long_description': '# Flask-Timeloop\nTimeloop is a service that can be used to run periodic tasks after a certain interval. It is **meant** to be used to with an underlying flask application.\n\nEach job runs on a separate thread and when the service is shut down, it waits till all tasks currently being executed are completed.\n\nForked and enhanced from [`here`](https://github.com/sankalpjonn/timeloop.git)\nForked and enchanced from [`here`](https://github.com/Ruggiero-Santo/timeloop.git)\n\n#### PyPi\n\nhttps://pypi.org/project/flask-timeloop/\n\n## Installation\n\n### pip\n\n```sh\npip install flask-timeloop\n```\n\n#### Clone and install\n```sh\ngit clone https://github.com/TafkaMax/timeloop.git\nsudo python setup.py install\n```\n\n#### Direct installation \n```sh\npip install git+https://github.com/TafkaMax/timeloop.git\n```\n\n#### Poetry\n```sh\npoetry add flask-timeloop\n```\n\n# Usage\n\nThe recommended way is to use this library with flask factory pattern.\n\n**NB! This is not the correct way to implement the flask extension, as I add the application context to the extension data. BUT Flask only works during requests, but this functionality is internal and does not care about requests. It\'s like a cron, but inside the application.**\n\n## Writing jobs\n\n### Factory pattern\n\n```python\n#python_project_folder/your_app_name/extensions.py\nfrom flask_timeloop import Timeloop\n\ntimeloop = Timeloop()\n```\n\n```python\n#python_project_folder/your_app_name/__init__.py\n#(this can also be main.py or whatever you want.)\n\nfrom flask import Flask\n\ndef create_app():\n    app = Flask(__name__)\n\n    # Import timeloop and join timeloop to flask application\n    from your_app_name.extensions import flask-timeloop\n    timeloop.init_app(app)\n    # Start the timeloop\n    timeloop.start()\n    return app\n```\n\n```python\n#python_project_folder/main.py\nfrom your_app_name import create_app\napp = create_app()\n```\n\n```python\nfrom your_app_name.extensions import timeloop\n\n@timeloop.job(interval=timedelta(minutes=10))\ndef do_something():\n    with timeloop.app.app_context():\n        do_something_that_needs_application_context()\n```\n\n### Basic one file application.\n\n```python\n#main.py\nimport time\n\nfrom timeloop import Timeloop\nfrom datetime import timedelta\n\nfrom flask import Flask\n\n\napp = Flask(__name__)\n\ntl = Timeloop(app)\n\n@tl.job(interval = timedelta(seconds = 2))\ndef sample_job_every_2s():\n    print( "2s job current time : {}".format(time.ctime()) )\n\n@tl.job(interval = 5)\ndef sample_job_every_5s():\n    print( "5s job current time : {}".format(time.ctime()) )\n\n\n@tl.job(interval = timedelta(seconds = 10))\ndef sample_job_every_10s():\n    print( "10s job current time : {}".format(time.ctime()) )\n```\n\n## Writing jobs with arguments\nAllow to create a job with specified parameters in input\n```python\nclass FileToMove:\n    tl = Timeloop()\n\n    def start(self):\n        self.tl.start(True)\n\n    # ATTENTION: If a job wants the self param must be declared as swarm.\n    # This is because the instance isn\'t already created when the job is registered\n    @tl.job(interval = 1, swarm = True, param_2 = "param")\n    def timedMethod(self, param_1, param_2):\n        print(self, "param_1:", param_1, "; param_2:", param_2)\n\n    @tl.job(interval = 2, param_1 = "uno", param_2 = "param")\n    def timedMethod_1(param_1, param_2):\n        print("param_1:", param_1, "; param_2:", param_2)\n\n    # produce the same effect of timedMethod_1. \n    @tl.job(interval = 3 )\n    def timedMethod_2(param_1 = "uno", param_2 = "param"):\n        print("param_1:", param_1, "; param_2:", param_2)\n\nif __name__ == "__main__":\n    ob1 = FileToMove()\n    ob1.timedMethod("try")\n    ob1.start()\n```\nor multiple jobs of the same function but with different parameters. It can be really useful in a situation like above when you want to call a class function.\n```python\n@tl.job(interval = timedelta(seconds = 5), swarm = True)\ndef sample_job(idx):\n    print( "Task id: {} | time: {}".format(idx, time.ctime()) )\n\n# example: queue jobs with different ids\nfor id in range(1, 3):\n\tsample_job(id)\n```\nIn the job declared with  `swarm = True` the param `interval` can be omitted. This allows you to create a swarm of job with different interval, including `interval = 2` or `interval = timedelta(seconds = 2)` in the creation, like example.\n```python\n@tl.job(swarm = True)\ndef sample_job(idx):\n    print( "Task id: {} | time: {}".format(idx, time.ctime()) )\n\n# example: same jobs with different interval\nfor id in range(1, 3):\n\tsample_job(id, interval = id)\n```\n\n## Writing jobs that stop if an exception occurs\n```python\n@tl.job(interval = timedelta(seconds = 2), exception = True)\ndef sample_job():\n    print( "I will die if any Exception occurs,time : {}".format(time.ctime()) )\n\n@tl.job(interval = 2, exception = AttributeError)\ndef sample_job():\n    print( "I will die soon, but only if AttributeError occurs" )\n    raise AttributeError\n\n@tl.job(interval = timedelta(seconds = 2))\ndef sample_job():\n    print( "I will die only if OSError occurs, becouse of start function" )\n\ntl.start(stop_on_exception = OSError)\n```\n## Mode to start jobs\n\n### Start timeloop in a separate thread\nBy default timeloop starts in a separate thread. When it\'s in this mode do not forget to call `tl.stop()` before exiting the program, or else the jobs wont shut down gracefully (or they will even not shutdown).\n```python\ntl.start() or tl.start(block=False)\n```\n\n### Start time loop in the main thread\nDoing this will automatically shut down the jobs gracefully when the program is killed, so no need to  call `tl.stop()`. The main thread that calls the `tl.start()` will be stuck until you kill him (kill command or Ctrl+C on shell).\n```python\ntl.start(block=True)\n```\n',
    'author': 'Taavi Ansper',
    'author_email': 'taavi.ansperr@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
