# coding=utf-8
# Copyright 2023 The TensorFlow Datasets Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""natural_instructions dataset."""

import json
import tensorflow_datasets.public_api as tfds

_DATASET_DIR = "https://raw.githubusercontent.com/allenai/natural-instructions/master/tasks"


class Builder(tfds.core.GeneratorBasedBuilder):
  """DatasetBuilder for natural_instructions dataset."""

  name = "natural_instructions"
  VERSION = tfds.core.Version("1.0.1")
  RELEASE_NOTES = {
      "1.0.1": (
          "Added task name field, and fixed ID used for shuffling to use "
          "stable IDs."
      ),
      "1.0.0": "Initial release.",
  }

  def _info(self) -> tfds.core.DatasetInfo:
    """Returns the dataset metadata."""
    return self.dataset_info_from_configs(
        features=tfds.features.FeaturesDict({
            "id": tfds.features.Text(),
            "input": tfds.features.Text(),
            "output": tfds.features.Text(),
            "source": tfds.features.Text(),
            "task_name": tfds.features.Text(),
            "definition": tfds.features.Text(),
        }),
        supervised_keys=None,
        homepage="https://github.com/allenai/natural-instructions",
    )

  def _split_generators(self, dl_manager: tfds.download.DownloadManager):
    """Returns SplitGenerators."""
    name_to_paths = {
        t_name: f"{_DATASET_DIR}/{t_name}.json" for t_name in _NIV2_TASKS_LIST
    }

    # Downloads the data and defines the splits
    data_dir = dl_manager.download_and_extract(name_to_paths)
    return {
        tfds.Split.TRAIN: self._generate_examples(path=data_dir),
    }

  def _generate_examples(self, path):
    """Yields examples."""
    filepaths = path if isinstance(path, dict) else {"test": path}
    for task_name, fp in filepaths.items():
      data = json.loads(fp.read_text())
      source = data["Source"]
      definition = data["Definition"]
      if isinstance(source, list):
        source = source[0]
      if isinstance(definition, list):
        definition = definition[0]
      for row in data["Instances"]:
        yield row["id"], {
            "id": row["id"],
            "input": row["input"],
            "output": row["output"][0],
            "source": source,
            "task_name": task_name,
            "definition": definition,
        }


_NIV2_TASKS_LIST = [
    "task001_quoref_question_generation",
    "task002_quoref_answer_generation",
    "task003_mctaco_question_generation_event_duration",
    "task004_mctaco_answer_generation_event_duration",
    "task005_mctaco_wrong_answer_generation_event_duration",
    "task006_mctaco_question_generation_transient_stationary",
    "task007_mctaco_answer_generation_transient_stationary",
    "task008_mctaco_wrong_answer_generation_transient_stationary",
    "task009_mctaco_question_generation_event_ordering",
    "task010_mctaco_answer_generation_event_ordering",
    "task011_mctaco_wrong_answer_generation_event_ordering",
    "task012_mctaco_question_generation_absolute_timepoint",
    "task013_mctaco_answer_generation_absolute_timepoint",
    "task014_mctaco_wrong_answer_generation_absolute_timepoint",
    "task015_mctaco_question_generation_frequency",
    "task016_mctaco_answer_generation_frequency",
    "task017_mctaco_wrong_answer_generation_frequency",
    "task018_mctaco_temporal_reasoning_presence",
    "task019_mctaco_temporal_reasoning_category",
    "task020_mctaco_span_based_question",
    "task021_mctaco_grammatical_logical",
    "task022_cosmosqa_passage_inappropriate_binary",
    "task023_cosmosqa_question_generation",
    "task024_cosmosqa_answer_generation",
    "task025_cosmosqa_incorrect_answer_generation",
    "task026_drop_question_generation",
    "task027_drop_answer_type_generation",
    "task028_drop_answer_generation",
    "task029_winogrande_full_object",
    "task030_winogrande_full_person",
    "task031_winogrande_question_generation_object",
    "task032_winogrande_question_generation_person",
    "task033_winogrande_answer_generation",
    "task034_winogrande_question_modification_object",
    "task035_winogrande_question_modification_person",
    "task036_qasc_topic_word_to_generate_related_fact",
    "task037_qasc_generate_related_fact",
    "task038_qasc_combined_fact",
    "task039_qasc_find_overlapping_words",
    "task040_qasc_question_generation",
    "task041_qasc_answer_generation",
    "task042_qasc_incorrect_option_generation",
    "task043_essential_terms_answering_incomplete_questions",
    "task044_essential_terms_identifying_essential_words",
    "task045_miscellaneous_sentence_paraphrasing",
    "task046_miscellaneous_question_typing",
    "task047_miscellaneous_answering_science_questions",
    "task048_multirc_question_generation",
    "task049_multirc_questions_needed_to_answer",
    "task050_multirc_answerability",
    "task051_multirc_correct_answer_single_sentence",
    "task052_multirc_identify_bad_question",
    "task053_multirc_correct_bad_question",
    "task054_multirc_write_correct_answer",
    "task055_multirc_write_incorrect_answer",
    "task056_multirc_classify_correct_answer",
    "task057_multirc_classify_incorrect_answer",
    "task058_multirc_question_answering",
    "task059_ropes_story_generation",
    "task060_ropes_question_generation",
    "task061_ropes_answer_generation",
    "task062_bigbench_repeat_copy_logic",
    "task063_first_i_elements",
    "task064_all_elements_except_first_i",
    "task065_timetravel_consistent_sentence_classification",
    "task066_timetravel_binary_consistency_classification",
    "task067_abductivenli_answer_generation",
    "task068_abductivenli_incorrect_answer_generation",
    "task069_abductivenli_classification",
    "task070_abductivenli_incorrect_classification",
    "task071_abductivenli_answer_generation",
    "task072_abductivenli_answer_generation",
    "task073_commonsenseqa_answer_generation",
    "task074_squad1.1_question_generation",
    "task075_squad1.1_answer_generation",
    "task076_splash_correcting_sql_mistake",
    "task077_splash_explanation_to_sql",
    "task078_all_elements_except_last_i",
    "task079_conala_concat_strings",
    "task080_piqa_answer_generation",
    "task081_piqa_wrong_answer_generation",
    "task082_babi_t1_single_supporting_fact_question_generation",
    "task083_babi_t1_single_supporting_fact_answer_generation",
    "task084_babi_t1_single_supporting_fact_identify_relevant_fact",
    "task085_unnatural_addsub_arithmetic",
    "task086_translated_symbol_arithmetic",
    "task087_new_operator_addsub_arithmetic",
    "task088_identify_typo_verification",
    "task089_swap_words_verification",
    "task090_equation_learner_algebra",
    "task091_all_elements_from_index_i_to_j",
    "task092_check_prime_classification",
    "task093_conala_normalize_lists",
    "task094_conala_calculate_mean",
    "task095_conala_max_absolute_value",
    "task096_conala_list_index_subtraction",
    "task097_conala_remove_duplicates",
    "task098_conala_list_intersection",
    "task099_reverse_elements_between_index_i_and_j",
    "task1000_pib_translation_tamil_malayalam",
    "task1001_pib_translation_gujarati_urdu",
    "task1002_pib_translation_urdu_gujarati",
    "task1003_pib_translation_bengali_malayalam",
    "task1004_pib_translation_malayalam_bengali",
    "task1005_pib_translation_malayalam_punjabi",
    "task1006_pib_translation_punjabi_malayalam",
    "task1007_pib_translation_english_punjabi",
    "task1008_pib_translation_punjabi_english",
    "task1009_pib_translation_bengali_hindi",
    "task100_concatenate_all_elements_from_index_i_to_j",
    "task1010_pib_translation_hindi_bengali",
    "task1011_pib_translation_hindi_punjabi",
    "task1012_pib_translation_punjabi_hindi",
    "task1013_pib_translation_gujarati_telugu",
    "task1014_pib_translation_telugu_gujarati",
    "task1015_pib_translation_punjabi_tamil",
    "task1016_pib_translation_tamil_punjabi",
    "task1017_pib_translation_hindi_malayalam",
    "task1018_pib_translation_malayalam_hindi",
    "task1019_pib_translation_oriya_telugu",
    "task101_reverse_and_concatenate_all_elements_from_index_i_to_j",
    "task1020_pib_translation_telugu_oriya",
    "task1021_pib_translation_english_malayalam",
    "task1022_pib_translation_malayalam_english",
    "task1023_pib_translation_english_hindi",
    "task1024_pib_translation_hindi_english",
    "task1025_pib_translation_bengali_punjabi",
    "task1026_pib_translation_punjabi_bengali",
    "task1027_pib_translation_marathi_telugu",
    "task1028_pib_translation_telugu_marathi",
    "task1029_pib_translation_marathi_punjabi",
    "task102_commongen_sentence_generation",
    "task1030_pib_translation_punjabi_marathi",
    "task1031_pib_translation_bengali_telugu",
    "task1032_pib_translation_telugu_bengali",
    "task1033_pib_translation_gujarati_hindi",
    "task1034_pib_translation_hindi_gujarati",
    "task1035_pib_translation_tamil_urdu",
    "task1036_pib_translation_urdu_tamil",
    "task1037_pib_translation_telugu_urdu",
    "task1038_pib_translation_urdu_telugu",
    "task1039_pib_translation_oriya_punjabi",
    "task103_facts2story_long_text_generation",
    "task1040_pib_translation_punjabi_oriya",
    "task1041_pib_translation_gujarati_malayalam",
    "task1042_pib_translation_malayalam_gujarati",
    "task1043_pib_translation_gujarati_punjabi",
    "task1044_pib_translation_punjabi_gujarati",
    "task1045_pib_translation_hindi_telugu",
    "task1046_pib_translation_telugu_hindi",
    "task1047_pib_translation_english_telugu",
    "task1048_pib_translation_telugu_english",
    "task1049_pib_translation_malayalam_telugu",
    "task104_semeval_2019_task10_closed_vocabulary_mathematical_answer_generation",
    "task1050_pib_translation_telugu_malayalam",
    "task1051_pib_translation_punjabi_urdu",
    "task1052_pib_translation_urdu_punjabi",
    "task1053_pib_translation_hindi_urdu",
    "task1054_pib_translation_urdu_hindi",
    "task1055_pib_translation_marathi_oriya",
    "task1056_pib_translation_oriya_marathi",
    "task1057_pib_translation_english_urdu",
    "task1058_pib_translation_urdu_english",
    "task1059_pib_translation_malayalam_urdu",
    "task105_story_cloze-rocstories_sentence_generation",
    "task1060_pib_translation_urdu_malayalam",
    "task1061_pib_translation_bengali_marathi",
    "task1062_pib_translation_marathi_bengali",
    "task1063_pib_translation_gujarati_tamil",
    "task1064_pib_translation_tamil_gujarati",
    "task1065_pib_translation_punjabi_telugu",
    "task1066_pib_translation_telugu_punjabi",
    "task1067_pib_translation_bengali_gujarati",
    "task1068_pib_translation_gujarati_bengali",
    "task1069_pib_translation_bengali_urdu",
    "task106_scruples_ethical_judgment",
    "task1070_pib_translation_urdu_bengali",
    "task1071_pib_translation_malayalam_marathi",
    "task1072_pib_translation_marathi_malayalam",
    "task1073_pib_translation_oriya_tamil",
    "task1074_pib_translation_tamil_oriya",
    "task1075_pib_translation_tamil_telugu",
    "task1076_pib_translation_telugu_tamil",
    "task1077_pib_translation_gujarati_oriya",
    "task1078_pib_translation_oriya_gujarati",
    "task1079_pib_translation_english_gujarati",
    "task107_splash_question_to_sql",
    "task1080_pib_translation_gujarati_english",
    "task1081_pib_translation_hindi_marathi",
    "task1082_pib_translation_marathi_hindi",
    "task1083_pib_translation_marathi_tamil",
    "task1084_pib_translation_tamil_marathi",
    "task1085_pib_translation_english_marathi",
    "task1086_pib_translation_marathi_english",
    "task1087_two_number_sum",
    "task1088_array_of_products",
    "task1089_check_monotonic_array",
    "task108_contextualabusedetection_classification",
    "task1090_ted_translation_en_gl",
    "task1091_ted_translation_en_it",
    "task1092_ted_translation_en_pl",
    "task1093_ted_translation_en_fa",
    "task1094_ted_translation_en_pt",
    "task1095_ted_translation_ja_gl",
    "task1096_ted_translation_ja_it",
    "task1097_ted_translation_ja_pl",
    "task1098_ted_translation_ja_fa",
    "task1099_ted_translation_ja_pt",
    "task109_smsspamcollection_spamsmsdetection",
    "task1100_ted_translation_es_gl",
    "task1101_ted_translation_es_it",
    "task1102_ted_translation_es_pl",
    "task1103_ted_translation_es_fa",
    "task1104_ted_translation_es_pt",
    "task1105_ted_translation_ar_gl",
    "task1106_ted_translation_ar_it",
    "task1107_ted_translation_ar_pl",
    "task1108_ted_translation_ar_fa",
    "task1109_ted_translation_ar_pt",
    "task110_logic2text_sentence_generation",
    "task1110_ted_translation_he_gl",
    "task1111_ted_translation_he_it",
    "task1112_ted_translation_he_pl",
    "task1113_ted_translation_he_fa",
    "task1114_ted_translation_he_pt",
    "task1115_alt_ja_id_translation",
    "task1116_alt_id_ja_translation",
    "task1117_alt_ja_id_answer_generation",
    "task1118_alt_ja_fil_translation",
    "task1119_alt_fil_ja_translation",
    "task111_asset_sentence_simplification",
    "task1120_alt_ja_fil_answer_generation",
    "task1121_alt_ja_khm_translation",
    "task1122_alt_khm_ja_translation",
    "task1123_alt_ja_khm_answer_generation",
    "task1124_alt_ja_lo_translation",
    "task1125_alt_lo_ja_translation",
    "task1126_alt_ja_lo_answer_generation",
    "task1127_alt_ja_th_translation",
    "task1128_alt_th_ja_translation",
    "task1129_alt_ja_th_answer_generation",
    "task112_asset_simple_sentence_identification",
    "task1130_xcsr_vi_commonsense_mc_classification",
    "task1131_xcsr_es_commonsense_mc_classification",
    "task1132_xcsr_ur_commonsense_mc_classification",
    "task1133_xcsr_nl_commonsense_mc_classification",
    "task1134_xcsr_hi_commonsense_mc_classification",
    "task1135_xcsr_en_commonsense_mc_classification",
    "task1136_xcsr_fr_commonsense_mc_classification",
    "task1137_xcsr_pt_commonsense_mc_classification",
    "task1138_xcsr_de_commonsense_mc_classification",
    "task1139_xcsr_ru_commonsense_mc_classification",
    "task113_count_frequency_of_letter",
    "task1140_xcsr_pl_commonsense_mc_classification",
    "task1141_xcsr_zh_commonsense_mc_classification",
    "task1142_xcsr_ar_commonsense_mc_classification",
    "task1143_xcsr_it_commonsense_mc_classification",
    "task1144_xcsr_sw_commonsense_mc_classification",
    "task1145_xcsr_jap_commonsense_mc_classification",
    "task1146_country_capital",
    "task1147_country_currency",
    "task1148_maximum_ascii_value",
    "task1149_item_check_edible",
    "task114_is_the_given_word_longest",
    "task1150_delete_max_min",
    "task1151_swap_max_min",
    "task1152_bard_analogical_reasoning_causation",
    "task1153_bard_analogical_reasoning_affordance",
    "task1154_bard_analogical_reasoning_travel",
    "task1155_bard_analogical_reasoning_trash_or_treasure",
    "task1156_bard_analogical_reasoning_tools",
    "task1157_bard_analogical_reasoning_rooms_for_containers",
    "task1158_bard_analogical_reasoning_manipulating_items",
    "task1159_bard_analogical_reasoning_containers",
    "task115_help_advice_classification",
    "task1161_coda19_title_generation",
    "task1162_coda19_title_classification",
    "task1163_coda19_section_classification",
    "task1164_coda19_section_correction_classification",
    "task1167_penn_treebank_coarse_pos_tagging",
    "task1168_brown_coarse_pos_tagging",
    "task1168_xcopa_commonsense_reasoning_ht",
    "task1169_xcopa_commonsense_cause_effect_ht",
    "task116_com2sense_commonsense_reasoning",
    "task1170_xcopa_commonsense_reasoning_id",
    "task1171_xcopa_commonsense_cause_effect_id",
    "task1172_xcopa_commonsense_reasoning_it",
    "task1173_xcopa_commonsense_cause_effect_it",
    "task1174_xcopa_commonsense_reasoning_sw",
    "task1175_xcopa_commonsense_cause_effect_sw",
    "task1176_xcopa_commonsense_reasoning_ta",
    "task1177_xcopa_commonsense_cause_effect_ta",
    "task1178_xcopa_commonsense_reasoning_th",
    "task1179_xcopa_commonsense_cause_effect_th",
    "task117_spl_translation_en_de",
    "task1180_xcopa_commonsense_reasoning_tr",
    "task1181_xcopa_commonsense_cause_effect_tr",
    "task1182_xcopa_commonsense_reasoning_vi",
    "task1183_xcopa_commonsense_cause_effect_vi",
    "task1184_xcopa_commonsense_reasoning_zh",
    "task1185_xcopa_commonsense_cause_effect_zh",
    "task1186_nne_hrngo_classification",
    "task1187_politifact_classification",
    "task1188_count_max_freq_char",
    "task1189_check_char_in_string",
    "task118_semeval_2019_task10_open_vocabulary_mathematical_answer_generation",
    "task1190_add_integer_to_list",
    "task1191_food_veg_nonveg",
    "task1192_food_flavor_profile",
    "task1193_food_course_classification",
    "task1194_kth_largest_element",
    "task1195_disflqa_disfluent_to_fluent_conversion",
    "task1196_atomic_classification_oeffect",
    "task1197_atomic_classification_oreact",
    "task1198_atomic_classification_owant",
    "task1199_atomic_classification_xattr",
    "task119_semeval_2019_task10_geometric_mathematical_answer_generation",
    "task1200_atomic_classification_xeffect",
    "task1201_atomic_classification_xintent",
    "task1202_atomic_classification_xneed",
    "task1203_atomic_classification_xreact",
    "task1204_atomic_classification_hinderedby",
    "task1205_atomic_classification_isafter",
    "task1206_atomic_classification_isbefore",
    "task1207_atomic_classification_atlocation",
    "task1208_atomic_classification_xreason",
    "task1209_atomic_classification_objectuse",
    "task120_zest_text_modification",
    "task1210_atomic_classification_madeupof",
    "task1211_atomic_classification_hassubevent",
    "task1212_atomic_classification_hasproperty",
    "task1213_atomic_classification_desires",
    "task1214_atomic_classification_xwant",
    "task1215_atomic_classification_capableof",
    "task1216_atomic_classification_causes",
    "task1217_atomic_answer_generation",
    "task1218_ted_translation_en_ja",
    "task1219_ted_translation_en_es",
    "task121_zest_text_modification",
    "task1220_ted_translation_en_ar",
    "task1221_ted_translation_en_he",
    "task1222_ted_translation_ja_en",
    "task1223_ted_translation_ja_es",
    "task1224_ted_translation_ja_ar",
    "task1225_ted_translation_ja_he",
    "task1226_ted_translation_es_en",
    "task1227_ted_translation_es_ja",
    "task1228_ted_translation_es_ar",
    "task1229_ted_translation_es_he",
    "task122_conala_list_index_addition",
    "task1230_ted_translation_ar_en",
    "task1231_ted_translation_ar_ja",
    "task1232_ted_translation_ar_es",
    "task1233_ted_translation_ar_he",
    "task1234_ted_translation_he_en",
    "task1235_ted_translation_he_ja",
    "task1236_ted_translation_he_es",
    "task1237_ted_translation_he_ar",
    "task1238_ted_translation_gl_en",
    "task1239_ted_translation_gl_ja",
    "task123_conala_sort_dictionary",
    "task1240_ted_translation_gl_es",
    "task1241_ted_translation_gl_ar",
    "task1242_ted_translation_gl_he",
    "task1243_ted_translation_gl_it",
    "task1244_ted_translation_gl_pl",
    "task1245_ted_translation_gl_fa",
    "task1246_ted_translation_gl_pt",
    "task1247_ted_translation_it_en",
    "task1248_ted_translation_it_ja",
    "task1249_ted_translation_it_es",
    "task124_conala_pair_averages",
    "task1250_ted_translation_it_ar",
    "task1251_ted_translation_it_he",
    "task1252_ted_translation_it_gl",
    "task1253_ted_translation_it_pl",
    "task1254_ted_translation_it_fa",
    "task1255_ted_translation_it_pt",
    "task1256_ted_translation_pl_en",
    "task1257_ted_translation_pl_ja",
    "task1258_ted_translation_pl_es",
    "task1259_ted_translation_pl_ar",
    "task125_conala_pair_differences",
    "task1260_ted_translation_pl_he",
    "task1261_ted_translation_pl_gl",
    "task1262_ted_translation_pl_it",
    "task1263_ted_translation_pl_fa",
    "task1264_ted_translation_pl_pt",
    "task1265_ted_translation_fa_en",
    "task1266_ted_translation_fa_ja",
    "task1267_ted_translation_fa_es",
    "task1268_ted_translation_fa_ar",
    "task1269_ted_translation_fa_he",
    "task126_scan_structured_text_generation_command_action_all",
    "task1270_ted_translation_fa_gl",
    "task1271_ted_translation_fa_it",
    "task1272_ted_translation_fa_pl",
    "task1273_ted_translation_fa_pt",
    "task1274_ted_translation_pt_en",
    "task1275_ted_translation_pt_ja",
    "task1276_ted_translation_pt_es",
    "task1277_ted_translation_pt_ar",
    "task1278_ted_translation_pt_he",
    "task1279_ted_translation_pt_gl",
    "task127_scan_long_text_generation_action_command_all",
    "task1280_ted_translation_pt_it",
    "task1281_ted_translation_pt_pl",
    "task1282_ted_translation_pt_fa",
    "task1283_hrngo_quality_classification",
    "task1284_hrngo_informativeness_classification",
    "task1285_kpa_keypoint_matching",
    "task1286_openbookqa_question_answering",
    "task1287_glue_qqp_paraphrasing",
    "task1288_glue_mrpc_paraphrasing",
    "task1289_trec_classification",
    "task128_scan_structured_text_generation_command_action_short",
    "task1290_xsum_summarization",
    "task1291_multi_news_summarization",
    "task1292_yelp_review_full_text_categorization",
    "task1293_kilt_tasks_hotpotqa_question_answering",
    "task1294_wiki_qa_answer_verification",
    "task1295_adversarial_qa_question_answering",
    "task1296_wiki_hop_question_answering",
    "task1297_qasc_question_answering",
    "task129_scan_long_text_generation_action_command_short",
    "task1308_amazonreview_category_classification",
    "task1309_amazonreview_summary_classification",
    "task130_scan_structured_text_generation_command_action_long",
    "task1310_amazonreview_rating_classification",
    "task1311_amazonreview_rating_classification",
    "task1312_amazonreview_polarity_classification",
    "task1313_amazonreview_polarity_classification",
    "task1314_country_abbreviation",
    "task1315_find_range_array",
    "task1316_remove_duplicates_string",
    "task1317_country_calling_code",
    "task1318_country_national_dish",
    "task1319_country_by_barcode_prefix",
    "task131_scan_long_text_generation_action_command_long",
    "task1320_country_domain_tld",
    "task1321_country_continent",
    "task1322_country_government_type",
    "task1323_open_subtitles_hi_en_translation",
    "task1324_open_subtitles_te_en_translation",
    "task1325_qa_zre_question_generation_on_subject_relation",
    "task1326_qa_zre_question_generation_from_answer",
    "task1327_qa_zre_answer_generation_from_question",
    "task1328_qa_zre_relation_generation_from_question",
    "task1329_open_subtitles_en_hi_translation",
    "task132_dais_text_modification",
    "task1330_open_subtitles_en_te_translation",
    "task1331_reverse_array",
    "task1332_check_leap_year",
    "task1333_check_validity_date_ddmmyyyy",
    "task1334_sqac_answer_generation",
    "task1335_sqac_question_generation",
    "task1336_peixian_equity_evaluation_corpus_gender_classifier",
    "task1338_peixian_equity_evaluation_corpus_sentiment_classifier",
    "task1339_peixian_equity_evaluation_corpus_text_completion",
    "task133_winowhy_reason_plausibility_detection",
    "task1340_msr_text_compression_compression",
    "task1341_msr_text_classification",
    "task1342_amazon_us_reviews_title",
    "task1343_amazon_us_reviews_rating",
    "task1344_glue_entailment_classification",
    "task1345_glue_qqp_question_paraprashing",
    "task1346_glue_cola_grammatical_correctness_classification",
    "task1347_glue_sts-b_similarity_classification",
    "task134_winowhy_reason_generation",
    "task1350_opus100_translation_en_gu",
    "task1351_opus100_translation_gu_en",
    "task1352_hind_encorp_translation_hi_en",
    "task1353_hind_encorp_translation_en_hi",
    "task1354_sent_comp_classification",
    "task1355_sent_comp_summarization",
    "task1356_xlsum_title_generation",
    "task1357_xlsum_summary_generation",
    "task1358_xlsum_title_generation",
    "task1359_numer_sense_answer_generation",
    "task135_winowhy_wrong_reason_generation",
    "task1360_numer_sense_multiple_choice_qa_generation",
    "task1361_movierationales_classification",
    "task1364_hans_answer_generation",
    "task1365_opustedtalks_translation",
    "task1366_healthfact_classification",
    "task1367_opustedtalks_translation",
    "task1368_healthfact_sentence_generation",
    "task1369_healthfact_sentence_generation",
    "task136_winowhy_knowledge_categorization",
    "task1370_newscomm_classification",
    "task1371_newscomm_translation",
    "task1373_newscomm_translation",
    "task1374_newscomm_translation",
    "task1375_newscomm_translation",
    "task1376_newscomm_translation",
    "task1377_newscomm_translation",
    "task1378_quarel_correct_answer_generation",
    "task1379_quarel_incorrect_answer_generation",
    "task137_detoxifying-lms_classification_toxicity",
    "task1380_quarel_correct_option_generation",
    "task1381_quarel_incorrect_option_generation",
    "task1382_quarel_write_correct_answer",
    "task1383_quarel_write_incorrect_answer",
    "task1384_deal_or_no_dialog_classification",
    "task1385_anli_r1_entailment",
    "task1386_anli_r2_entailment",
    "task1387_anli_r3_entailment",
    "task1388_cb_entailment",
    "task1389_hellaswag_completion",
    "task138_detoxifying-lms_classification_fluency",
    "task1390_wscfixed_coreference",
    "task1391_winogrande_easy_answer_generation",
    "task1392_superglue_multirc_answer_verification",
    "task1393_superglue_copa_text_completion",
    "task1394_meta_woz_task_classification",
    "task1395_europa_ecdc_tm_en_sv_translation",
    "task1396_europa_ecdc_tm_en_de_translation",
    "task1397_europa_ecdc_tm_fr_en_translation",
    "task1398_obqa_question_generation",
    "task1399_obqa_answer_generation",
    "task139_detoxifying-lms_classification_topicality",
    "task1400_obqa_incorrect_answer_generation",
    "task1401_obqa_sentence_generation",
    "task1402_clue_question_generation",
    "task1403_check_validity_date_mmddyyyy",
    "task1404_date_conversion",
    "task1405_find_median",
    "task1406_kth_smallest_element",
    "task1407_dart_question_generation",
    "task1408_dart_similarity_classification",
    "task1409_dart_text_generation",
    "task140_detoxifying-lms_classification_style",
    "task1410_dart_relationship_extraction",
    "task1411_dart_subject_identification",
    "task1412_web_questions_question_answering",
    "task1413_dart_object_identification",
    "task1414_ajgt_twitter_ar_classification",
    "task1415_youtube_caption_corrections_grammar_correction",
    "task1416_youtube_caption_corrections_incorrect_grammar_classification",
    "task1418_bless_semantic_relation_classification",
    "task1419_mathqa_gain",
    "task141_odd-man-out_classification_category",
    "task1420_mathqa_general",
    "task1421_mathqa_other",
    "task1422_mathqa_physics",
    "task1423_mathqa_geometry",
    "task1424_mathqa_probability",
    "task1425_country_iso_numeric",
    "task1426_country_independence_year",
    "task1427_country_region_in_world",
    "task1428_country_surface_area",
    "task1429_evalution_semantic_relation_classification",
    "task142_odd-man-out_classification_no_category",
    "task1431_head_qa_answer_generation",
    "task1432_head_qa_language_translation_en_to_es",
    "task1433_head_qa_language_translation_es_to_en",
    "task1434_head_qa_classification",
    "task1435_ro_sts_parallel_language_translation_ro_to_en",
    "task1436_ro_sts_parallel_language_translation_en_to_ro",
    "task1437_doqa_cooking_question_generation",
    "task1438_doqa_cooking_answer_generation",
    "task1439_doqa_cooking_isanswerable",
    "task143_odd-man-out_classification_generate_category",
    "task1440_doqa_movies_question_generation",
    "task1441_doqa_movies_answer_generation",
    "task1442_doqa_movies_isanswerable",
    "task1443_string_to_number",
    "task1444_round_power_of_two",
    "task1445_closest_integers",
    "task1446_farthest_integers",
    "task1447_drug_extraction_ade",
    "task1448_disease_entity_extraction_ncbi_dataset",
    "task1449_disease_entity_extraction_bc5cdr_dataset",
    "task144_subjqa_question_answering",
    "task1451_drug_dose_extraction",
    "task1452_location_entity_extraction_btc_corpus",
    "task1453_person_entity_extraction_btc_corpus",
    "task145_afs_argument_similarity_death_penalty",
    "task146_afs_argument_similarity_gun_control",
    "task1479_organization_entity_extraction_btc_corpus",
    "task147_afs_argument_similarity_gay_marriage",
    "task1480_gene_extraction_jnlpba_dataset",
    "task1481_gene_extraction_bc2gm_dataset",
    "task1482_gene_extraction_chemprot_dataset",
    "task1483_chemical_extraction_chemprot_dataset",
    "task1484_gene_extraction_linnaeus_dataset",
    "task1485_organ_extraction_anem_dataset",
    "task1486_cell_extraction_anem_dataset",
    "task1487_organism_substance_extraction_anem_dataset",
    "task1488_sarcasmdetection_headline_classification",
    "task1489_sarcasmdetection_tweet_classification",
    "task148_afs_argument_quality_gay_marriage",
    "task1490_bengali_personal_hate_speech_binary_classification",
    "task1491_bengali_political_hate_speech_binary_classification",
    "task1492_bengali_religious_hate_speech_binary_classification",
    "task1493_bengali_geopolitical_hate_speech_binary_classification",
    "task1494_bengali_hate_speech_classification",
    "task1495_adverse_drug_event_classification",
    "task1496_bengali_reviews_sentiment_classification",
    "task1497_bengali_book_reviews_sentiment_classification",
    "task1498_24hour_to_12hour_clock",
    "task1499_dstc3_summarization",
    "task149_afs_argument_quality_death_penalty",
    "task1500_dstc3_classification",
    "task1501_dstc3_answer_generation",
    "task1502_hatexplain_classification",
    "task1503_hatexplain_classification",
    "task1504_hatexplain_answer_generation",
    "task1505_root09_semantic_relation_classification",
    "task1506_celebrity_minimal_dob_span",
    "task1507_boolean_temporal_reasoning",
    "task1508_wordnet_antonyms",
    "task1509_evalution_antonyms",
    "task150_afs_argument_quality_gun_control",
    "task1510_evalution_relation_extraction",
    "task1514_flores_translation_entone",
    "task1515_imppres_longtextgeneration",
    "task1516_imppres_naturallanguageinference",
    "task1517_limit_classfication",
    "task1518_limit_answer_generation",
    "task1519_qa_srl_question_generation",
    "task151_tomqa_find_location_easy_clean",
    "task1520_qa_srl_answer_generation",
    "task1529_scitail1.1_classification",
    "task1530_scitail1.1_sentence_generation",
    "task152_tomqa_find_location_easy_noise",
    "task1531_daily_dialog_type_classification",
    "task1532_daily_dialog_emotion_classification",
    "task1533_daily_dialog_formal_classification",
    "task1534_daily_dialog_question_classification",
    "task1535_daily_dialog_uniqueness_classification",
    "task1536_daily_dialog_happiness_classification",
    "task1537_tamil_offenseval_dravidian_classification",
    "task1538_malayalam_offenseval_dravidian_classification",
    "task1539_kannada_offenseval_dravidian_classification",
    "task153_tomqa_find_location_hard_clean",
    "task1540_parsed_pdfs_summarization",
    "task1541_agnews_classification",
    "task1542_every_ith_element_from_starting",
    "task1543_conll2002_parts_of_speech_tagging_answer_generation",
    "task1544_conll2002_named_entity_recognition_answer_generation",
    "task1545_conll2002_person_name_extraction_answer_generation",
    "task1546_conll2002_location_name_extraction_answer_generation",
    "task1548_wiqa_binary_classification",
    "task1549_wiqa_answer_generation_missing_step",
    "task154_tomqa_find_location_hard_noise",
    "task1551_every_ith_element_from_kth_element",
    "task1552_scitail_question_generation",
    "task1553_cnn_dailymail_summarization",
    "task1554_scitail_classification",
    "task1555_scitail_answer_generation",
    "task1556_scitail_passage_generation",
    "task1557_jfleg_answer_generation",
    "task1558_jfleg_incorrect_answer_generation",
    "task1559_blimp_binary_classification",
    "task155_count_nouns_verbs",
    "task1560_blimp_binary_classification",
    "task1561_clickbait_new_bg_summarization",
    "task1562_zest_text_modification",
    "task1564_triviaqa_answer_generation",
    "task1565_triviaqa_classification",
    "task1566_propara_structured_text_generation",
    "task1567_propara_question_generation",
    "task1568_propara_classification",
    "task1569_cmrc2018_question_generation",
    "task156_codah_classification_adversarial",
    "task1570_cmrc2018_answer_generation",
    "task1571_cmrc2018_answer_generation_starting_index",
    "task1572_samsum_summary",
    "task1573_samsum_classification",
    "task1574_amazon_reviews_multi_language_identification",
    "task1575_amazon_reviews_multi_sentiment_classification",
    "task1576_amazon_reviews_multi_english_language_classification",
    "task1577_amazon_reviews_multi_japanese_language_classification",
    "task1579_gigaword_incorrect_summarization",
    "task157_count_vowels_and_consonants",
    "task1580_eqasc-perturbed_question_generation",
    "task1581_eqasc-perturbed_answer_generation",
    "task1582_bless_hypernym_generation",
    "task1583_bless_meronym_classification",
    "task1584_evalution_meronym_classification",
    "task1585_root09_hypernym_generation",
    "task1586_scifact_title_generation",
    "task1587_scifact_classification",
    "task1588_tecla_classification",
    "task1589_scifact_classification",
    "task158_count_frequency_of_words",
    "task1590_diplomacy_text_generation",
    "task1591_allocine_classification",
    "task1592_yahoo_answers_topics_classfication",
    "task1593_yahoo_answers_topics_classification",
    "task1594_yahoo_answers_topics_question_generation",
    "task1595_event2mind_text_generation_1",
    "task1596_event2mind_text_generation_2",
    "task1597_nyc_slot_filling",
    "task1598_nyc_long_text_generation",
    "task1599_smcalflow_classification",
    "task159_check_frequency_of_words_in_sentence_pair",
    "task1600_smcalflow_sentence_generation",
    "task1601_webquestions_answer_generation",
    "task1602_webquestion_question_genreation",
    "task1603_smcalflow_sentence_generation",
    "task1604_ethos_text_classification",
    "task1605_ethos_text_classification",
    "task1606_ethos_text_classification",
    "task1607_ethos_text_classification",
    "task1608_xquad_en_answer_generation",
    "task1609_xquad_en_question_generation",
    "task160_replace_letter_in_a_sentence",
    "task1610_xquad_es_answer_generation",
    "task1611_xquad_es_question_generation",
    "task1612_sick_label_classification",
    "task1613_sick_given_category_generate_sentence",
    "task1614_sick_text_modify",
    "task1615_sick_tclassify_b_relation_a",
    "task1616_cc_alligned_translate_eng_tel",
    "task1617_cc_alligned_translate_tel_eng",
    "task1618_cc_alligned_classify_tel_eng",
    "task1619_menyo20k-mt_en_yo_translation",
    "task161_count_words_containing_letter",
    "task1620_menyo20k-mt_yo_en_translation",
    "task1621_menyo20k-mt_en_yo_language_identification",
    "task1622_disfl_qa_text_modication",
    "task1623_disfl_qa_disfluent_question_classification",
    "task1624_disfl_qa_question_yesno_classification",
    "task1625_disfl_qa_asnwer_generation",
    "task1626_copa_hr_question_answering",
    "task1627_copa_hr_classification",
    "task1628_copa_hr_question_answering",
    "task1629_copa_hr_classification",
    "task162_count_words_starting_with_letter",
    "task1630_openpi_classification",
    "task1631_openpi_answer_generation",
    "task1637_doqa2.1_cooking_text_summarization",
    "task1638_doqa2.1_movies_text_summarization",
    "task1639_doqa2.1_travel_text_summarization",
    "task163_count_words_ending_with_letter",
    "task1640_aqa1.0_answerable_unanswerable_question_classification",
    "task1645_medical_question_pair_dataset_text_classification",
    "task1646_dataset_card_for_catalonia_independence_corpus_text_classification",
    "task1647_opus_books_en-pt_translation",
    "task1648_opus_books_en-sv_translation",
    "task1649_opus_books_en-no_translation",
    "task164_mcscript_question_answering_text",
    "task1650_opus_books_en-fi_translation",
    "task1651_opus_books_en-es__translation",
    "task1652_opus_books_ca-en_translation",
    "task1654_mkb_translation",
    "task1655_mkb_translation",
    "task1656_gooaq_answer_generation",
    "task1657_gooaq_question_generation",
    "task1658_billsum_summarization",
    "task1659_title_generation",
    "task165_mcscript_question_answering_commonsense",
    "task1660_super_glue_question_generation",
    "task1661_super_glue_classification",
    "task1662_cedr_ru_classification",
    "task1663_cedr_ru_incorrect_classification",
    "task1664_winobias_text_generation",
    "task1665_trainglecopa_question_generation",
    "task1666_cail2018_answer_generation",
    "task1667_cail2018_answer_generation",
    "task1669_md_gender_bias_text_modification",
    "task166_clariq_sentence_generation",
    "task1670_md_gender_bias_text_modification",
    "task1676_xquad-ca_translation",
    "task1677_xquad-ca_translation",
    "task1678_mathqa_answer_selection",
    "task167_strategyqa_question_generation",
    "task1685_menyo20k_translation",
    "task1686_menyo20k_translation",
    "task1689_qed_amara_translation",
    "task168_strategyqa_question_decomposition",
    "task1690_qed_amara_translation",
    "task1691_qed_amara_translation",
    "task1692_qed_amara_translation",
    "task169_strategyqa_sentence_generation",
    "task1703_ljspeech_textmodification",
    "task1704_ljspeech_textmodification",
    "task1705_ljspeech_classification",
    "task1706_ljspeech_classification",
    "task170_hotpotqa_answer_generation",
    "task1711_poki_text_generation",
    "task1712_poki_classification",
    "task1713_convai3_sentence_generation",
    "task1714_convai3_sentence_generation",
    "task171_spl_translation_en_es",
    "task1720_civil_comments_toxicity_classification",
    "task1721_civil_comments_obscenity_classification",
    "task1722_civil_comments_threat_classification",
    "task1723_civil_comments_sexuallyexplicit_classification",
    "task1724_civil_comments_insult_classification",
    "task1725_civil_comments_severtoxicity_classification",
    "task1726_mathqa_correct_answer_generation",
    "task1727_wiqa_what_is_the_effect",
    "task1728_web_nlg_data_to_text",
    "task1729_personachat_generate_next",
    "task172_spl_translation_en_fa",
    "task1730_personachat_choose_next",
    "task1731_quartz_question_answering",
    "task173_spl_translation_en_it",
    "task174_spl_translation_en_ja",
    "task175_spl_translation_en_pl",
    "task176_break_decompose_questions",
    "task177_para-nmt_paraphrasing",
    "task178_quartz_question_answering",
    "task179_participant_extraction",
    "task180_intervention_extraction",
    "task181_outcome_extraction",
    "task182_duorc_question_generation",
    "task183_rhyme_generation",
    "task184_break_generate_question",
    "task184_snli_entailment_to_neutral_text_modification",
    "task185_snli_contradiction_to_neutral_text_modification",
    "task186_snli_contradiction_to_entailment_text_modification",
    "task187_snli_entailment_to_contradiction_text_modification",
    "task188_snli_neutral_to_entailment_text_modification",
    "task189_snli_neutral_to_contradiction_text_modification",
    "task190_snli_classification",
    "task191_hotpotqa_question_generation",
    "task192_hotpotqa_sentence_generation",
    "task193_duorc_question_generation",
    "task194_duorc_answer_generation",
    "task195_sentiment140_classification",
    "task196_sentiment140_answer_generation",
    "task197_mnli_domain_answer_generation",
    "task198_mnli_domain_classification",
    "task199_mnli_classification",
    "task200_mnli_entailment_classification",
    "task201_mnli_neutral_classification",
    "task202_mnli_contradiction_classification",
    "task203_mnli_sentence_generation",
    "task204_mnli_same_genre_classification",
    "task205_remove_even_elements",
    "task206_collatz_conjecture",
    "task207_max_element_lists",
    "task208_combinations_of_list",
    "task209_stancedetection_classification",
    "task210_logic2text_structured_text_generation",
    "task211_logic2text_classification",
    "task212_logic2text_classification",
    "task213_rocstories_correct_ending_classification",
    "task214_rocstories_incorrect_ending_classification",
    "task215_rocstories_incorrect_answer_generation",
    "task216_rocstories_correct_answer_generation",
    "task217_rocstories_ordering_answer_generation",
    "task218_rocstories_swap_order_answer_generation",
    "task219_rocstories_title_answer_generation",
    "task220_rocstories_title_classification",
    "task221_rocstories_two_choice_classification",
    "task222_rocstories_two_chioce_slotting_classification",
    "task223_quartz_explanation_generation",
    "task224_scruples_anecdotes_ethical_judgment",
    "task225_english_language_answer_generation",
    "task226_english_language_answer_relevance_classification",
    "task227_clariq_classification",
    "task228_arc_answer_generation_easy",
    "task229_arc_answer_generation_hard",
    "task230_iirc_passage_classification",
    "task231_iirc_link_classification",
    "task232_iirc_link_number_classification",
    "task233_iirc_link_exists_classification",
    "task234_iirc_passage_line_answer_generation",
    "task235_iirc_question_from_subtext_answer_generation",
    "task236_iirc_question_from_passage_answer_generation",
    "task237_iirc_answer_from_subtext_answer_generation",
    "task238_iirc_answer_from_passage_answer_generation",
    "task239_tweetqa_answer_generation",
    "task240_tweetqa_question_generation",
    "task241_tweetqa_classification",
    "task242_tweetqa_classification",
    "task243_count_elements_in_set_intersection",
    "task244_count_elements_in_set_union",
    "task245_check_presence_in_set_intersection",
    "task246_dream_question_generation",
    "task247_dream_answer_generation",
    "task248_dream_classification",
    "task249_enhanced_wsc_pronoun_disambiguation",
    "task250_spl_translation_en_ar",
    "task251_spl_translation_en_fi",
    "task252_spl_translation_en_tr",
    "task253_spl_translation_en_zh",
    "task254_spl_translation_fi_en",
    "task255_spl_translation_it_en",
    "task256_spl_translation_de_en",
    "task257_spl_translation_ar_en",
    "task258_spl_translation_fa_en",
    "task259_spl_translation_tr_en",
    "task260_spl_translation_zh_en",
    "task261_spl_translation_es_en",
    "task262_spl_translation_ja_en",
    "task263_spl_translation_pl_en",
    "task264_paper_reviews_accept_or_reject_classification",
    "task265_paper_reviews_language_identification",
    "task266_paper_reviews_reviewer_perspective_classification",
    "task267_concatenate_and_reverse_all_elements_from_index_i_to_j",
    "task268_casehold_legal_answer_generation",
    "task269_csrg_counterfactual_story_generation",
    "task270_csrg_counterfactual_context_generation",
    "task271_europarl_translation",
    "task272_europarl_translation",
    "task273_europarl_classification",
    "task274_overruling_legal_classification",
    "task275_enhanced_wsc_paraphrase_generation",
    "task276_enhanced_wsc_classification",
    "task277_stereoset_sentence_generation_stereotype",
    "task278_stereoset_sentence_generation_antistereotype",
    "task279_stereoset_classification_stereotype",
    "task280_stereoset_classification_stereotype_type",
    "task281_points_of_correspondence",
    "task282_scruples_event_time",
    "task283_dream_incorrect_answer_generation",
    "task284_imdb_classification",
    "task285_imdb_answer_generation",
    "task286_olid_offense_judgment",
    "task287_casehold_legal_incorrect_answer_generation",
    "task288_gigaword_summarization",
    "task289_gigaword_summarization",
    "task290_tellmewhy_question_answerability",
    "task291_semeval_2020_task4_commonsense_validation",
    "task292_storycommonsense_character_text_generation",
    "task293_storycommonsense_emotion_text_generation",
    "task294_storycommonsense_motiv_text_generation",
    "task295_semeval_2020_task4_commonsense_reasoning",
    "task296_storycloze_correct_end_classification",
    "task297_storycloze_incorrect_end_classification",
    "task298_storycloze_correct_end_classification",
    "task299_storycloze_sentence_generation",
    "task300_storycloze_order_generation",
    "task301_record_question_generation",
    "task302_record_classification",
    "task303_record_incorrect_answer_generation",
    "task304_numeric_fused_head_resolution",
    "task305_jeopardy_answer_generation_normal",
    "task306_jeopardy_answer_generation_double",
    "task307_jeopardy_answer_generation_final",
    "task308_jeopardy_answer_generation_all",
    "task309_race_answer_generation",
    "task310_race_classification",
    "task311_race_question_generation",
    "task312_europarl_sv_en_translation",
    "task313_europarl_en_sv_translation",
    "task314_europarl_sv-en_classification",
    "task315_europarl_sv-en_language_identification",
    "task316_crows-pairs_classification_stereotype",
    "task317_crows-pairs_classification_stereotype_type",
    "task318_stereoset_classification_gender",
    "task319_stereoset_classification_profession",
    "task320_stereoset_classification_race",
    "task321_stereoset_classification_religion",
    "task322_jigsaw_classification_threat",
    "task323_jigsaw_classification_sexually_explicit",
    "task324_jigsaw_classification_disagree",
    "task325_jigsaw_classification_identity_attack",
    "task326_jigsaw_classification_obscene",
    "task327_jigsaw_classification_toxic",
    "task328_jigsaw_classification_insult",
    "task329_gap_classification",
    "task330_gap_answer_generation",
    "task331_gap_incorrect_answer_generation",
    "task332_tellmewhy_answer_generation",
    "task333_hateeval_classification_hate_en",
    "task334_hateeval_classification_hate_es",
    "task335_hateeval_classification_aggresive_en",
    "task336_hateeval_classification_aggresive_es",
    "task337_hateeval_classification_individual_en",
    "task338_hateeval_classification_individual_es",
    "task339_record_answer_generation",
    "task340_winomt_classification_gender_pro",
    "task341_winomt_classification_gender_anti",
    "task342_winomt_classification_profession_pro",
    "task343_winomt_classification_profession_anti",
    "task344_hybridqa_answer_generation",
    "task345_hybridqa_answer_generation",
    "task346_hybridqa_classification",
    "task347_hybridqa_incorrect_answer_generation",
    "task348_squad2.0_unanswerable_question_generation",
    "task349_squad2.0_answerable_unanswerable_question_classification",
    "task350_winomt_classification_gender_identifiability_pro",
    "task351_winomt_classification_gender_identifiability_anti",
    "task352_coda-19_classification",
    "task353_casino_classification_negotiation_elicit_pref",
    "task354_casino_classification_negotiation_no_need",
    "task355_casino_classification_negotiation_other_need",
    "task356_casino_classification_negotiation_self_need",
    "task357_casino_classification_negotiation_small_talk",
    "task358_casino_classification_negotiation_uv_part",
    "task359_casino_classification_negotiation_vouch_fair",
    "task360_spolin_yesand_response_generation",
    "task361_spolin_yesand_prompt_response_classification",
    "task362_spolin_yesand_prompt_response_sub_classification",
    "task363_sst2_polarity_classification",
    "task364_regard_social_impact_classification",
    "task365_synthetic_remove_vowels",
    "task366_synthetic_return_primes",
    "task367_synthetic_remove_floats",
    "task368_synthetic_even_or_odd_calculation",
    "task369_synthetic_remove_odds",
    "task370_synthetic_remove_divisible_by_3",
    "task371_synthetic_product_of_list",
    "task372_synthetic_palindrome_numbers",
    "task373_synthetic_round_tens_place",
    "task374_synthetic_pos_or_neg_calculation",
    "task375_classify_type_of_sentence_in_debate",
    "task376_reverse_order_of_words",
    "task377_remove_words_of_given_length",
    "task378_reverse_words_of_given_length",
    "task379_agnews_topic_classification",
    "task380_boolq_yes_no_question",
    "task381_boolq_question_generation",
    "task382_hybridqa_answer_generation",
    "task383_matres_classification",
    "task384_socialiqa_question_classification",
    "task385_socialiqa_incorrect_answer_generation",
    "task386_semeval_2018_task3_irony_detection",
    "task387_semeval_2018_task3_irony_classification",
    "task388_torque_token_classification",
    "task389_torque_generate_temporal_question",
    "task390_torque_text_span_selection",
    "task391_causal_relationship",
    "task392_inverse_causal_relationship",
    "task393_plausible_result_generation",
    "task394_persianqa_question_generation",
    "task395_persianqa_answer_generation",
    "task396_persianqa_classification",
    "task397_semeval_2018_task1_tweet_anger_detection",
    "task398_semeval_2018_task1_tweet_joy_detection",
    "task399_semeval_2018_task1_tweet_sadness_detection",
    "task400_paws_paraphrase_classification",
    "task401_numeric_fused_head_reference",
    "task402_grailqa_paraphrase_generation",
    "task403_creak_commonsense_inference",
    "task404_grailqa_paraphrase_validation",
    "task405_narrativeqa_question_generation",
    "task406_mickey_fr_sentence_perturbation_generation",
    "task407_mickey_hi_sentence_perturbation_generation",
    "task408_mickey_it_sentence_perturbation_generation",
    "task409_mickey_nl_sentence_perturbation_generation",
    "task410_mickey_ru_sentence_perturbation_generation",
    "task411_mickey_vi_sentence_perturbation_generation",
    "task412_mickey_zh_sentence_perturbation_generation",
    "task413_mickey_en_sentence_perturbation_generation",
    "task414_mickey_ar_sentence_perturbation_generation",
    "task415_mickey_bg_sentence_perturbation_generation",
    "task416_mickey_de_sentence_perturbation_generation",
    "task417_mickey_es_sentence_perturbation_generation",
    "task418_persent_title_generation",
    "task419_persent_answer_generation",
    "task420_persent_document_sentiment_classification",
    "task421_persent_sentence_sentiment_classification",
    "task422_persent_sentence_sentiment_verification",
    "task423_persent_document_sentiment_verification",
    "task424_hindienglish_corpora_hi_en_translation",
    "task425_hindienglish_corpora_en_hi_translation",
    "task426_hindienglish_corpora_hi-en_classification",
    "task427_hindienglish_corpora_hi-en_language_identification",
    "task428_senteval_inversion",
    "task429_senteval_tense",
    "task430_senteval_subject_count",
    "task431_senteval_object_count",
    "task432_alt_en_hi_translation",
    "task433_alt_hi_en_translation",
    "task434_alt_en_hi_answer_generation",
    "task435_alt_en_ja_translation",
    "task436_alt_ja_en_translation",
    "task437_alt_en_ja_answer_generation",
    "task438_eng_guj_parallel_corpus_en_gu_translation",
    "task439_eng_guj_parallel_corpus_gu_en_translation",
    "task440_eng_guj_parallel_corpus_gu-en_classification",
    "task441_eng_guj_parallel_corpus_gu-en_language_identification",
    "task442_com_qa_paraphrase_question_generation",
    "task443_com_qa_ans_question_generation",
    "task444_com_qa_question_paraphrases_answer_generation",
    "task446_opus_paracrawl_en_so_translation",
    "task447_opus_paracrawl_classification",
    "task448_opus_paracrawl_en_tl_translation",
    "task449_opus_paracrawl_ig_en_translation",
    "task450_opus_paracrawl_so_en_translation",
    "task451_opus_paracrawl_tl_en_translation",
    "task452_opus_paracrawl_en_ig_translation",
    "task453_swag_answer_generation",
    "task454_swag_incorrect_answer_generation",
    "task455_swag_context_generation",
    "task456_matres_intention_classification",
    "task457_matres_conditional_classification",
    "task458_matres_negation_classification",
    "task459_matres_static_classification",
    "task460_qasper_answer_generation",
    "task461_qasper_question_generation",
    "task462_qasper_classification",
    "task463_parsinlu_entailment_classification",
    "task464_parsinlu_entailment_sentence_generation",
    "task465_parsinlu_qqp_classification",
    "task466_parsinlu_qqp_text_modification",
    "task467_parsinlu_rc_answer_generation",
    "task468_parsinlu_rc_question_generation",
    "task469_mrqa_answer_generation",
    "task470_mrqa_question_generation",
    "task471_haspart_answer_generation",
    "task472_haspart_classification",
    "task473_parsinlu_mc_classification",
    "task474_parsinlu_mc_classification",
    "task475_yelp_polarity_classification",
    "task476_cls_english_books_classification",
    "task477_cls_english_dvd_classification",
    "task478_cls_english_music_classification",
    "task479_cls_german_books_classification",
    "task480_cls_german_dvd_classification",
    "task481_cls_german_music_classification",
    "task482_cls_french_books_classification",
    "task483_cls_french_dvd_classification",
    "task484_cls_french_music_classification",
    "task485_cls_japanese_books_classification",
    "task486_cls_japanese_dvd_classification",
    "task487_cls_japanese_music_classification",
    "task488_extract_all_alphabetical_elements_from_list_in_order",
    "task489_mwsc_question_generation",
    "task490_mwsc_options_generation",
    "task491_mwsc_answer_generation",
    "task492_mwsc_incorrect_answer_generation",
    "task493_review_polarity_classification",
    "task494_review_polarity_answer_generation",
    "task495_semeval_headline_classification",
    "task496_semeval_answer_generation",
    "task497_extract_all_numbers_from_list_in_order",
    "task498_scruples_anecdotes_whoiswrong_classification",
    "task499_extract_and_add_all_numbers_from_list",
    "task500_scruples_anecdotes_title_generation",
    "task501_scruples_anecdotes_post_type_verification",
    "task502_scruples_anecdotes_whoiswrong_verification",
    "task503_scruples_anecdotes_isanswerable",
    "task504_count_all_alphabetical_elements_in_list",
    "task505_count_all_numerical_elements_in_list",
    "task506_position_of_all_alphabetical_elements_in_list",
    "task507_position_of_all_numerical_elements_in_list",
    "task508_scruples_dilemmas_more_ethical_isidentifiable",
    "task509_collate_of_all_alphabetical_and_numerical_elements_in_list_separately",
    "task510_reddit_tifu_title_summarization",
    "task511_reddit_tifu_long_text_summarization",
    "task512_twitter_emotion_classification",
    "task513_argument_stance_classification",
    "task514_argument_consequence_classification",
    "task515_senteval_odd_word_out",
    "task516_senteval_conjoints_inversion",
    "task517_emo_classify_emotion_of_dialogue",
    "task518_emo_different_dialogue_emotions",
    "task519_aquamuse_question_generation",
    "task520_aquamuse_answer_given_in_passage",
    "task521_trivia_question_classification",
    "task522_news_editorial_summary",
    "task523_find_if_numbers_or_alphabets_are_more_in_list",
    "task524_parsinlu_food_aspect_classification",
    "task525_parsinlu_movie_aspect_classification",
    "task526_parsinlu_movie_overal_classification",
    "task527_parsinlu_food_overal_classification",
    "task528_parsinlu_movie_aspect_detection",
    "task529_parsinlu_food_aspect_detection",
    "task530_europarl_en_es_translation",
    "task531_europarl_es_en_translation",
    "task532_europarl_en-es_classification",
    "task533_europarl_es-en_language_identification",
    "task534_farstail_entailment",
    "task535_alt_translation_ch_en",
    "task536_alt_translation_vi_en",
    "task537_alt_translation_th_en",
    "task538_alt_translation_bu_en",
    "task539_alt_translation_ma_en",
    "task540_alt_translation_la_en",
    "task541_alt_translation_kh_en",
    "task542_alt_translation_ja_en",
    "task543_alt_translation_bh_en",
    "task544_alt_translation_hi_en",
    "task545_alt_translation_fi_en",
    "task546_alt_translation_bg_en",
    "task547_alt_translation_entk_en",
    "task548_alt_translation_en_ch",
    "task549_alt_translation_en_vi",
    "task550_discofuse_sentence_generation",
    "task551_alt_translation_en_th",
    "task552_alt_translation_en_bu",
    "task553_alt_translation_en_ma",
    "task554_alt_translation_en_la",
    "task555_alt_translation_en_kh",
    "task556_alt_translation_en_ja",
    "task557_alt_translation_en_ba",
    "task558_alt_translation_en_hi",
    "task559_alt_translation_en_fi",
    "task560_alt_translation_en_entk",
    "task561_alt_translation_en_bg",
    "task562_alt_language_identification",
    "task563_discofuse_answer_generation",
    "task564_discofuse_classification",
    "task565_circa_answer_generation",
    "task566_circa_classification",
    "task567_circa_text_generation",
    "task568_circa_question_generation",
    "task569_recipe_nlg_text_generation",
    "task570_recipe_nlg_ner_generation",
    "task571_recipe_nlg_ner_generation",
    "task572_recipe_nlg_text_generation",
    "task573_air_dialogue_classification",
    "task574_air_dialogue_sentence_generation",
    "task575_air_dialogue_classification",
    "task576_curiosity_dialogs_answer_generation",
    "task577_curiosity_dialogs_classification",
    "task578_curiosity_dialogs_answer_generation",
    "task579_socialiqa_classification",
    "task580_socialiqa_answer_generation",
    "task581_socialiqa_question_generation",
    "task582_naturalquestion_answer_generation",
    "task583_udeps_eng_coarse_pos_tagging",
    "task584_udeps_eng_fine_pos_tagging",
    "task585_preposition_classification",
    "task586_amazonfood_polarity_classification",
    "task587_amazonfood_polarity_correction_classification",
    "task588_amazonfood_rating_classification",
    "task589_amazonfood_summary_text_generation",
    "task590_amazonfood_summary_correction_classification",
    "task591_sciq_answer_generation",
    "task592_sciq_incorrect_answer_generation",
    "task593_sciq_explanation_generation",
    "task594_sciq_question_generation",
    "task595_mocha_answer_generation",
    "task596_mocha_question_generation",
    "task597_cuad_answer_generation",
    "task598_cuad_answer_generation",
    "task599_cuad_question_generation",
    "task600_find_the_longest_common_substring_in_two_strings",
    "task601_flores_translation_sntoen",
    "task602_wikitext-103_answer_generation",
    "task603_wikitext-103_fill_in_the_blank",
    "task604_flores_translation_entosn",
    "task605_find_the_longest_common_subsequence_in_two_lists",
    "task606_sum_of_all_numbers_in_list_between_positions_i_and_j",
    "task607_sbic_intentional_offense_binary_classification",
    "task608_sbic_sexual_offense_binary_classification",
    "task609_sbic_potentially_offense_binary_classification",
    "task610_conllpp_ner",
    "task611_mutual_multi_turn_dialogue",
    "task612_yorubabbc_classification",
    "task613_politifact_text_generation",
    "task614_glucose_cause_event_detection",
    "task615_moviesqa_answer_generation",
    "task616_cola_classification",
    "task617_amazonreview_category_text_generation",
    "task618_amazonreview_summary_text_generation",
    "task619_ohsumed_abstract_title_generation",
    "task620_ohsumed_medical_subject_headings_answer_generation",
    "task621_ohsumed_yes_no_numerical_answer_generation",
    "task622_replace_alphabets_in_a_list_by_their_position_in_english_alphabet",
    "task623_ohsumed_yes_no_answer_generation",
    "task624_ohsumed_question_answering",
    "task625_xlwic_true_or_false_answer_generation",
    "task626_xlwic_sentence_based_on_given_word_sentence_generation",
    "task627_xlwic_word_with_same_meaning_sentence_generation",
    "task628_xlwic_word_with_different_meaning_sentence_generation",
    "task629_dbpedia_14_classification",
    "task630_dbpedia_14_classification",
    "task631_dbpedia_14_incorrect_answer_generation",
    "task632_dbpedia_14_classification",
    "task633_dbpedia_14_answer_generation",
    "task634_allegro_reviews_classification",
    "task635_allegro_reviews_answer_generation",
    "task636_extract_and_sort_unique_alphabets_in_a_list",
    "task637_extract_and_sort_unique_digits_in_a_list",
    "task638_multi_woz_classification",
    "task639_multi_woz_user_utterance_generation",
    "task640_esnli_classification",
    "task641_esnli_classification",
    "task642_esnli_classification",
    "task643_refresd_classification",
    "task644_refresd_translation",
    "task645_summarization",
    "task646_answer_generation",
    "task647_answer_generation",
    "task648_answer_generation",
    "task649_race_blank_question_generation",
    "task650_opus100_ar_en_translation",
    "task651_opus100_en_ar_translation",
    "task652_parsinlu_en_fa_translation",
    "task653_parsinlu_fa_en_translation",
    "task654_bible_fa_en_translation",
    "task655_bible_en_fa_translation",
    "task656_quran_en_fa_translation",
    "task657_quran_fa_en_translation",
    "task658_tep_en_fa_translation",
    "task659_tep_fa_en_translation",
    "task660_mizan_fa_en_translation",
    "task661_mizan_en_fa_translation",
    "task662_global_voices_fa_en_translation",
    "task663_global_voices_en_fa_translation",
    "task664_mmmlu_answer_generation_abstract_algebra",
    "task665_mmmlu_answer_generation_anatomy",
    "task666_mmmlu_answer_generation_astronomy",
    "task667_mmmlu_answer_generation_business_ethics",
    "task668_extreme_abstract_summarization",
    "task669_ambigqa_answer_generation",
    "task670_ambigqa_question_generation",
    "task671_ambigqa_text_generation",
    "task672_amazon_and_yelp_summarization_dataset_summarization",
    "task672_nummersense",
    "task673_google_wellformed_query_classification",
    "task674_google_wellformed_query_sentence_generation",
    "task675_google_wellformed_query_sentence_generation",
    "task676_ollie_relationship_answer_generation",
    "task677_ollie_sentence_answer_generation",
    "task678_ollie_actual_relationship_answer_generation",
    "task679_hope_edi_english_text_classification",
    "task680_hope_edi_tamil_text_classification",
    "task681_hope_edi_malayalam_text_classification",
    "task682_online_privacy_policy_text_classification",
    "task683_online_privacy_policy_text_purpose_answer_generation",
    "task684_online_privacy_policy_text_information_type_generation",
    "task685_mmmlu_answer_generation_clinical_knowledge",
    "task686_mmmlu_answer_generation_college_biology",
    "task687_mmmlu_answer_generation_college_chemistry",
    "task688_mmmlu_answer_generation_college_computer_science",
    "task689_mmmlu_answer_generation_college_mathematics",
    "task690_mmmlu_answer_generation_college_medicine",
    "task691_mmmlu_answer_generation_college_physics",
    "task692_mmmlu_answer_generation_computer_security",
    "task693_mmmlu_answer_generation_conceptual_physics",
    "task694_mmmlu_answer_generation_econometrics",
    "task695_mmmlu_answer_generation_electrical_engineering",
    "task696_mmmlu_answer_generation_elementary_mathematics",
    "task697_mmmlu_answer_generation_formal_logic",
    "task698_mmmlu_answer_generation_global_facts",
    "task699_mmmlu_answer_generation_high_school_biology",
    "task700_mmmlu_answer_generation_high_school_chemistry",
    "task701_mmmlu_answer_generation_high_school_computer_science",
    "task702_mmmlu_answer_generation_high_school_european_history",
    "task703_mmmlu_answer_generation_high_school_geography",
    "task704_mmmlu_answer_generation_high_school_government_and_politics",
    "task705_mmmlu_answer_generation_high_school_macroeconomics",
    "task706_mmmlu_answer_generation_high_school_mathematics",
    "task707_mmmlu_answer_generation_high_school_microeconomics",
    "task708_mmmlu_answer_generation_high_school_physics",
    "task709_mmmlu_answer_generation_high_school_psychology",
    "task710_mmmlu_answer_generation_high_school_statistics",
    "task711_mmmlu_answer_generation_high_school_us_history",
    "task712_mmmlu_answer_generation_high_school_world_history",
    "task713_mmmlu_answer_generation_human_aging",
    "task714_mmmlu_answer_generation_human_sexuality",
    "task715_mmmlu_answer_generation_international_law",
    "task716_mmmlu_answer_generation_jurisprudence",
    "task717_mmmlu_answer_generation_logical_fallacies",
    "task718_mmmlu_answer_generation_machine_learning",
    "task719_mmmlu_answer_generation_management",
    "task720_mmmlu_answer_generation_marketing",
    "task721_mmmlu_answer_generation_medical_genetics",
    "task722_mmmlu_answer_generation_random_topic",
    "task723_mmmlu_answer_generation_moral_disputes",
    "task724_mmmlu_answer_generation_moral_scenarios",
    "task725_mmmlu_answer_generation_nutrition",
    "task726_mmmlu_answer_generation_philosophy",
    "task727_mmmlu_answer_generation_prehistory",
    "task728_mmmlu_answer_generation_professional_accounting",
    "task729_mmmlu_answer_generation_professional_law",
    "task730_mmmlu_answer_generation_professional_medicine",
    "task731_mmmlu_answer_generation_professional_psychology",
    "task732_mmmlu_answer_generation_public_relations",
    "task733_mmmlu_answer_generation_security_studies",
    "task734_mmmlu_answer_generation_sociology",
    "task735_mmmlu_answer_generation_us_foreign_policy",
    "task736_mmmlu_answer_generation_virology",
    "task737_mmmlu_answer_generation_world_religions",
    "task738_perspectrum_classification",
    "task739_lhoestq_question_generation",
    "task740_lhoestq_answer_generation_quantity",
    "task741_lhoestq_answer_generation_place",
    "task742_lhoestq_answer_generation_frequency",
    "task743_eurlex_summarization",
    "task744_eurlex_classification",
    "task745_ai2_arithmetic_questions_arithmetic",
    "task746_yelp_restaurant_review_classification",
    "task747_glucose_cause_emotion_detection",
    "task748_glucose_reverse_cause_event_detection",
    "task749_glucose_reverse_cause_emotion_detection",
    "task750_aqua_multiple_choice_answering",
    "task751_svamp_subtraction_question_answering",
    "task752_svamp_multiplication_question_answering",
    "task753_svamp_addition_question_answering",
    "task754_svamp_common-division_question_answering",
    "task755_find_longest_substring_and_replace_its_sorted_lowercase_version_in_both_lists",
    "task756_find_longert_substring_and_return_all_unique_alphabets_in_it",
    "task757_msr_sqa_question_generation",
    "task758_msr_sqa_question_answer_generation",
    "task759_msr_sqa_incorrect_answer_generation",
    "task760_msr_sqa_long_text_generation",
    "task761_app_review_classification",
    "task762_emea_fr_sk_translation",
    "task763_emea_es_lt_translation",
    "task764_emea_bg_el_classification",
    "task765_emea_bg_el_translation",
    "task766_craigslist_bargains_classification",
    "task767_craigslist_bargains_classification",
    "task768_qed_text_span_selection",
    "task769_qed_summarization",
    "task770_pawsx_english_text_modification",
    "task771_pawsx_korean_text_modification",
    "task772_pawsx_french_text_modification",
    "task773_pawsx_spanish_text_modification",
    "task774_pawsx_german_text_modification",
    "task775_pawsx_chinese_text_modification",
    "task776_pawsx_japanese_text_modification",
    "task777_pawsx_english_korean_translation",
    "task778_pawsx_english_french_translation",
    "task779_pawsx_english_spanish_translation",
    "task780_pawsx_english_german_translation",
    "task781_pawsx_english_chinese_translation",
    "task782_pawsx_english_japanese_translation",
    "task783_pawsx_korean_english_translation",
    "task784_pawsx_korean_french_translation",
    "task785_pawsx_korean_spanish_translation",
    "task786_pawsx_korean_german_translation",
    "task787_pawsx_korean_chinese_translation",
    "task788_pawsx_korean_japanese_translation",
    "task789_pawsx_french_english_translation",
    "task790_pawsx_french_korean_translation",
    "task791_pawsx_french_spanish_translation",
    "task792_pawsx_french_german_translation",
    "task793_pawsx_french_chinese_translation",
    "task794_pawsx_french_japanese_translation",
    "task795_pawsx_spanish_english_translation",
    "task796_pawsx_spanish_korean_translation",
    "task797_pawsx_spanish_french_translation",
    "task798_pawsx_spanish_german_translation",
    "task799_pawsx_spanish_chinese_translation",
    "task800_pawsx_spanish_japanese_translation",
    "task801_pawsx_german_english_translation",
    "task802_pawsx_german_korean_translation",
    "task803_pawsx_german_french_translation",
    "task804_pawsx_german_spanish_translation",
    "task805_pawsx_german_chinese_translation",
    "task806_pawsx_german_japanese_translation",
    "task807_pawsx_chinese_english_translation",
    "task808_pawsx_chinese_korean_translation",
    "task809_pawsx_chinese_french_translation",
    "task810_pawsx_chinese_spanish_translation",
    "task811_pawsx_chinese_german_translation",
    "task812_pawsx_chinese_japanese_translation",
    "task813_pawsx_japanese_english_translation",
    "task814_pawsx_japanese_korean_translation",
    "task815_pawsx_japanese_french_translation",
    "task816_pawsx_japanese_spanish_translation",
    "task817_pawsx_japanese_german_translation",
    "task818_pawsx_japanese_chinese_translation",
    "task819_pec_sentiment_classification",
    "task820_protoqa_answer_generation",
    "task821_protoqa_question_generation",
    "task823_peixian-rtgender_sentiment_analysis",
    "task827_copa_commonsense_reasoning",
    "task828_copa_commonsense_cause_effect",
    "task829_giga_fren_translation",
    "task830_poleval2019_mt_translation",
    "task831_giga_fren_classification",
    "task832_poleval2019_mt_classification",
    "task833_poem_sentiment_classification",
    "task834_mathdataset_classification",
    "task835_mathdataset_answer_generation",
    "task836_viquiquad_question_generation",
    "task837_viquiquad_answer_generation",
    "task838_cdt_classification",
    "task839_cdt_classification",
    "task840_para_pdt_en_es_translation",
    "task841_para_pdt_de_en_translation",
    "task842_para_pdt_cs_en_translation",
    "task843_financial_phrasebank_classification",
    "task844_financial_phrasebank_classification",
    "task845_pubmedqa_question_generation",
    "task846_pubmedqa_classification",
    "task847_pubmedqa_question_generation",
    "task848_pubmedqa_classification",
    "task849_pubmedqa_answer_generation",
    "task850_synthetic_longest_palindrome",
    "task851_synthetic_multiply_evens",
    "task852_synthetic_multiply_odds",
    "task853_hippocorpus_long_text_generation",
    "task854_hippocorpus_classification",
    "task855_conv_ai_2_classification",
    "task856_conv_ai_2_classification",
    "task857_inquisitive_question_generation",
    "task858_inquisitive_span_detection",
    "task859_prost_question_generation",
    "task860_prost_mcq_generation",
    "task861_asdiv_addsub_question_answering",
    "task861_prost_mcq_answers_generation",
    "task862_asdiv_multidiv_question_answering",
    "task863_asdiv_multiop_question_answering",
    "task864_asdiv_singleop_question_answering",
    "task865_mawps_addsub_question_answering",
    "task866_mawps_multidiv_question_answering",
    "task867_mawps_multiop_question_answering",
    "task868_cfq_mcd1_explanation_to_sql",
    "task868_mawps_singleop_question_answering",
    "task869_cfq_mcd1_sql_to_explanation",
    "task870_msmarco_answer_generation",
    "task871_msmarco_question_generation",
    "task872_opus_xhosanavy_translation_eng_xhosa",
    "task873_opus_xhosanavy_translation_xhosa_eng",
    "task874_opus_xhosanavy_sr",
    "task875_emotion_classification",
    "task877_kde4_translation",
    "task878_kde4_translation",
    "task879_schema_guided_dstc8_classification",
    "task880_schema_guided_dstc8_classification",
    "task881_schema_guided_dstc8_classification",
    "task886_quail_question_generation",
    "task887_quail_answer_generation",
    "task888_reviews_classification",
    "task889_goemotions_classification",
    "task890_gcwd_classification",
    "task891_gap_coreference_resolution",
    "task892_gap_reverse_coreference_resolution",
    "task893_gap_fill_the_blank_coreference_resolution",
    "task896_miam_language_classification",
    "task897_freebase_qa_topic_question_generation",
    "task898_freebase_qa_answer_generation",
    "task899_freebase_qa_topic_generation",
    "task900_freebase_qa_category_classification",
    "task901_freebase_qa_category_question_generation",
    "task902_deceptive_opinion_spam_classification",
    "task903_deceptive_opinion_spam_classification",
    "task904_hate_speech_offensive_classification",
    "task905_hate_speech_offensive_classification",
    "task906_dialogre_identify_names",
    "task907_dialogre_identify_relationships",
    "task908_dialogre_identify_familial_relationships",
    "task909_dialogre_prevalent_speakers",
    "task910_bianet_classification",
    "task911_bianet_translation",
    "task912_bianet_classification",
    "task913_bianet_translation",
    "task914_bianet_translation",
    "task917_coqa_question_generation",
    "task918_coqa_answer_generation",
    "task919_coqa_incorrect_answer_generation",
    "task921_code_x_glue_information_retreival",
    "task922_event2mind_word_generation",
    "task923_event2mind_classifier",
    "task924_event2mind_word_generation",
    "task925_coached_conv_pref_classifier",
    "task926_coached_conv_pref_word_generation",
    "task927_yelp_negative_to_positive_style_transfer",
    "task928_yelp_positive_to_negative_style_transfer",
    "task929_products_reviews_classification",
    "task930_dailydialog_classification",
    "task931_dailydialog_classification",
    "task932_dailydialog_classification",
    "task933_wiki_auto_style_transfer",
    "task934_turk_simplification",
    "task935_defeasible_nli_atomic_classification",
    "task936_defeasible_nli_snli_classification",
    "task937_defeasible_nli_social_classification",
    "task938_copa_hi_commonsense_reasoning",
    "task939_copa_hi_commonsense_cause_effect",
    "task940_copa_gu_commonsense_reasoning",
    "task941_copa_gu_commonsense_cause_effect",
    "task942_copa_mr_commonsense_reasoning",
    "task943_copa_mr_commonsense_cause_effect",
    "task944_wiki_cloze_as_multiple_choice_question_answering",
    "task945_wiki_cloze_bn_multiple_choice_question_answering",
    "task946_wiki_cloze_gu_multiple_choice_question_answering",
    "task947_wiki_cloze_hi_multiple_choice_question_answering",
    "task948_wiki_cloze_kn_multiple_choice_question_answering",
    "task949_wiki_cloze_ml_multiple_choice_question_answering",
    "task950_wiki_cloze_mr_multiple_choice_question_answering",
    "task951_wiki_cloze_or_multiple_choice_question_answering",
    "task952_wiki_cloze_pa_multiple_choice_question_answering",
    "task953_wiki_cloze_ta_multiple_choice_question_answering",
    "task954_wiki_cloze_te_multiple_choice_question_answering",
    "task955_wiki_auto_style_transfer",
    "task956_leetcode_420_strong_password_check",
    "task957_e2e_nlg_text_generation_generate",
    "task958_e2e_nlg_text_generation_parse",
    "task959_e2e_nlg_text_generation_identify",
    "task960_ancora-ca-ner_named_entity_recognition",
    "task961_ancora-ca-ner_text_auto_completion",
    "task962_ancora-ca-ner_missing_word_prediction",
    "task963_librispeech_asr_next_word_prediction",
    "task964_librispeech_asr_text_auto_completion",
    "task965_librispeech_asr_missing_word_prediction",
    "task966_ruletaker_fact_checking_based_on_given_context",
    "task967_ruletaker_incorrect_fact_generation_based_on_given_paragraph",
    "task968_xcopa_commonsense_reasoning_et",
    "task969_xcopa_commonsense_cause_effect_et",
    "task970_sherliic_causal_relationship",
    "task974_prachathai67k_sentiment_classification",
    "task975_prachathai67k_same_genre_classification",
    "task976_pib_indian_language_identification",
    "task977_pib_translation_oriya_urdu",
    "task978_pib_translation_urdu_oriya",
    "task979_pib_translation_malayalam_oriya",
    "task980_pib_translation_oriya_malayalam",
    "task981_pib_translation_bengali_tamil",
    "task982_pib_translation_tamil_bengali",
    "task983_pib_translation_gujarati_marathi",
    "task984_pib_translation_marathi_gujarati",
    "task985_pib_translation_hindi_oriya",
    "task986_pib_translation_oriya_hindi",
    "task987_pib_translation_english_oriya",
    "task988_pib_translation_oriya_english",
    "task989_pib_translation_marathi_urdu",
    "task990_pib_translation_urdu_marathi",
    "task991_pib_translation_english_tamil",
    "task992_pib_translation_tamil_english",
    "task993_pib_translation_hindi_tamil",
    "task994_pib_translation_tamil_hindi",
    "task995_pib_translation_bengali_english",
    "task996_pib_translation_english_bengali",
    "task997_pib_translation_bengali_oriya",
    "task998_pib_translation_oriya_bengali",
    "task999_pib_translation_malayalam_tamil",
]
