"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const openapi_types_1 = require("openapi-types");
const concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (scope, path, method, { integrations, authType, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { function: { functionArn }, } = integrations[operationName];
    // Generate the lambda invocation arn as the uri for the API Gateway integration
    const stack = aws_cdk_lib_1.Stack.of(scope);
    const uri = `arn:${stack.partition}:apigateway:${stack.region}:lambda:path/2015-03-31/functions/${functionArn}/invocations`;
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri,
            passthroughBehavior: aws_apigateway_1.PassthroughBehavior.WHEN_NO_MATCH,
        },
        "x-amazon-apigateway-auth": {
            type: authType || aws_apigateway_1.AuthorizationType.NONE,
        },
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${(corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS).join(",")}'`,
    "Access-Control-Allow-Methods": `'${(corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS).join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (openapi_types_1.OpenAPIV3.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode || 204;
    return {
        [openapi_types_1.OpenAPIV3.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (scope, path, pathItem, options, getOperationName) => {
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(openapi_types_1.OpenAPIV3.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(scope, path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (scope, spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[concatMethodAndPath(methodAndPath)];
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            [aws_apigateway_1.ResponseType.BAD_REQUEST_BODY.responseType]: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(scope, path, pathDetails, options, getOperationName),
            ])),
        },
    };
};
//# sourceMappingURL=data:application/json;base64,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