"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": ["export * from './api';", "export * from './sample-api';"].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.ts": `import { OpenApiGatewayLambdaApi, OpenApiGatewayLambdaApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import spec from "../spec/parsed-spec.json";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayLambdaApiProps, "spec" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Construct for the API Gateway resources defined by the spec
 */
export class Api extends OpenApiGatewayLambdaApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Construct } from "constructs";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { AuthorizationType } from "aws-cdk-lib/aws-apigateway";
import * as path from "path";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      authType: AuthorizationType.IAM,
      integrations: {
        sayHello: {
          function: new NodejsFunction(scope, "SayHelloHandler", {
            entry: path.join(__dirname, "say-hello.handler.ts"),
          }),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "say-hello.handler.ts": `import { sayHelloHandler, ApiError } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler<ApiError>(async (input) => {
  return {
    statusCode: 200,
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.srcdir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        function: new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        }),
      }]),
      ) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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