"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./cross-region-support-stack");
const full_action_descriptor_1 = require("./full-action-descriptor");
const stage_1 = require("./stage");
const validation_1 = require("./validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ name: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            const encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                // remove the key - there is a grace period of a few days before it's gone for good,
                // that should be enough for any emergency access to the bucket artifacts
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: s3.BucketEncryption.KMS,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
            // add an alias to make finding the key in the console easier
            new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                aliasName: this.generateNameForDefaultBucketKeyAlias(),
                targetKey: encryptionKey,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(action);
        // get the role for the given action
        const actionRole = this.getRoleForAction(stage, action, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(action.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = action.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRegion;
        let otherStack;
        const actionResource = action.actionProperties.resource;
        if (actionResource) {
            const actionResourceStack = core_1.Stack.of(actionResource);
            if (pipelineStack.region !== actionResourceStack.region) {
                actionRegion = actionResourceStack.region;
                // if the resource is from a different stack in another region but the same account,
                // use that stack as home for the cross-region support resources
                if (pipelineStack.account === actionResourceStack.account) {
                    otherStack = actionResourceStack;
                }
            }
        }
        else {
            actionRegion = action.actionProperties.region;
        }
        // if actionRegion is undefined,
        // it means the action is in the same region as the pipeline -
        // so, just return the artifactBucket
        if (!actionRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // get the region the Pipeline itself is in
        const pipelineRegion = this.requireRegion();
        // if the action is in the same region as the pipeline, nothing to do
        if (actionRegion === pipelineRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        // the stack containing the replication bucket must be deployed before the pipeline
        pipelineStack.addDependency(crossRegionSupport.stack);
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: actionRegion,
        };
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id);
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = this.node.uniqueId;
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        if (!this.artifactBucket.encryptionKey) {
            throw new Error('The Pipeline is being used in a cross-account manner, ' +
                'but its artifact bucket does not have a KMS key defined. ' +
                'A KMS key is required for a cross-account Pipeline. ' +
                'Make sure to pass a Bucket with a Key when creating the Pipeline');
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${this.node.uniqueId}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
//# sourceMappingURL=data:application/json;base64,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