"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, commandHooks, } = props;
        const outputPath = path.posix.join(aws_cdk_lib_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: aws_cdk_lib_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            commandHooks,
        });
        this.image = image ?? aws_cdk_lib_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = ['bash', '-c', chain(bundlingCommands)];
        this.environment = props.environment;
    }
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        bundlingCommands.push(`cp -rTL ${options.inputDir}/ ${options.outputDir}`);
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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