# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['visual_graph_datasets',
 'visual_graph_datasets.examples',
 'visual_graph_datasets.experiments',
 'visual_graph_datasets.generation',
 'visual_graph_datasets.processing',
 'visual_graph_datasets.visualization']

package_data = \
{'': ['*'], 'visual_graph_datasets': ['templates/*']}

install_requires = \
['cairosvg>=2.5.0',
 'click>=7.1.2',
 'imageio>=2.22.4',
 'jinja2>=3.0.3',
 'matplotlib>=3.5.3',
 'networkx>=2.8.8',
 'numpy>=1.23.2',
 'orjson>=3.8.1',
 'poetry-bumpversion>=0.3.0',
 'psutil>=5.7.2',
 'pycomex>=0.6.1',
 'pytest>=7.2.0',
 'pyyaml>=0.6.0',
 'rdkit>=2022.9.0']

setup_kwargs = {
    'name': 'visual-graph-datasets',
    'version': '0.8.0',
    'description': 'Datasets for the training of graph neural networks (GNNs) and subsequent visualization of attributional explanations of XAI methods',
    'long_description': '|made-with-python| |python-version| |os-linux|\n\n.. |os-linux| image:: https://img.shields.io/badge/os-linux-orange.svg\n   :target: https://www.python.org/\n\n.. |python-version| image:: https://img.shields.io/badge/Python-3.8.0-green.svg\n   :target: https://www.python.org/\n\n.. |made-with-kgcnn| image:: https://img.shields.io/badge/Made%20with-KGCNN-blue.svg\n   :target: https://github.com/aimat-lab/gcnn_keras\n\n.. |made-with-python| image:: https://img.shields.io/badge/Made%20with-Python-1f425f.svg\n   :target: https://www.python.org/\n\n=====================\nVisual Graph Datasets\n=====================\n\nThis package supplies management and utilities for **graph datasets** used to train **graph neural networks**\nand more specifically aimed at **explainable AI (XAI)** methods\n\nW.r.t to the structure and management of these datasets this package employs a different philosophy. Instead of the\nusual minimal packaging to CSV files, a visual graph dataset (VGD) represents each **dataset as a folder** where\neach element is represented by two files:\n\n- A ``json`` file containing metadata information, including the **full graph representation**\n- A ``png`` file containing a canonical visualization of the graph.\n\nWe believe that providing a canonical graph representation as well as a canonical visualization will help to\nmake AI methods trained on such datasets more comparable. The canonical visualization and standard utilities for\nthe visualization of attributional XAI explanations specifically are aimed to improve the comparability and\nreproducability of XAI methods in the future.\n\nInstallation\n============\n\nFirst clone this repository:\n\n.. code-block:: console\n\n    git clone https://github/username/visual_graph_datasets.git\n\nThen install it like this:\n\n.. code-block:: console\n\n    cd visual_graph_datasets\n    pip3 install -e .\n\nCommand Line Interface\n======================\n\nDownload datasets\n-----------------\n\n    **NOTE**: We *strongly* encourage to store datasets on an SSD instead of an HDD, as this can make a\n    difference of multiple hours(!) when loading especially large datasets.\n\nDatasets can simply be downloaded by name by using the ``download`` command:\n\n.. code-block:: console\n\n    // Example for the dataset \'rb_dual_motifs\'\n    python3 -m visual_graph_datasets.cli download "rb_dual_motifs"\n\nBy default this dataset will be downloaded into the folder ``$HOME/.visual_graph_datasets/datasets``\nwhere HOME is the current users home directory.\n\nThe dataset download destination can be changed in a config file by using the ``config`` command:\n\n.. code-block:: console\n\n    python3 -m visual_graph_datasets.cli config\n\nThis command will open the config file at ``$HOME/.visual_graph_datasets/config.yaml`` using the systems\ndefault text editor.\n\nList available datasets\n-----------------------\n\nYou can display a list of all the currently available datasets of the current remote file share provider\nand some metadata information about them by using the command ``list``:\n\n.. code-block:: console\n\n    python3 -m visual_graph_datasets.cli list\n\nQuickstart\n==========\n\nThe datasets are mainly intended to be used in combination with other packages, but this package provides\nsome basic utilities to load and explore the datasets themselves within python programs.\n\n.. code-block:: python\n\n    import os\n    import typing as t\n    import matplotlib.pyplot as plt\n\n    from visual_graph_datasets.config import Config\n    from visual_graph_datasets.web import ensure_dataset\n    from visual_graph_datasets.data import load_visual_graph_dataset\n    from visual_graph_datasets.visualization.base import draw_image\n    from visual_graph_datasets.visualization.importances import plot_node_importances_border\n    from visual_graph_datasets.visualization.importances import plot_edge_importances_border\n\n    # This object will load the settings from the main config file. This config file contains options\n    # such as changing the default datasets folder and defining custom alternative file share providers\n    config = Config()\n    config.load()\n\n    # First of all we need to make sure that the dataset exists locally, this function will download it from\n    # the default file share provider if it does not exist.\n    ensure_dataset(\'rb_dual_motifs\', config)\n\n    # Afterwards we can be sure that the datasets exists and can now load it from the default datasets path.\n    # The data will be loaded as a dictionary whose int keys are the indices of the corresponding elements\n    # and the values are dictionaries which contain all the relevant data about the dataset element,\n    # (Dataset format is explained below)\n    dataset_path = os.path.join(config.get_datasets_path(), \'rb_dual_motifs\')\n    data_index_map: t.Dict[int, dict] = {}\n    _, data_index_map = load_visual_graph_dataset(dataset_path)\n\n    # Using this information we can visualize the ground truth importance explanation annotations for one\n    # element of the dataset like this.\n    index = 0\n    data = data_index_map[index]\n    # This is the dictionary which represents the graph structure of the dataset element. Descriptive\n    # string keys and numpy array values.\n    g = data[\'metadata\'][\'graph\']\n    fig, ax = plt.subplots(ncols=1, nrows=1, figsize=(10, 10))\n    draw_image(ax, image_path=data[\'image_path\'])\n    plot_node_importances_border(\n        ax=ax,\n        g=g,\n        node_positions=g[\'image_node_positions\'],\n        node_importances=g[\'node_importances_2\'][:, 0],\n    )\n    plot_edge_importances_border(\n        ax=ax,\n        g=g,\n        node_positions=g[\'image_node_positions\'],\n        edge_importances=g[\'edge_importances_2\'][:, 0],\n    )\n    fig_path = os.path.join(os.getcwd(), \'importances.pdf\')\n    fig.savefig(fig_path)\n\n\nDataset Format\n==============\n\nAssuming the following shape definitions:\n\n- V - the number of nodes in a graph\n- E - the number of edges in a graph\n- N - the number of node attributes / features associated with each node\n- M - the number of edge attributes / features associated with each edge\n- K - the number of importance channels\n\nOne such content dictionary which are the values of the two dicts returned by the function have the\nfollowing nested dictionary structure:\n\n- ``image_path``: The absolute path to the image file that visualizes this element\n- ``metadata_path``: the absolute path to the metadata file\n- ``metadata``: A dict which contains all the metadata for that element\n    - ``target``: a 1d array containing the target values for the element. For classification this usually\n      a one-hot encoded class vector already.\n    - ``index``: The canonical index of this element within the dataset\n    - ``graph``: A dictionary which contains the entire graph representation of this element.\n        - ``node_indices``: array of shape (V, 1) with the integer node indices.\n        - ``node_attributes``: array of shape (V, N)\n        - ``edge_indices``: array of shape (E, 2) which are the tuples of integer node indices that\n          determine edges\n        - ``edge_attributes``: array of shape (E, M)\n        - ``node_positions`` array of shape (V, 2) which are the xy positions of each node in pixel\n          values within the corresponding image visualization of the element. This is the crucial\n          information which is required to use the existing image representations to visualize attributional\n          explanations!\n        - (``node_importances_{K}_{suffix}``) array of shape (V, K) containing ground truth node importance\n          explanations, which assign an importance value of 0 to 1 to each node of the graph across K channels.\n          One dataset element may have none or multiple such annotations with different suffixes\n          determining the number of explanation channels and origin.\n        - (``edge_importances_{K}_{suffix}``) array of shape (E, K) containing ground truth edge importance\n          explanations, which assign an importance value of 0 to 1 to each edge of the graph across K channels.\n          One dataset element may have none or multiple such annotations with different suffixes\n          determining the number of explanation channels and origin.\n\n\nDatasets\n========\n\nHere is a list of the datasets currently included.\n\nFor more information about the individual datasets use the ``list`` command in the CLI (see above).\n\n* TO BE DONE\n\n',
    'author': 'awa59kst120df',
    'author_email': 'awa59kst120df@gmail.com',
    'maintainer': 'awa59kst120df',
    'maintainer_email': 'awa59kst120df@gmail.com',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8',
}


setup(**setup_kwargs)
