"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeSnippetWidget = void 0;
const jsx_runtime_1 = require("react/jsx-runtime");
/*
 * Copyright 2018-2022 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
require("../style/index.css");
const metadata_common_1 = require("@elyra/metadata-common");
const ui_components_1 = require("@elyra/ui-components");
const apputils_1 = require("@jupyterlab/apputils");
const cells_1 = require("@jupyterlab/cells");
const codeeditor_1 = require("@jupyterlab/codeeditor");
const coreutils_1 = require("@jupyterlab/coreutils");
const fileeditor_1 = require("@jupyterlab/fileeditor");
const notebook_1 = require("@jupyterlab/notebook");
const ui_components_2 = require("@jupyterlab/ui-components");
const algorithm_1 = require("@lumino/algorithm");
const coreutils_2 = require("@lumino/coreutils");
const dragdrop_1 = require("@lumino/dragdrop");
const CodeSnippetService_1 = require("./CodeSnippetService");
const METADATA_EDITOR_ID = 'elyra-metadata-editor';
const SNIPPET_DRAG_IMAGE_CLASS = 'elyra-codeSnippet-drag-image';
const CODE_SNIPPETS_METADATA_CLASS = 'elyra-metadata-code-snippets';
/**
 * The threshold in pixels to start a drag event.
 */
const DRAG_THRESHOLD = 5;
/**
 * The mimetype used for Jupyter cell data.
 */
const JUPYTER_CELL_MIME = 'application/vnd.jupyter.cells';
/**
 * A React Component for code-snippets display list.
 */
class CodeSnippetDisplay extends metadata_common_1.MetadataDisplay {
    constructor(props) {
        super(props);
        this.editors = {};
        // Handle code snippet insertion into an editor
        this.insertCodeSnippet = (snippet) => __awaiter(this, void 0, void 0, function* () {
            var _a, _b, _c, _d, _e, _f, _g, _h;
            const widget = this.props.getCurrentWidget();
            const codeSnippet = snippet.metadata.code.join('\n');
            const snippetLanguage = snippet.metadata.language;
            if (widget === null) {
                return;
            }
            if (this.isFileEditor(widget)) {
                const fileEditor = widget.content.editor;
                const markdownRegex = /^\.(md|mkdn?|mdown|markdown)$/;
                const editorLanguage = this.getEditorLanguage(widget);
                if (coreutils_1.PathExt.extname(widget.context.path).match(markdownRegex) !== null &&
                    snippetLanguage.toLowerCase() !== 'markdown') {
                    (_a = fileEditor.replaceSelection) === null || _a === void 0 ? void 0 : _a.call(fileEditor, this.addMarkdownCodeBlock(snippetLanguage, codeSnippet));
                }
                else if (editorLanguage) {
                    this.verifyLanguageAndInsert(snippet, editorLanguage, fileEditor);
                }
                else {
                    (_b = fileEditor.replaceSelection) === null || _b === void 0 ? void 0 : _b.call(fileEditor, codeSnippet);
                }
            }
            else if (widget instanceof notebook_1.NotebookPanel) {
                const notebookWidget = widget;
                const notebookCell = notebookWidget.content.activeCell;
                const notebookCellIndex = notebookWidget.content
                    .activeCellIndex;
                if (notebookCell === null) {
                    return;
                }
                const notebookCellEditor = notebookCell.editor;
                if (notebookCell instanceof cells_1.CodeCell) {
                    const kernelInfo = yield ((_d = (_c = notebookWidget.sessionContext.session) === null || _c === void 0 ? void 0 : _c.kernel) === null || _d === void 0 ? void 0 : _d.info);
                    const kernelLanguage = (kernelInfo === null || kernelInfo === void 0 ? void 0 : kernelInfo.language_info.name) || '';
                    this.verifyLanguageAndInsert(snippet, kernelLanguage, notebookCellEditor);
                }
                else if (notebookCell instanceof cells_1.MarkdownCell &&
                    snippetLanguage.toLowerCase() !== 'markdown') {
                    (_e = notebookCellEditor.replaceSelection) === null || _e === void 0 ? void 0 : _e.call(notebookCellEditor, this.addMarkdownCodeBlock(snippetLanguage, codeSnippet));
                }
                else {
                    (_f = notebookCellEditor.replaceSelection) === null || _f === void 0 ? void 0 : _f.call(notebookCellEditor, codeSnippet);
                }
                const cell = (_g = notebookWidget.model) === null || _g === void 0 ? void 0 : _g.contentFactory.createCodeCell({});
                if (cell === undefined) {
                    return;
                }
                (_h = notebookWidget.model) === null || _h === void 0 ? void 0 : _h.cells.insert(notebookCellIndex + 1, cell);
            }
            else {
                this.showErrDialog('Code snippet insert failed: Unsupported widget');
            }
        });
        // Verify if a given widget is a FileEditor
        this.isFileEditor = (widget) => {
            return widget.content instanceof fileeditor_1.FileEditor;
        };
        // Return the language of the editor or empty string
        this.getEditorLanguage = (widget) => {
            const editorLanguage = widget.context.sessionContext.kernelPreference.language;
            return editorLanguage === 'null' ? '' : editorLanguage;
        };
        // Return the given code wrapped in a markdown code block
        this.addMarkdownCodeBlock = (language, code) => {
            return '```' + language + '\n' + code + '\n```';
        };
        // Handle language compatibility between code snippet and editor
        this.verifyLanguageAndInsert = (snippet, editorLanguage, editor) => __awaiter(this, void 0, void 0, function* () {
            var _j, _k;
            const codeSnippet = snippet.metadata.code.join('\n');
            const snippetLanguage = snippet.metadata.language;
            if (editorLanguage &&
                snippetLanguage.toLowerCase() !== editorLanguage.toLowerCase()) {
                const result = yield this.showWarnDialog(editorLanguage, snippet.display_name);
                if (result.button.accept) {
                    (_j = editor.replaceSelection) === null || _j === void 0 ? void 0 : _j.call(editor, codeSnippet);
                }
            }
            else {
                // Language match or editorLanguage is unavailable
                (_k = editor.replaceSelection) === null || _k === void 0 ? void 0 : _k.call(editor, codeSnippet);
            }
        });
        // Display warning dialog when inserting a code snippet incompatible with editor's language
        this.showWarnDialog = (editorLanguage, snippetName) => __awaiter(this, void 0, void 0, function* () {
            return apputils_1.showDialog({
                title: 'Warning',
                body: `Code snippet "${snippetName}" is incompatible with ${editorLanguage}. Continue?`,
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()]
            });
        });
        // Display error dialog when inserting a code snippet into unsupported widget (i.e. not an editor)
        this.showErrDialog = (errMsg) => {
            return apputils_1.showDialog({
                title: 'Error',
                body: errMsg,
                buttons: [apputils_1.Dialog.okButton()]
            });
        };
        this.actionButtons = (metadata) => {
            return [
                {
                    title: 'Copy',
                    icon: ui_components_2.copyIcon,
                    feedback: 'Copied!',
                    onClick: () => {
                        apputils_1.Clipboard.copyToSystem(metadata.metadata.code.join('\n'));
                    }
                },
                {
                    title: 'Insert',
                    icon: ui_components_1.importIcon,
                    onClick: () => {
                        this.insertCodeSnippet(metadata);
                    }
                },
                {
                    title: 'Edit',
                    icon: ui_components_2.editIcon,
                    onClick: () => {
                        this.props.openMetadataEditor({
                            onSave: this.props.updateMetadata,
                            schemaspace: CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE,
                            schema: CodeSnippetService_1.CODE_SNIPPET_SCHEMA,
                            name: metadata.name
                        });
                    }
                },
                {
                    title: 'Delete',
                    icon: ui_components_1.trashIcon,
                    onClick: () => {
                        CodeSnippetService_1.CodeSnippetService.deleteCodeSnippet(metadata)
                            .then((deleted) => {
                            if (deleted) {
                                this.props.updateMetadata();
                                delete this.editors[metadata.name];
                                const editorWidget = algorithm_1.find(this.props.shell.widgets('main'), (value, index) => {
                                    return (value.id ==
                                        `${METADATA_EDITOR_ID}:${CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE}:${CodeSnippetService_1.CODE_SNIPPET_SCHEMA}:${metadata.name}`);
                                });
                                if (editorWidget) {
                                    editorWidget.dispose();
                                }
                            }
                        })
                            .catch(error => ui_components_1.RequestErrors.serverError(error));
                    }
                }
            ];
        };
        // Render display of a code snippet
        this.renderMetadata = (metadata) => {
            return (jsx_runtime_1.jsx("div", Object.assign({ "data-item-id": metadata.display_name, className: metadata_common_1.METADATA_ITEM, style: this.state.metadata.includes(metadata) ? {} : { display: 'none' } }, { children: jsx_runtime_1.jsx(ui_components_1.ExpandableComponent, Object.assign({ displayName: this.getDisplayName(metadata), tooltip: metadata.metadata.description, actionButtons: this.actionButtons(metadata), onExpand: () => {
                        this.editors[metadata.name].refresh();
                    }, onMouseDown: (event) => {
                        this.handleDragSnippet(event, metadata);
                    } }, { children: jsx_runtime_1.jsx("div", { id: metadata.name }, void 0) }), void 0) }), metadata.name));
        };
        this.createPreviewEditors = () => {
            const editorFactory = this.props.editorServices.factoryService
                .newInlineEditor;
            const getMimeTypeByLanguage = this.props.editorServices.mimeTypeService
                .getMimeTypeByLanguage;
            this.props.metadata.map((codeSnippet) => {
                if (codeSnippet.name in this.editors) {
                    // Make sure code is up to date
                    this.editors[codeSnippet.name].model.value.text = codeSnippet.metadata.code.join('\n');
                }
                else {
                    // Add new snippets
                    const snippetElement = document.getElementById(codeSnippet.name);
                    if (snippetElement === null) {
                        return;
                    }
                    this.editors[codeSnippet.name] = editorFactory({
                        config: { readOnly: true },
                        host: snippetElement,
                        model: new codeeditor_1.CodeEditor.Model({
                            value: codeSnippet.metadata.code.join('\n'),
                            mimeType: getMimeTypeByLanguage({
                                name: codeSnippet.metadata.language,
                                codemirror_mode: codeSnippet.metadata.language
                            })
                        })
                    });
                }
            });
        };
        this._drag = null;
        this._dragData = null;
        this.handleDragMove = this.handleDragMove.bind(this);
        this._evtMouseUp = this._evtMouseUp.bind(this);
    }
    // Initial setup to handle dragging a code snippet
    handleDragSnippet(event, metadata) {
        const { button } = event;
        // do nothing if left mouse button is clicked
        if (button !== 0) {
            return;
        }
        this._dragData = {
            pressX: event.clientX,
            pressY: event.clientY,
            dragImage: null
        };
        const mouseUpListener = (event) => {
            this._evtMouseUp(event, metadata, mouseMoveListener);
        };
        const mouseMoveListener = (event) => {
            this.handleDragMove(event, metadata, mouseMoveListener, mouseUpListener);
        };
        const target = event.target;
        target.addEventListener('mouseup', mouseUpListener, {
            once: true,
            capture: true
        });
        target.addEventListener('mousemove', mouseMoveListener, true);
        // since a browser has its own drag'n'drop support for images and some other elements.
        target.ondragstart = () => false;
    }
    _evtMouseUp(event, metadata, mouseMoveListener) {
        event.preventDefault();
        event.stopPropagation();
        const target = event.target;
        target.removeEventListener('mousemove', mouseMoveListener, true);
    }
    handleDragMove(event, metadata, mouseMoveListener, mouseUpListener) {
        event.preventDefault();
        event.stopPropagation();
        const data = this._dragData;
        if (data &&
            this.shouldStartDrag(data.pressX, data.pressY, event.clientX, event.clientY)) {
            // Create drag image
            const element = document.createElement('div');
            element.innerHTML = this.getDisplayName(metadata);
            element.classList.add(SNIPPET_DRAG_IMAGE_CLASS);
            data.dragImage = element;
            // Remove mouse listeners and start the drag.
            const target = event.target;
            target.removeEventListener('mousemove', mouseMoveListener, true);
            target.removeEventListener('mouseup', mouseUpListener, true);
            void this.startDrag(data.dragImage, metadata, event.clientX, event.clientY);
        }
    }
    /**
     * Detect if a drag event should be started. This is down if the
     * mouse is moved beyond a certain distance (DRAG_THRESHOLD).
     *
     * @param prevX - X Coordinate of the mouse pointer during the mousedown event
     * @param prevY - Y Coordinate of the mouse pointer during the mousedown event
     * @param nextX - Current X Coordinate of the mouse pointer
     * @param nextY - Current Y Coordinate of the mouse pointer
     */
    shouldStartDrag(prevX, prevY, nextX, nextY) {
        const dx = Math.abs(nextX - prevX);
        const dy = Math.abs(nextY - prevY);
        return dx >= 0 || dy >= DRAG_THRESHOLD;
    }
    startDrag(dragImage, metadata, clientX, clientY) {
        return __awaiter(this, void 0, void 0, function* () {
            const contentFactory = new notebook_1.NotebookModel.ContentFactory({});
            const language = metadata.metadata.language;
            const model = language.toLowerCase() !== 'markdown'
                ? contentFactory.createCodeCell({})
                : contentFactory.createMarkdownCell({});
            const content = metadata.metadata.code.join('\n');
            model.value.text = content;
            this._drag = new dragdrop_1.Drag({
                mimeData: new coreutils_2.MimeData(),
                dragImage: dragImage,
                supportedActions: 'copy-move',
                proposedAction: 'copy',
                source: this
            });
            const selected = [model.toJSON()];
            this._drag.mimeData.setData(JUPYTER_CELL_MIME, selected);
            this._drag.mimeData.setData('text/plain', content);
            return this._drag.start(clientX, clientY).then(() => {
                this._drag = null;
                this._dragData = null;
            });
        });
    }
    getDisplayName(metadata) {
        return `[${metadata.metadata.language}] ${metadata.display_name}`;
    }
    sortMetadata() {
        this.props.metadata.sort((a, b) => this.getDisplayName(a).localeCompare(this.getDisplayName(b)));
    }
    matchesSearch(searchValue, metadata) {
        searchValue = searchValue.toLowerCase();
        // True if search string is in name, display_name, or language of snippet
        // or if the search string is empty
        return (metadata.name.toLowerCase().includes(searchValue) ||
            metadata.display_name.toLowerCase().includes(searchValue) ||
            metadata.metadata.language.toLowerCase().includes(searchValue));
    }
    componentDidMount() {
        this.createPreviewEditors();
    }
    componentDidUpdate() {
        this.createPreviewEditors();
    }
}
/**
 * A widget for Code Snippets.
 */
class CodeSnippetWidget extends metadata_common_1.MetadataWidget {
    constructor(props) {
        super(props);
        this.props = props;
    }
    // Request code snippets from server
    fetchMetadata() {
        return __awaiter(this, void 0, void 0, function* () {
            return CodeSnippetService_1.CodeSnippetService.findAll().catch(error => ui_components_1.RequestErrors.serverError(error));
        });
    }
    renderDisplay(metadata) {
        if (Array.isArray(metadata) && !metadata.length) {
            // Empty metadata
            return (jsx_runtime_1.jsxs("div", { children: [jsx_runtime_1.jsx("br", {}, void 0),
                    jsx_runtime_1.jsx("h6", Object.assign({ className: "elyra-no-metadata-msg" }, { children: "Click the + button to add a new Code Snippet" }), void 0)] }, void 0));
        }
        return (jsx_runtime_1.jsx(CodeSnippetDisplay, { metadata: metadata, openMetadataEditor: this.openMetadataEditor, updateMetadata: this.updateMetadata, schemaspace: CodeSnippetService_1.CODE_SNIPPET_SCHEMASPACE, schema: CodeSnippetService_1.CODE_SNIPPET_SCHEMA, getCurrentWidget: this.props.getCurrentWidget, className: CODE_SNIPPETS_METADATA_CLASS, editorServices: this.props.editorServices, shell: this.props.app.shell, sortMetadata: true }, void 0));
    }
}
exports.CodeSnippetWidget = CodeSnippetWidget;
//# sourceMappingURL=CodeSnippetWidget.js.map