"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.sonarqubeScanner = exports.createSonarqubeProject = exports.generateSonarqubeReports = void 0;
const cube = (path, action = "GET") => `curl -X ${action} -u $SONARQUBE_TOKEN: "$SONARQUBE_ENDPOINT${path}"`;
const setupSonarqubeQualityGates = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_GATE=\`${cube(`/api/qualitygates/search?gateName=${defaultProfileOrGateName}`)}\``,
    `export SPECIFIC_GATE=\`${cube(`/api/qualitygates/search?gateName=${specificProfileOrGateName}`)}\``,
    `if [[ "$(echo $SPECIFIC_GATE | jq .errors)" == "null" && "$(echo $SPECIFIC_GATE | jq '.results | length')" -gt 0 ]]; then export GATE_NAME=${specificProfileOrGateName}; else export GATE_NAME=${defaultProfileOrGateName}; fi`,
    `${cube("/api/qualitygates/select?projectKey=$PROJECT_NAME&gateName=$GATE_NAME", "POST")}`,
];
const setupSonarqubeQualityProfiles = (defaultProfileOrGateName, specificProfileOrGateName) => [
    `export DEFAULT_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${defaultProfileOrGateName}`)} | jq .profiles\``,
    `export SPECIFIC_PROFILE=\`${cube(`/api/qualityprofiles/search?qualityProfile=${specificProfileOrGateName}`)} | jq .profiles\``,
    `export MERGED_PROFILES=\`jq --argjson arr1 "$DEFAULT_PROFILE" --argjson arr2 "$SPECIFIC_PROFILE" -n '$arr1 + $arr2 | group_by(.language) | map(.[-1])'\``,
    `echo $MERGED_PROFILES | jq -c '.[]' | while read i; do ${cube("/api/qualityprofiles/add_project?project=$PROJECT_NAME&language=`echo $i | jq -r .language`&qualityProfile=`echo $i | jq -r .name`", "POST")}; done`,
];
const setupSonarqubePermissions = (authorizedGroup) => !authorizedGroup
    ? []
    : [
        "admin",
        "codeviewer",
        "issueadmin",
        "securityhotspotadmin",
        "scan",
        "user",
    ].map((p) => `${cube(`/api/permissions/add_group?projectKey=$PROJECT_NAME&groupName=${authorizedGroup}&permission=${p}`, "POST")}`);
const setupSonarqubeProject = ({ sonarqubeTags, sonarqubeAuthorizedGroup, sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName, }) => {
    return [
        ...setupSonarqubePermissions(sonarqubeAuthorizedGroup),
        `${cube("/api/project_branches/rename?project=$PROJECT_NAME&name=mainline", "POST")}`,
        `${cube(`/api/project_tags/set?project=$PROJECT_NAME&tags=${[
            sonarqubeAuthorizedGroup,
            ...(sonarqubeTags || []),
        ].join(",")}`, "POST")}`,
        ...setupSonarqubeQualityProfiles(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
        ...setupSonarqubeQualityGates(sonarqubeDefaultProfileOrGateName, sonarqubeSpecificProfileOrGateName),
    ].join(";");
};
exports.generateSonarqubeReports = () => [
    cube("/api/bitegarden/report/pdf_issues_breakdown?resource=$PROJECT_NAME&branch=mainline --output reports/prototype-issues-report.pdf"),
    cube("/api/bitegarden/report/pdf?resource=$PROJECT_NAME&branch=mainline --output reports/prototype-executive-report.pdf"),
    cube("/api/security_reports/download?project=$PROJECT_NAME --output reports/prototype-security-report.pdf"),
];
exports.createSonarqubeProject = (props) => [
    `CREATE_PROJECT_OUTPUT=\`${cube("/api/projects/create?name=$PROJECT_NAME&project=$PROJECT_NAME&visibility=private", "POST")}\``,
    `if [[ "$(echo $CREATE_PROJECT_OUTPUT | jq .errors)" == "null" ]]; then ${setupSonarqubeProject(props)}; fi;`,
];
exports.sonarqubeScanner = () => [
    "npx sonarqube-scanner -Dsonar.login=$SONARQUBE_TOKEN",
    "-Dsonar.projectKey=$PROJECT_NAME",
    "-Dsonar.projectName=$PROJECT_NAME",
    "-Dsonar.projectVersion=`git rev-parse --short HEAD`",
    "-Dsonar.branch.name=mainline",
    "-Dsonar.host.url=$SONARQUBE_ENDPOINT",
    "-Dsonar.cfn.nag.reportFiles=reports/cfn-nag-report.json",
    "-Dsonar.dependencyCheck.htmlReportPath=reports/dependency-check-report.html",
].join(" ");
//# sourceMappingURL=data:application/json;base64,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