"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.49" };
//# sourceMappingURL=data:application/json;base64,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