from datetime import datetime
import logging
import os
import shutil
import time
from typing import Dict

from .MTFSupporting import ModelExperimentBaseError, ExperimentOption
from .ModelTracker import ModelTracker

logger = logging.getLogger("mtf_logger")

# TODO: Midway through converting all params to class definition
class ModelExperimentBase(ModelTracker):

    def __init__(self, model_name:str, debug_skips_preprop_steps:bool, 
                 existing_tracker_path:str, exp_description:str, 
                 parent_sv_dir:str, prev_run_notes:str="", 
                 preprop_kwargs:Dict ={}, preprop_debug_kwargs:Dict={},
                 train_kwargs:Dict={}, updt_kwargs: dict = {}, 
                 debug_sv_dir:str=None, force_columns:bool=False):
        """ModelExperimentBase contains keys methods for running experiments and updating an underlying model tracker.
        An experiment is defined as a single observation in the model tracker. 
        The class has been designed such that a (or multiple) "ProjectModelExperiment" class(es) is (are) defined which implement 
        generic sections of the model development process such as the preprocessing stages and evaluation stages. Individual 
        experiments can subsiquently inherit from the "ProjectModelExperiment", implementing experiment specific changes whilst still 
        ultimately updating the same underlying model tracker.
        self.results is utilised by self.run_experiment and can be updated by any method. It is intended to hold model evaluation 
        metrics but can store any experiment specific values which are required to be incorporated into the model tracker. The attribute
        is of the form {metric_names(column title in tracker):metric_value}
        self.model_sv_loc is the location into which self should save physical outputs such as graphs. This is updated by 
        _create_output_sub_loc

        Args:
            model_name (str): Name of the experiment. If inheriting this class, this variable should not be perminently defined.
            debug_skips_preprop_steps (bool): Defines whether self.preprocessing_debug replaces self.preprocessing_steps or follows it 
            when running in debug model. If set to True, the self.preprocessing_debug will replace self.preprocessing_steps. 
            This attribute is designed to be perminently set wherever the preprocessing steps are defined.
            existing_tracker_path (str): Location of existing json tracker to update. 
            exp_description (str): A description of the experiment. 
            parent_sv_dir (str): The location of the parent directory where the subdirectory should be made to store any 
            outputs such as graphs.. 
            prev_run_notes (str, optional): A description of the differences compared to a previous experiment. Defaults to "".
            preprop_kwargs (dict, optional): kwargs relating to the self.preprocessing_steps
            preprop_debug_kwargs (dict, optional): kwargs relating to the self.preprocessing_debug
            train_kwargs (dict, optional): kwargs relating to self.train_model. Defaults to {}.
            debug_sv_dir (str): Assigns location to model_save_loc if the debug option is selected
            force_columns (bool): Option to force new columns into the tracker
            updt_kwargs (dict, optional): kwargs relating to self.import_existing_json_tracker. See ModelTracker for more 
            info. Defaults to {}.
        """
        super().__init__() 
        print("ModelExperimentBase init")
        print(existing_tracker_path)
        self.debug_skips_preprop_steps = debug_skips_preprop_steps
        self.model_name = model_name
        self.results = {}
        self.model_sv_loc = None
        self.existing_tracker_path = existing_tracker_path
        self.exp_description = exp_description 
        self.parent_sv_dir = parent_sv_dir 
        self.prev_run_notes = prev_run_notes 
        self.preprop_kwargs = preprop_kwargs
        self.preprop_debug_kwargs = preprop_debug_kwargs 
        self.train_kwargs = train_kwargs
        self.updt_kwargs = updt_kwargs 
        self.debug_sv_dir = debug_sv_dir 
        self.force_columns = force_columns


    def _create_output_sub_loc(self, parent_loc:str, sub_dir_nm: str = None):
        """Creates directory within the parent_loc directory with the model_name as the directory name

        Args:
            parent_loc (str): Location of in which the subdirectory should be set. It is advised to set this at a project level
            sub_dir_nm (str): The name of the created directory. If set to None, this will default to self.model_name. This 
            feature has primarily been implemented for use in the run_experiment method when the user specifies the "duplicate"
            option. Defaults to None.
        """
        # TODO: Confirm better way of handling this!
        if sub_dir_nm is None: 
            sub_dir_nm = self.model_name
        sub_loc_path = os.path.join(parent_loc, sub_dir_nm)
        os.mkdir(sub_loc_path)
        self.model_sv_loc = sub_loc_path

    def evaluate_model(self):
        raise NotImplementedError("evaluate_model method should be implemented on a experiment type basis or pass if not required.")
    
    def preprocessing_steps(self, **kwargs):
        raise NotImplementedError("preprocessing_steps method should be implemented on a experiment type basis or pass if not required.")

    def preprocessing_debug(self, **kwargs):
        # Note: If the debug_skips_preprop_steps attribute is set to True, this behaves as an alternate set of preprocessing steps 
        # to preprocessing_steps i.e. if a different dataset should be imported. 
        # If debug_skips_preprop_steps is set to False, this behaves as a postprocessing to the preprocessing_steps i.e. it could sample 
        # from a dataset generated by preprocessing_steps
        raise NotImplementedError("preprocessing_debug method should be implemented on a experiment type basis or pass if not required.")

    def preprocessing(self, debug: bool=False):
        """Runs any preprocessing steps implemented in self.preprocessing_steps and/or self.preprocessing_debug when in debug mode.

        Args:
            debug (bool, optional): Specifies whether the preprocessing should be run in debug mode. Refer to self.run_experiment
            for more information. Defaults to False.
        """
        if debug:
            if not self.debug_skips_preprop_steps:
                self.preprocessing_steps(**self.preprop_kwargs)
            try:
                self.preprocessing_debug(**self.preprop_debug_kwargs)
            except NotImplementedError:
                logger.warning("preprocessing_debug not implemented however skipping error.")
        else:
            self.preprocessing_steps(**self.preprop_kwargs)

    def train_model(self, **kwargs):
        raise NotImplementedError("train_model method should be implemented on a per experiment basis")

    def run_experiment(self, debug=False, 
                       dupe_model_nms: ExperimentOption = ExperimentOption(
                           None)):
        """Runs an experiment in either 'normal' or debug mode specified by the debug parameter. 
        An experiment in 'normal' mode consists of the following:
        1. Create or import an existing tracker from json. If the tracker is imported, check whether an entry with the same 
        self.model_name exists. If it does, handle depending on what is specified by dupe_model_nms
        2. If preprocessing steps have been implemented in self.preprocessing_steps run them
        3. Train the model using self.train_model
        4. Evaluate the model outputs using self.evaluate_model 
        5. Update the tracker and re-save the json

        If run in debug mode. No results will be saved and self.preprocessing will be run in debug mode. Refer to 
        self.preprocessing_debug for more information on how self.preprocessing is effected by debug mode
        

        Args:
            dupe_model_nms (ExperimentOption, optional): Defines what action should be taken if an experiment with the same 
            self.model_name is found in the tracker. 
            ExperimentOption('overwrite') will overwrite the previous experiment. 
            ExperimentOption('duplicate') will add both experiments and will append _1 to the new experiment (note if self.model_name_1 also exists, _2 
            will be used etc)
            debug (bool, optional): Defines whether the experiment should be run in debug mode. Defaults to False.
        """
        if debug:
            logger.info(" ***** Running in debug mode ***** ")
            logger.info("""No results will be captured in debug mode. 
            If preprocessing method has been defined with debug facilities, this will also run.""")
            self.model_sv_loc = self.debug_sv_dir
            try:
                self.preprocessing(debug=True)
            except NotImplementedError:
                logger.warning("preprocessing_steps not implemented however skipping error.")
            self.train_model(**self.train_kwargs)

        else:
            logger.info(" ***** Importing existing tracker ***** ")
            if self.check_tracker_exists(existing_tracker_path=
                                         self.existing_tracker_path):
                logger.info("Tracker identified. Importing...")
                # TODO: Implement the ability to specify what type of tracker to import i.e. dataframe/csv/json etc
                self.import_existing_json_tracker(self.existing_tracker_path, 
                                                  **self.updt_kwargs)
            else:
                logger.info("Could not find tracker at location, creating new tracker")

            logger.info(" ***** Checking whether model exists in tracker ***** ")
            curr_model_nms = [rw["model_name"] for rw in self.rows]
            if self.model_name in curr_model_nms:
                logger.info("{} run already exists".format(self.model_name))
                if dupe_model_nms.exp_option == "overwrite":
                    logger.info("Overwriting previous run")
                    # Cannot just use index incase duplicates have already slipped through!
                    dupe_indices = [idx for idx, mdl_nm in enumerate(curr_model_nms) if mdl_nm == self.model_name]
                    dupe_indices.sort(reverse=True)
                    for idx in dupe_indices:
                        del self.rows[idx]
                    if os.path.exists(os.path.join(self.parent_sv_dir, 
                                                   self.model_name)):
                        shutil.rmtree(os.path.join(self.parent_sv_dir, 
                                                   self.model_name))
                        dir_not_rmved = True
                        while(dir_not_rmved):
                            time.sleep(1)
                            dir_not_rmved = os.path.exists(os.path.join(
                                self.parent_sv_dir, self.model_name))
                    self._create_output_sub_loc(self.parent_sv_dir)
                elif dupe_model_nms.exp_option == "duplicate":
                    logger.info("Keeping both runs")
                    avail_loc = False
                    sup_dir_idx = 1
                    # Loop through adding 1 to the end each time to find available name to save model 
                    while avail_loc == False:
                        sup_dir = "{}_{}".format(self.model_name, sup_dir_idx)
                        if not os.path.exists(os.path.join(self.parent_sv_dir, 
                                                           sup_dir)):
                            avail_loc = True
                            self._create_output_sub_loc(self.parent_sv_dir, 
                                                        sup_dir)
                        sup_dir_idx  =+ 1
                elif dupe_model_nms.exp_option == None:
                    raise ModelExperimentBaseError(
                "Run with model name {} already exists therefore duplicate or overwrite must be specified in the dupe_model_nms option".format(self.model_name))
            else:
                self._create_output_sub_loc(self.parent_sv_dir)

            logger.info(" **** Training model ***** ")
            train_time_strt = datetime.now().strftime("%d/%m/%Y %H:%M:%S")
            try:
                self.preprocessing()
            except NotImplementedError:
                logger.warning("preprocessing_steps not implemented however skipping error.")
            self.train_model(**self.train_kwargs)
            train_time_end = datetime.now().strftime("%d/%m/%Y %H:%M:%S")
            logger.info(" ***** Evaluating model ***** ")
            self.evaluate_model()
            new_tracker_line = {**self.results,
                                "model_name": self.model_name,
                                "experiment_description": self.exp_description,
                                "prev_run_notes": self.prev_run_notes,
                                "train_time_strt": train_time_strt,
                                "train_time_end": train_time_end,
                                "output_save_location": self.model_sv_loc,
                                **self.train_kwargs, **self.preprop_kwargs
                                }
            logger.info(" ***** Updating tracker ***** ")
            self.update_tracker_w_dict(new_tracker_line, 
                                       force_columns=self.force_columns)
            logger.info(" ***** Updating json ***** ")
            self.tracker_to_json(json_dir=self.existing_tracker_path)


