"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureLambda = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const path = require("path");
function ensureLambda(scope, timeout) {
    const stack = cdk.Stack.of(scope);
    const lambdaName = 'DbMigrationManager';
    const existing = stack.node.tryFindChild(lambdaName);
    const roleName = `${lambdaName}-Role`;
    if (existing) {
        return {
            function: existing,
            role: stack.node.tryFindChild(roleName),
        };
    }
    const role = new iam.Role(stack, roleName, {
        roleName: `${stack.stackName}-${lambdaName}`,
        description: `Used by Lambda ${lambdaName}, which is a custom CFN resource, managing DB migrations`,
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
        ],
    });
    const fn = new lambda.Function(stack, lambdaName, {
        functionName: `${stack.stackName}-${lambdaName}`,
        role: role,
        description: 'Custom CFN resource: Manages DB migrations',
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
        timeout: timeout || cdk.Duration.seconds(30),
    });
    return {
        function: fn,
        role: role,
    };
}
exports.ensureLambda = ensureLambda;
//# sourceMappingURL=data:application/json;base64,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