"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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