"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * Elasticsearch version.
 *
 * @stability stable
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @stability stable
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
_a = JSII_RTTI_SYMBOL_1;
ElasticsearchVersion[_a] = { fqn: "@aws-cdk/aws-elasticsearch.ElasticsearchVersion", version: "1.138.0" };
/**
 * AWS Elasticsearch 1.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * AWS Elasticsearch 2.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * AWS Elasticsearch 5.1.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * AWS Elasticsearch 5.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * AWS Elasticsearch 5.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * AWS Elasticsearch 5.6.
 *
 * @stability stable
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * AWS Elasticsearch 6.0.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * AWS Elasticsearch 6.2.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * AWS Elasticsearch 6.3.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * AWS Elasticsearch 6.4.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * AWS Elasticsearch 6.5.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * AWS Elasticsearch 6.7.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * AWS Elasticsearch 6.8.
 *
 * @stability stable
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * AWS Elasticsearch 7.1.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * AWS Elasticsearch 7.4.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * AWS Elasticsearch 7.7.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * AWS Elasticsearch 7.8.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_8 = ElasticsearchVersion.of('7.8');
/**
 * AWS Elasticsearch 7.9.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_9 = ElasticsearchVersion.of('7.9');
/**
 * AWS Elasticsearch 7.10.
 *
 * @stability stable
 */
ElasticsearchVersion.V7_10 = ElasticsearchVersion.of('7.10');
/**
 * The minimum TLS version required for traffic to the domain.
 *
 * @stability stable
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @stability stable
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @stability stable
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this Domain.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @stability stable
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @stability stable
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @stability stable
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @stability stable
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @stability stable
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Elasticsearch domain.
 *
 * @stability stable
 */
class Domain extends DomainBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32;
        super(scope, id, {
            physicalName: props.domainName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_elasticsearch_DomainProps(props);
        const defaultInstanceType = 'r5.large.elasticsearch';
        const warmDefaultInstanceType = 'ultrawarm1.medium.elasticsearch';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, (_c = props.capacity) === null || _c === void 0 ? void 0 : _c.masterNodeInstanceType);
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType);
        const instanceCount = (_h = (_g = props.capacity) === null || _g === void 0 ? void 0 : _g.dataNodes) !== null && _h !== void 0 ? _h : 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.warmInstanceType);
        const warmCount = (_l = (_k = props.capacity) === null || _k === void 0 ? void 0 : _k.warmNodes) !== null && _l !== void 0 ? _l : 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = (_o = (_m = props.zoneAwareness) === null || _m === void 0 ? void 0 : _m.availabilityZoneCount) !== null && _o !== void 0 ? _o : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_q = (_p = props.zoneAwareness) === null || _p === void 0 ? void 0 : _p.enabled) !== null && _q !== void 0 ? _q : ((_r = props.zoneAwareness) === null || _r === void 0 ? void 0 : _r.availabilityZoneCount) != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, (_s = props.vpcSubnets) !== null && _s !== void 0 ? _s : [{ subnetType: ec2.SubnetType.PRIVATE }]);
            securityGroups = (_t = props.securityGroups) !== null && _t !== void 0 ? _t : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.elasticsearch')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".elasticsearch".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_u = props.useUnsignedBasicAuth) !== null && _u !== void 0 ? _u : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_v = props.encryptionAtRest) === null || _v === void 0 ? void 0 : _v.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserArn;
        const masterUserNameProps = (_x = props.fineGrainedAccessControl) === null || _x === void 0 ? void 0 : _x.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_y = props.fineGrainedAccessControl) === null || _y === void 0 ? void 0 : _y.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_0 = (_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.enabled) !== null && _0 !== void 0 ? _0 : (((_1 = props.encryptionAtRest) === null || _1 === void 0 ? void 0 : _1.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_2 = props.nodeToNodeEncryption) !== null && _2 !== void 0 ? _2 : unsignedBasicAuthEnabled;
        const volumeSize = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeSize) !== null && _4 !== void 0 ? _4 : 10;
        const volumeType = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.volumeType) !== null && _6 !== void 0 ? _6 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_8 = (_7 = props.ebs) === null || _7 === void 0 ? void 0 : _7.enabled) !== null && _8 !== void 0 ? _8 : true;
        const enforceHttps = (_9 = props.enforceHttps) !== null && _9 !== void 0 ? _9 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryDatanodeInstanceType(...instanceTypes) {
            return instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.appLogEnabled) {
                throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.8 && warmEnabled) {
            throw new Error('UltraWarm requires Elasticsearch 6.8 or later.');
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryDatanodeInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_13 = props.logging) === null || _13 === void 0 ? void 0 : _13.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_14 = props.logging.slowSearchLogGroup) !== null && _14 !== void 0 ? _14 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_16 = props.logging.slowIndexLogGroup) !== null && _16 !== void 0 ? _16 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.appLogEnabled) {
            this.appLogGroup = (_18 = props.logging.appLogGroup) !== null && _18 !== void 0 ? _18 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_19 = props.logging) === null || _19 === void 0 ? void 0 : _19.auditLogEnabled) {
            this.auditLogGroup = (_20 = props.logging.auditLogGroup) !== null && _20 !== void 0 ? _20 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_21 = this.auditLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_22 = props.ebs) === null || _22 === void 0 ? void 0 : _22.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_24 = (_23 = props.encryptionAtRest) === null || _23 === void 0 ? void 0 : _23.kmsKey) === null || _24 === void 0 ? void 0 : _24.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.identityPoolId,
                roleArn: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.role.roleArn,
                userPoolId: (_27 = props.cognitoKibanaAuth) === null || _27 === void 0 ? void 0 : _27.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_28 = props.tlsSecurityPolicy) !== null && _28 !== void 0 ? _28 : TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_29 = this.masterUserPassword) === null || _29 === void 0 ? void 0 : _29.toString(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if ((_30 = props.customEndpoint) === null || _30 === void 0 ? void 0 : _30.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_31 = props.accessPolicies) !== null && _31 !== void 0 ? _31 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_32 = props.encryptionAtRest) === null || _32 === void 0 ? void 0 : _32.kmsKey) {
                // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @stability stable
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @stability stable
     */
    static fromDomainAttributes(scope, id, attrs) {
        var _c;
        jsiiDeprecationWarnings._aws_cdk_aws_elasticsearch_DomainAttributes(attrs);
        const { domainArn, domainEndpoint } = attrs;
        const domainName = (_c = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName) !== null && _c !== void 0 ? _c : extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint.replace(/^https?:\/\//, '');
            }
        };
    }
    /**
     * Manages network connections to the domain.
     *
     * This will throw an error in case the domain
     * is not placed inside a VPC.
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
}
exports.Domain = Domain;
_b = JSII_RTTI_SYMBOL_1;
Domain[_b] = { fqn: "@aws-cdk/aws-elasticsearch.Domain", version: "1.138.0" };
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,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