"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const aws_lambda_shared_1 = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-docgen');
jest.mock('../../../backend/shared/code-artifact.lambda-shared');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    process.env.TARGET_LANGUAGE = 'typescript';
    done();
});
afterEach((done) => {
    AWSMock.restore();
    delete process.env.TARGET_LANGUAGE;
    aws_lambda_shared_1.reset();
    done();
});
describe('VPC Endpoints', () => {
    const previousEnv = process.env;
    const endpoint = 'codeartifact.d.bermuda-triangle-1.amazonaws.com';
    const apiEndpoint = 'codeartifact.api.bermuda-triangle-1.amazonaws.com';
    const domain = 'domain-name';
    const domainOwner = '123456789012';
    beforeAll(() => {
        process.env = {
            ...previousEnv,
            CODE_ARTIFACT_REPOSITORY_ENDPOINT: endpoint,
            CODE_ARTIFACT_DOMAIN_NAME: domain,
            CODE_ARTIFACT_DOMAIN_OWNER: domainOwner,
            CODE_ARTIFACT_API_ENDPOINT: apiEndpoint,
        };
    });
    afterAll(() => {
        process.env = { ...previousEnv };
    });
    test('happy path', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageScope = 'scope';
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const event = {
            bucket: 'dummy-bucket',
            assembly: {
                key: `${constants.STORAGE_KEY_PREFIX}@${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                versionId: 'VersionId',
            },
        };
        const assembly = {
            targets: { python: {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs('/docs-typescript.md');
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.length).toEqual(1);
        expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('../../../backend/shared/code-artifact.lambda-shared').logInWithCodeArtifact).toHaveBeenCalledWith({
            endpoint,
            domain,
            domainOwner,
            apiEndpoint,
        });
    });
});
test('uploads a file per language (scoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const event = {
        bucket: 'dummy-bucket',
        assembly: {
            key: `${constants.STORAGE_KEY_PREFIX}@${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
            versionId: 'VersionId',
        },
    };
    const assembly = {
        targets: { python: {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.length).toEqual(1);
    expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
});
test('uploads a file per submodule (unscoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const event = {
        bucket: 'dummy-bucket',
        assembly: {
            key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
            versionId: 'VersionId',
        },
    };
    const assembly = {
        targets: { python: {} },
        submodules: { '@scope/package-name.sub1': {}, '@scope/package-name.sub2': {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md', '/docs-sub1-typescript.md', '/docs-sub2-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.map(({ key }) => key)).toEqual([
        `data/${packageName}/v${packageVersion}/docs-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub1-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub2-typescript.md`,
    ]);
});
describe('markers for un-supported languages', () => {
    beforeEach((done) => {
        // Switch language, as TypeScript is always supported 🙃
        process.env.TARGET_LANGUAGE = 'python';
        done();
    });
    afterEach((done) => {
        delete process.env.TARGET_LANGUAGE;
        done();
    });
    test('uploads ".not-supported" markers as relevant', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const event = {
            bucket: 'dummy-bucket',
            assembly: {
                key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                versionId: 'VersionId',
            },
        };
        const assembly = {
            targets: { phony: {} },
            submodules: { 'package-name.sub1': {}, 'package-name.sub2': {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs(`/docs-python.md${constants.NOT_SUPPORTED_SUFFIX}`, `/docs-sub1-python.md${constants.NOT_SUPPORTED_SUFFIX}`, `/docs-sub2-python.md${constants.NOT_SUPPORTED_SUFFIX}`);
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.map(({ key }) => key)).toEqual([
            `data/${packageName}/v${packageVersion}/docs-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
            `data/${packageName}/v${packageVersion}/docs-sub1-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
            `data/${packageName}/v${packageVersion}/docs-sub2-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
        ]);
    });
});
class MockDocumentation {
    constructor(target) {
        this.target = target;
    }
    render() {
        return {
            render: () => `docs for ${this.target}`,
        };
    }
}
function mockFetchAssembly(response) {
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        if (request.Key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            callback(null, {
                Body: JSON.stringify(response),
            });
        }
        else {
            throw new Error(`Unexpected GET request: ${request.Key}`);
        }
    });
}
function mockPutDocs(...suffixes) {
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        if (suffixes.filter(s => request.Key.endsWith(s)).length > 0) {
            callback(null, { VersionId: `versionId-${request.Key}` });
        }
        else {
            throw new Error(`Unexpected PUT request: ${request.Key}`);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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