"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // determine if this is a request to rebuild the catalog (basically, an empty event)
    const rebuild = !(event === null || event === void 0 ? void 0 : event.package);
    if (rebuild) {
        console.log('Requesting catalog rebuild (empty event)');
    }
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    let data;
    if (!rebuild) {
        console.log('Loading existing catalog...');
        data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
    }
    // if event is empty, we're doing a full rebuild
    if (!(data === null || data === void 0 ? void 0 : data.Body) || rebuild) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST Python or TypeScript docs.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) && !(await aws.s3ObjectExists(bucket, pyDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2F0YWxvZy1idWlsZGVyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2NhdGFsb2ctYnVpbGRlci9jYXRhbG9nLWJ1aWxkZXIubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUE4QjtBQUc5QiwrREFBeUQ7QUFHekQsbUNBQWdDO0FBQ2hDLDJDQUFxQztBQUNyQyw0RUFBbUU7QUFFbkUsbURBQW1EO0FBQ25ELGlEQUFpRDtBQUNqRCxtRUFBeUQ7QUFFekQsTUFBTSxpQkFBaUIsR0FBRyw2QkFBNkIsQ0FBQztBQUV4RDs7Ozs7Ozs7O0dBU0c7QUFDSSxLQUFLLFVBQVUsT0FBTyxDQUFDLEtBQTBCLEVBQUUsT0FBZ0I7SUFDeEUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUU1QyxvRkFBb0Y7SUFDcEYsTUFBTSxPQUFPLEdBQUcsRUFBQyxLQUFLLGFBQUwsS0FBSyx1QkFBTCxLQUFLLENBQUUsT0FBTyxDQUFBLENBQUM7SUFDaEMsSUFBSSxPQUFPLEVBQUU7UUFDWCxPQUFPLENBQUMsR0FBRyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7S0FDekQ7SUFFRCxNQUFNLFdBQVcsR0FBRyw4QkFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRTlDLE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxFQUFvQyxDQUFDO0lBQzdELE1BQU0sUUFBUSxHQUFHLE1BQU0scUNBQWMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUVsRCxJQUFJLElBQXdDLENBQUM7SUFFN0MsSUFBSSxDQUFDLE9BQU8sRUFBRTtRQUNaLE9BQU8sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUUzQyxJQUFJLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsU0FBUyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFO2FBQzNGLEtBQUssQ0FBQyxDQUFDLEdBQWEsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksS0FBSyxXQUFXO1lBQ2hELENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQztZQUNyQixDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLGFBQWEsQ0FBd0IsQ0FBQyxDQUFDLENBQUM7S0FDakU7SUFFRCxnREFBZ0Q7SUFDaEQsSUFBSSxFQUFDLElBQUksYUFBSixJQUFJLHVCQUFKLElBQUksQ0FBRSxJQUFJLENBQUEsSUFBSSxPQUFPLEVBQUU7UUFDMUIsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sUUFBUSxHQUFRLEVBQUUsQ0FBQztRQUN6QixJQUFJLEtBQUssRUFBRSxNQUFNLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxJQUFJLGVBQWUsQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUNoRSxJQUFJO2dCQUNGLE1BQU0sYUFBYSxDQUFDLFFBQVEsRUFBRSxNQUFPLEVBQUUsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO2FBQy9EO1lBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQ1YsUUFBUSxDQUFDLE1BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUN2QjtTQUNGO1FBQ0QsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDbkQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsR0FBRyxLQUFLLEtBQUssRUFBRSxDQUFDLENBQUM7U0FDbkQ7UUFFRCxNQUFNLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssSUFBSSxFQUFFO1lBQ3hDLE9BQU8sQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUN4QyxNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sQ0FBQztZQUNqRCxPQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsV0FBVyxrQkFBa0IsQ0FBQyxDQUFDO1lBQ3RELE9BQU8sQ0FBQyxTQUFTLENBQUMsNEJBQTRCLEVBQUUsV0FBVyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0UsQ0FBQyxDQUFDLEVBQUUsQ0FBQztLQUVOO1NBQU07UUFDTCxPQUFPLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDekMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ3hELEtBQUssTUFBTSxJQUFJLElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUNuQyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQzVCLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLEdBQUcsRUFBRSxDQUFDLENBQUM7YUFDcEM7WUFDRCxRQUFRLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNoRDtRQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUMzQyxrRUFBa0U7UUFDbEUsNkNBQTZDO1FBQzdDLE1BQU0sYUFBYSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxXQUFXLEVBQUUsUUFBUSxDQUFDLENBQUM7S0FDekU7SUFFRCxrQ0FBa0M7SUFDbEMsT0FBTyxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQ3hDLE1BQU0sT0FBTyxHQUFHLEVBQUUsUUFBUSxFQUFFLElBQUksS0FBSyxFQUFlLEVBQUUsT0FBTyxFQUFFLElBQUksSUFBSSxFQUFFLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztJQUMxRixLQUFLLE1BQU0sTUFBTSxJQUFJLFFBQVEsQ0FBQyxNQUFNLEVBQUUsRUFBRTtRQUN0QyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNqQyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUM1QjtLQUNGO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLG9DQUFvQyxDQUFDLENBQUM7SUFDMUYsTUFBTSxrQ0FBVyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxLQUFLLElBQUksRUFBRTtRQUN4QyxPQUFPLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDeEMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxnQ0FBZ0MsRUFBRSxPQUFPLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNGLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFTCxvQ0FBb0M7SUFDcEMsT0FBTyxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDO1FBQ3hCLE1BQU0sRUFBRSxXQUFXO1FBQ25CLEdBQUcsRUFBRSxTQUFTLENBQUMsV0FBVztRQUMxQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN0QyxXQUFXLEVBQUUsa0JBQWtCO1FBQy9CLFlBQVksRUFBRSxxQkFBcUI7UUFDbkMsUUFBUSxFQUFFO1lBQ1Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLFlBQVk7WUFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7WUFDMUMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxZQUFZO1lBQ3JDLGVBQWUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFO1NBQzlDO0tBQ0YsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0FBRWYsQ0FBQztBQTVGRCwwQkE0RkM7QUFFRDs7OztHQUlHO0FBQ0gsS0FBSyxTQUFTLENBQUMsQ0FBQyxlQUFlLENBQUMsTUFBYzs7SUFDNUMsTUFBTSxPQUFPLEdBQTRCLEVBQUUsTUFBTSxFQUFFLE1BQU0sRUFBRSxNQUFNLEVBQUUsU0FBUyxDQUFDLGtCQUFrQixFQUFFLENBQUM7SUFDbEcsR0FBRztRQUNELE1BQU0sTUFBTSxHQUFHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUMvRCxLQUFLLE1BQU0sTUFBTSxVQUFJLE1BQU0sQ0FBQyxRQUFRLG1DQUFJLEVBQUUsRUFBRTtZQUMxQyxJQUFJLFFBQUMsTUFBTSxDQUFDLEdBQUcsMENBQUUsUUFBUSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsRUFBQyxFQUFFO2dCQUN2RCxTQUFTO2FBQ1Y7WUFDRCw2RUFBNkU7WUFDN0UsTUFBTSxNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsTUFBTSxDQUFDLEdBQUcsQ0FBQyxNQUFNLEdBQUcsU0FBUyxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxHQUFHLFNBQVMsQ0FBQywwQkFBMEIsRUFBRSxDQUFDO1lBQzVJLE1BQU0sTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxHQUFHLENBQUMsTUFBTSxHQUFHLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxTQUFTLENBQUMsc0JBQXNCLEVBQUUsQ0FBQztZQUN4SSxJQUFJLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUMsRUFBRTtnQkFDOUYsU0FBUzthQUNWO1lBQ0QsTUFBTSxNQUFNLENBQUM7U0FDZDtRQUNELE9BQU8sQ0FBQyxpQkFBaUIsR0FBRyxNQUFNLENBQUMscUJBQXFCLENBQUM7S0FDMUQsUUFBUSxPQUFPLENBQUMsaUJBQWlCLElBQUksSUFBSSxFQUFFO0FBQzlDLENBQUM7QUFFRCxLQUFLLFVBQVUsYUFBYSxDQUFDLFFBQWEsRUFBRSxNQUFjLEVBQUUsVUFBa0IsRUFBRSxRQUF3Qjs7SUFDdEcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUN6QyxNQUFNLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxDQUFDLEdBQUcsU0FBUyxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUUsQ0FBQztJQUNyRixNQUFNLE9BQU8sR0FBRyxJQUFJLGVBQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN2QyxNQUFNLEtBQUssU0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQywwQ0FBRSxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzVELElBQUksS0FBSyxJQUFJLElBQUksSUFBSSxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDeEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxZQUFZLFdBQVcsSUFBSSxPQUFPLDhDQUE4QyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUM3RyxPQUFPO0tBQ1I7SUFFRCxPQUFPLENBQUMsR0FBRyxDQUFDLGVBQWUsV0FBVyxJQUFJLE9BQU8sQ0FBQyxPQUFPLDJCQUEyQixDQUFDLENBQUM7SUFDdEYsTUFBTSxPQUFPLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzlELElBQUksT0FBTyxFQUFFO1FBQ1gsT0FBTyxDQUFDLEdBQUcsQ0FBQyxZQUFZLFdBQVcsSUFBSSxPQUFPLENBQUMsT0FBTyxpREFBaUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDbEksT0FBTztLQUNSO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxlQUFlLFdBQVcsSUFBSSxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBRXJELG1FQUFtRTtJQUNuRSxNQUFNLEdBQUcsR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUMsRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQ3BGLE1BQU0sV0FBVyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLGtCQUFrQixFQUFFLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ2hHLE1BQU0sZ0JBQWdCLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUN0RyxNQUFNLFFBQVEsR0FBRyxNQUFNLElBQUksT0FBTyxDQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFO1FBQ3BELGFBQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFLLENBQUMsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsRUFBRTtZQUMxQyxJQUFJLEdBQUcsRUFBRTtnQkFDUCxPQUFPLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNoQjtZQUNELG9CQUFPLEVBQUU7aUJBQ04sRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUU7Z0JBQ3BDLElBQUksTUFBTSxDQUFDLElBQUksS0FBSyxzQkFBc0IsRUFBRTtvQkFDMUMsb0VBQW9FO29CQUNwRSxPQUFPLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztpQkFDM0I7Z0JBQ0QsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztnQkFDbkMsT0FBTyxNQUFNO3FCQUNWLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO3FCQUN0RCxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRTtvQkFDaEIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztvQkFDMUIsSUFBSSxFQUFFLENBQUM7Z0JBQ1QsQ0FBQyxDQUFDO3FCQUNELE1BQU0sRUFBRSxDQUFDO1lBQ2QsQ0FBQyxDQUFDO2lCQUNELElBQUksQ0FBQyxRQUFRLEVBQUUsR0FBRyxFQUFFO2dCQUNuQixFQUFFLENBQUMsSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQyxDQUFDO1lBQ25FLENBQUMsQ0FBQztpQkFDRCxLQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLEVBQUU7Z0JBQ3ZCLElBQUksUUFBUSxFQUFFO29CQUNaLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztpQkFDZDtZQUNILENBQUMsQ0FBQyxDQUFDO1FBQ1AsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUNELDJDQUEyQztJQUM3QyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUMzRCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxhQUFDLGdCQUFnQixhQUFoQixnQkFBZ0IsdUJBQWhCLGdCQUFnQixDQUFFLElBQUksMENBQUUsUUFBUSxDQUFDLE9BQU8sb0NBQUssSUFBSSxDQUFDLENBQUM7SUFDbEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxlQUFNLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQztJQUNwRCxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEVBQUU7UUFDbkMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLElBQUksR0FBRyxFQUFFLENBQUMsQ0FBQztLQUMzQztJQUNELFFBQVEsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBRSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUU7UUFDekMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxNQUFNO1FBQzFCLFdBQVcsRUFBRSxXQUFXLENBQUMsV0FBVztRQUNwQyxRQUFRLEVBQUUsV0FBVyxDQUFDLFFBQVE7UUFDOUIsU0FBUyxFQUFFLFdBQVcsQ0FBQyxJQUFJLENBQUMsT0FBTztRQUNuQyxPQUFPLEVBQUUsV0FBVyxDQUFDLE9BQU87UUFDNUIsS0FBSztRQUNMLFFBQVEsRUFBRSxXQUFXO1FBQ3JCLElBQUksRUFBRSxXQUFXLENBQUMsSUFBSTtRQUN0QixPQUFPLEVBQUUsV0FBVyxDQUFDLE9BQU87S0FDN0IsQ0FBQyxDQUFDO0FBRUwsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGd1bnppcCB9IGZyb20gJ3psaWInO1xuXG5pbXBvcnQgdHlwZSB7IEFzc2VtYmx5VGFyZ2V0cyB9IGZyb20gJ0Bqc2lpL3NwZWMnO1xuaW1wb3J0IHsgbWV0cmljU2NvcGUsIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQgfSBmcm9tICdhd3MtbGFtYmRhJztcbmltcG9ydCB7IEFXU0Vycm9yLCBTMyB9IGZyb20gJ2F3cy1zZGsnO1xuaW1wb3J0IHsgU2VtVmVyIH0gZnJvbSAnc2VtdmVyJztcbmltcG9ydCB7IGV4dHJhY3QgfSBmcm9tICd0YXItc3RyZWFtJztcbmltcG9ydCB7IERlbnlMaXN0Q2xpZW50IH0gZnJvbSAnLi4vZGVueS1saXN0L2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB0eXBlIHsgQ2F0YWxvZ0J1aWxkZXJJbnB1dCB9IGZyb20gJy4uL3BheWxvYWQtc2NoZW1hJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0ICogYXMgY29uc3RhbnRzIGZyb20gJy4uL3NoYXJlZC9jb25zdGFudHMnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5cbmNvbnN0IE1FVFJJQ1NfTkFNRVNQQUNFID0gJ0NvbnN0cnVjdEh1Yi9DYXRhbG9nQnVpbGRlcic7XG5cbi8qKlxuICogUmVnZW5lcmF0ZXMgdGhlIGBjYXRhbG9nLmpzb25gIG9iamVjdCBpbiB0aGUgY29uZmlndXJlZCBTMyBidWNrZXQuXG4gKlxuICogQHBhcmFtIGV2ZW50IGNvbmZpZ3VyYXRpb24gZm9yIHRoZSByZWJ1aWxkIGpvYi4gSW4gcGFydGljdWxhciwgdGhlIGByZWJ1aWxkYFxuICogICAgICAgICAgICAgIHByb3BlcnR5IGNhbiBiZSBzZXQgdG8gYHRydWVgIGluIG9yZGVyIHRvIHRyaWdnZXIgYSBmdWxsIChpLmU6XG4gKiAgICAgICAgICAgICAgbm9uLWluY3JlbWVudGFsKSByZWJ1aWxkIG9mIHRoZSBvYmplY3QuXG4gKiBAcGFyYW0gY29udGV4dCB0aGUgbGFtYmRhIGNvbnRleHQgaW4gd2hpY2ggdGhpcyBleGVjdXRpb24gcnVucy5cbiAqXG4gKiBAcmV0dXJucyB0aGUgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHVwZGF0ZWQgUzMgb2JqZWN0LlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gaGFuZGxlcihldmVudDogQ2F0YWxvZ0J1aWxkZXJJbnB1dCwgY29udGV4dDogQ29udGV4dCkge1xuICBjb25zb2xlLmxvZyhKU09OLnN0cmluZ2lmeShldmVudCwgbnVsbCwgMikpO1xuXG4gIC8vIGRldGVybWluZSBpZiB0aGlzIGlzIGEgcmVxdWVzdCB0byByZWJ1aWxkIHRoZSBjYXRhbG9nIChiYXNpY2FsbHksIGFuIGVtcHR5IGV2ZW50KVxuICBjb25zdCByZWJ1aWxkID0gIWV2ZW50Py5wYWNrYWdlO1xuICBpZiAocmVidWlsZCkge1xuICAgIGNvbnNvbGUubG9nKCdSZXF1ZXN0aW5nIGNhdGFsb2cgcmVidWlsZCAoZW1wdHkgZXZlbnQpJyk7XG4gIH1cblxuICBjb25zdCBCVUNLRVRfTkFNRSA9IHJlcXVpcmVFbnYoJ0JVQ0tFVF9OQU1FJyk7XG5cbiAgY29uc3QgcGFja2FnZXMgPSBuZXcgTWFwPHN0cmluZywgTWFwPG51bWJlciwgUGFja2FnZUluZm8+PigpO1xuICBjb25zdCBkZW55TGlzdCA9IGF3YWl0IERlbnlMaXN0Q2xpZW50Lm5ld0NsaWVudCgpO1xuXG4gIGxldCBkYXRhOiB1bmRlZmluZWQgfCBBV1MuUzMuR2V0T2JqZWN0T3V0cHV0O1xuXG4gIGlmICghcmVidWlsZCkge1xuICAgIGNvbnNvbGUubG9nKCdMb2FkaW5nIGV4aXN0aW5nIGNhdGFsb2cuLi4nKTtcblxuICAgIGRhdGEgPSBhd2FpdCBhd3MuczMoKS5nZXRPYmplY3QoeyBCdWNrZXQ6IEJVQ0tFVF9OQU1FLCBLZXk6IGNvbnN0YW50cy5DQVRBTE9HX0tFWSB9KS5wcm9taXNlKClcbiAgICAgIC5jYXRjaCgoZXJyOiBBV1NFcnJvcikgPT4gZXJyLmNvZGUgIT09ICdOb1N1Y2hLZXknXG4gICAgICAgID8gUHJvbWlzZS5yZWplY3QoZXJyKVxuICAgICAgICA6IFByb21pc2UucmVzb2x2ZSh7IC8qIG5vIGRhdGEgKi8gfSBhcyBTMy5HZXRPYmplY3RPdXRwdXQpKTtcbiAgfVxuXG4gIC8vIGlmIGV2ZW50IGlzIGVtcHR5LCB3ZSdyZSBkb2luZyBhIGZ1bGwgcmVidWlsZFxuICBpZiAoIWRhdGE/LkJvZHkgfHwgcmVidWlsZCkge1xuICAgIGNvbnNvbGUubG9nKCdDYXRhbG9nIG5vdCBmb3VuZC4gUmVjcmVhdGluZy4uLicpO1xuICAgIGNvbnN0IGZhaWx1cmVzOiBhbnkgPSB7fTtcbiAgICBmb3IgYXdhaXQgKGNvbnN0IHsgS2V5OiBwa2dLZXkgfSBvZiByZWxldmFudE9iamVjdHMoQlVDS0VUX05BTUUpKSB7XG4gICAgICB0cnkge1xuICAgICAgICBhd2FpdCBhcHBlbmRQYWNrYWdlKHBhY2thZ2VzLCBwa2dLZXkhLCBCVUNLRVRfTkFNRSwgZGVueUxpc3QpO1xuICAgICAgfSBjYXRjaCAoZSkge1xuICAgICAgICBmYWlsdXJlc1twa2dLZXkhXSA9IGU7XG4gICAgICB9XG4gICAgfVxuICAgIGZvciAoY29uc3QgW2tleSwgZXJyb3JdIG9mIE9iamVjdC5lbnRyaWVzKGZhaWx1cmVzKSkge1xuICAgICAgY29uc29sZS5sb2coYEZhaWxlZCBwcm9jZXNzaW5nICR7a2V5fTogJHtlcnJvcn1gKTtcbiAgICB9XG5cbiAgICBhd2FpdCBtZXRyaWNTY29wZSgobWV0cmljcykgPT4gYXN5bmMgKCkgPT4ge1xuICAgICAgbWV0cmljcy5zZXROYW1lc3BhY2UoTUVUUklDU19OQU1FU1BBQ0UpO1xuICAgICAgY29uc3QgZmFpbGVkQ291bnQgPSBPYmplY3Qua2V5cyhmYWlsdXJlcykubGVuZ3RoO1xuICAgICAgY29uc29sZS5sb2coYE1hcmtpbmcgJHtmYWlsZWRDb3VudH0gZmFpbGVkIHBhY2thZ2VzYCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYygnRmFpbGVkUGFja2FnZXNPblJlY3JlYXRpb24nLCBmYWlsZWRDb3VudCwgVW5pdC5Db3VudCk7XG4gICAgfSkoKTtcblxuICB9IGVsc2Uge1xuICAgIGNvbnNvbGUubG9nKCdDYXRhbG9nIGZvdW5kLiBMb2FkaW5nLi4uJyk7XG4gICAgY29uc3QgY2F0YWxvZyA9IEpTT04ucGFyc2UoZGF0YS5Cb2R5LnRvU3RyaW5nKCd1dGYtOCcpKTtcbiAgICBmb3IgKGNvbnN0IGluZm8gb2YgY2F0YWxvZy5wYWNrYWdlcykge1xuICAgICAgaWYgKCFwYWNrYWdlcy5oYXMoaW5mby5uYW1lKSkge1xuICAgICAgICBwYWNrYWdlcy5zZXQoaW5mby5uYW1lLCBuZXcgTWFwKCkpO1xuICAgICAgfVxuICAgICAgcGFja2FnZXMuZ2V0KGluZm8ubmFtZSkhLnNldChpbmZvLm1ham9yLCBpbmZvKTtcbiAgICB9XG4gICAgY29uc29sZS5sb2coJ1JlZ2lzdGVyaW5nIG5ldyBwYWNrYWdlcy4uLicpO1xuICAgIC8vIG5vdGUgdGhhdCB3ZSBpbnRlbnRpb25hbGx5IGRvbid0IGNhdGNoIGVycm9ycyBoZXJlIHRvIGxldCB0aGVzZVxuICAgIC8vIGV2ZW50IGdvIHRvIHRoZSBETFEgZm9yIG1hbnVhbCBpbnNwZWN0aW9uLlxuICAgIGF3YWl0IGFwcGVuZFBhY2thZ2UocGFja2FnZXMsIGV2ZW50LnBhY2thZ2Uua2V5LCBCVUNLRVRfTkFNRSwgZGVueUxpc3QpO1xuICB9XG5cbiAgLy8gQnVpbGQgdGhlIGZpbmFsIGRhdGEgcGFja2FnZS4uLlxuICBjb25zb2xlLmxvZygnQ29uc29saWRhdGluZyBjYXRhbG9nLi4uJyk7XG4gIGNvbnN0IGNhdGFsb2cgPSB7IHBhY2thZ2VzOiBuZXcgQXJyYXk8UGFja2FnZUluZm8+KCksIHVwZGF0ZWQ6IG5ldyBEYXRlKCkudG9JU09TdHJpbmcoKSB9O1xuICBmb3IgKGNvbnN0IG1ham9ycyBvZiBwYWNrYWdlcy52YWx1ZXMoKSkge1xuICAgIGZvciAoY29uc3QgcGtnIG9mIG1ham9ycy52YWx1ZXMoKSkge1xuICAgICAgY2F0YWxvZy5wYWNrYWdlcy5wdXNoKHBrZyk7XG4gICAgfVxuICB9XG5cbiAgY29uc29sZS5sb2coYFRoZXJlIGFyZSBub3cgJHtjYXRhbG9nLnBhY2thZ2VzLmxlbmd0aH0gcmVnaXN0ZXJlZCBwYWNrYWdlIG1ham9yIHZlcnNpb25zYCk7XG4gIGF3YWl0IG1ldHJpY1Njb3BlKChtZXRyaWNzKSA9PiBhc3luYyAoKSA9PiB7XG4gICAgbWV0cmljcy5zZXROYW1lc3BhY2UoTUVUUklDU19OQU1FU1BBQ0UpO1xuICAgIG1ldHJpY3MucHV0TWV0cmljKCdSZWdpc3RlcmVkUGFja2FnZXNNYWpvclZlcnNpb24nLCBjYXRhbG9nLnBhY2thZ2VzLmxlbmd0aCwgVW5pdC5Db3VudCk7XG4gIH0pKCk7XG5cbiAgLy8gVXBsb2FkIHRoZSByZXN1bHQgdG8gUzMgYW5kIGV4aXQuXG4gIHJldHVybiBhd3MuczMoKS5wdXRPYmplY3Qoe1xuICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgS2V5OiBjb25zdGFudHMuQ0FUQUxPR19LRVksXG4gICAgQm9keTogSlNPTi5zdHJpbmdpZnkoY2F0YWxvZywgbnVsbCwgMiksXG4gICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMsIG1heC1hZ2U9MzAwJywgLy8gRXhwaXJlIGZyb20gY2FjaGUgYWZ0ZXIgNSBtaW51dGVzXG4gICAgTWV0YWRhdGE6IHtcbiAgICAgICdMYW1iZGEtTG9nLUdyb3VwJzogY29udGV4dC5sb2dHcm91cE5hbWUsXG4gICAgICAnTGFtYmRhLUxvZy1TdHJlYW0nOiBjb250ZXh0LmxvZ1N0cmVhbU5hbWUsXG4gICAgICAnTGFtYmRhLVJ1bi1JZCc6IGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgJ1BhY2thZ2UtQ291bnQnOiBgJHtjYXRhbG9nLnBhY2thZ2VzLmxlbmd0aH1gLFxuICAgIH0sXG4gIH0pLnByb21pc2UoKTtcblxufVxuXG4vKipcbiAqIEEgZ2VuZXJhdG9yIHRoYXQgYXN5bmNocm9ub3VzbHkgdHJhdmVyc2VzIHRoZSBzZXQgb2YgXCJpbnRlcmVzdGluZ1wiIG9iamVjdHNcbiAqIGZvdW5kIGJ5IGxpc3RpbmcgdGhlIGNvbmZpZ3VyZWQgUzMgYnVja2V0LiBUaG9zZSBvYmplY3RzIGNvcnJlc3BvbmQgdG8gYWxsXG4gKiBucG0gcGFja2FnZSB0YXJiYWxscyBwcmVzZW50IHVuZGVyIHRoZSBgcGFja2FnZXMvYCBwcmVmaXggaW4gdGhlIGJ1Y2tldC5cbiAqL1xuYXN5bmMgZnVuY3Rpb24qIHJlbGV2YW50T2JqZWN0cyhidWNrZXQ6IHN0cmluZykge1xuICBjb25zdCByZXF1ZXN0OiBTMy5MaXN0T2JqZWN0c1YyUmVxdWVzdCA9IHsgQnVja2V0OiBidWNrZXQsIFByZWZpeDogY29uc3RhbnRzLlNUT1JBR0VfS0VZX1BSRUZJWCB9O1xuICBkbyB7XG4gICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgYXdzLnMzKCkubGlzdE9iamVjdHNWMihyZXF1ZXN0KS5wcm9taXNlKCk7XG4gICAgZm9yIChjb25zdCBvYmplY3Qgb2YgcmVzdWx0LkNvbnRlbnRzID8/IFtdKSB7XG4gICAgICBpZiAoIW9iamVjdC5LZXk/LmVuZHNXaXRoKGNvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVgpKSB7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgLy8gV2Ugb25seSByZWdpc3RlciBwYWNrYWdlcyBpZiB0aGV5IGhhdmUgQVQgTEVBU1QgUHl0aG9uIG9yIFR5cGVTY3JpcHQgZG9jcy5cbiAgICAgIGNvbnN0IHRzRG9jcyA9IGAke29iamVjdC5LZXkuc3Vic3RyaW5nKDAsIG9iamVjdC5LZXkubGVuZ3RoIC0gY29uc3RhbnRzLlBBQ0tBR0VfS0VZX1NVRkZJWC5sZW5ndGgpfSR7Y29uc3RhbnRzLkRPQ1NfS0VZX1NVRkZJWF9UWVBFU0NSSVBUfWA7XG4gICAgICBjb25zdCBweURvY3MgPSBgJHtvYmplY3QuS2V5LnN1YnN0cmluZygwLCBvYmplY3QuS2V5Lmxlbmd0aCAtIGNvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVgubGVuZ3RoKX0ke2NvbnN0YW50cy5ET0NTX0tFWV9TVUZGSVhfUFlUSE9OfWA7XG4gICAgICBpZiAoIShhd2FpdCBhd3MuczNPYmplY3RFeGlzdHMoYnVja2V0LCB0c0RvY3MpKSAmJiAhKGF3YWl0IGF3cy5zM09iamVjdEV4aXN0cyhidWNrZXQsIHB5RG9jcykpKSB7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgeWllbGQgb2JqZWN0O1xuICAgIH1cbiAgICByZXF1ZXN0LkNvbnRpbnVhdGlvblRva2VuID0gcmVzdWx0Lk5leHRDb250aW51YXRpb25Ub2tlbjtcbiAgfSB3aGlsZSAocmVxdWVzdC5Db250aW51YXRpb25Ub2tlbiAhPSBudWxsKTtcbn1cblxuYXN5bmMgZnVuY3Rpb24gYXBwZW5kUGFja2FnZShwYWNrYWdlczogYW55LCBwa2dLZXk6IHN0cmluZywgYnVja2V0TmFtZTogc3RyaW5nLCBkZW55TGlzdDogRGVueUxpc3RDbGllbnQpIHtcbiAgY29uc29sZS5sb2coYFByb2Nlc3Npbmcga2V5OiAke3BrZ0tleX1gKTtcbiAgY29uc3QgWywgcGFja2FnZU5hbWUsIHZlcnNpb25TdHJdID0gY29uc3RhbnRzLlNUT1JBR0VfS0VZX0ZPUk1BVF9SRUdFWC5leGVjKHBrZ0tleSkhO1xuICBjb25zdCB2ZXJzaW9uID0gbmV3IFNlbVZlcih2ZXJzaW9uU3RyKTtcbiAgY29uc3QgZm91bmQgPSBwYWNrYWdlcy5nZXQocGFja2FnZU5hbWUpPy5nZXQodmVyc2lvbi5tYWpvcik7XG4gIGlmIChmb3VuZCAhPSBudWxsICYmIHZlcnNpb24uY29tcGFyZShmb3VuZC52ZXJzaW9uKSA8PSAwKSB7XG4gICAgY29uc29sZS5sb2coYFNraXBwaW5nICR7cGFja2FnZU5hbWV9QCR7dmVyc2lvbn0gYmVjYXVzZSBpdCBpcyBub3QgbmV3ZXIgdGhhbiB0aGUgZXhpc3RpbmcgJHtmb3VuZC52ZXJzaW9ufWApO1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGNvbnNvbGUubG9nKGBDaGVja2luZyBpZiAke3BhY2thZ2VOYW1lfUAke3ZlcnNpb24udmVyc2lvbn0gbWF0Y2hlcyBhIGRlbnkgbGlzdCBydWxlYCk7XG4gIGNvbnN0IGJsb2NrZWQgPSBkZW55TGlzdC5sb29rdXAocGFja2FnZU5hbWUsIHZlcnNpb24udmVyc2lvbik7XG4gIGlmIChibG9ja2VkKSB7XG4gICAgY29uc29sZS5sb2coYFNraXBwaW5nICR7cGFja2FnZU5hbWV9QCR7dmVyc2lvbi52ZXJzaW9ufSBiZWNhdXNlIGl0IGlzIGJsb2NrZWQgYnkgdGhlIGRlbnkgbGlzdCBydWxlOiAke0pTT04uc3RyaW5naWZ5KGJsb2NrZWQpfWApO1xuICAgIHJldHVybjtcbiAgfVxuXG4gIGNvbnNvbGUubG9nKGBSZWdpc3RlcmluZyAke3BhY2thZ2VOYW1lfUAke3ZlcnNpb259YCk7XG5cbiAgLy8gRG9ud2xvYWQgdGhlIHRhcmJhbGwgdG8gaW5zcGVjdCB0aGUgYHBhY2thZ2UuanNvbmAgZGF0YSB0aGVyZWluLlxuICBjb25zdCBwa2cgPSBhd2FpdCBhd3MuczMoKS5nZXRPYmplY3QoeyBCdWNrZXQ6IGJ1Y2tldE5hbWUsIEtleTogcGtnS2V5IH0pLnByb21pc2UoKTtcbiAgY29uc3QgbWV0YWRhdGFLZXkgPSBwa2dLZXkucmVwbGFjZShjb25zdGFudHMuUEFDS0FHRV9LRVlfU1VGRklYLCBjb25zdGFudHMuTUVUQURBVEFfS0VZX1NVRkZJWCk7XG4gIGNvbnN0IG1ldGFkYXRhUmVzcG9uc2UgPSBhd2FpdCBhd3MuczMoKS5nZXRPYmplY3QoeyBCdWNrZXQ6IGJ1Y2tldE5hbWUsIEtleTogbWV0YWRhdGFLZXkgfSkucHJvbWlzZSgpO1xuICBjb25zdCBtYW5pZmVzdCA9IGF3YWl0IG5ldyBQcm9taXNlPEJ1ZmZlcj4oKG9rLCBrbykgPT4ge1xuICAgIGd1bnppcChCdWZmZXIuZnJvbShwa2cuQm9keSEpLCAoZXJyLCB0YXIpID0+IHtcbiAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgcmV0dXJuIGtvKGVycik7XG4gICAgICB9XG4gICAgICBleHRyYWN0KClcbiAgICAgICAgLm9uKCdlbnRyeScsIChoZWFkZXIsIHN0cmVhbSwgbmV4dCkgPT4ge1xuICAgICAgICAgIGlmIChoZWFkZXIubmFtZSAhPT0gJ3BhY2thZ2UvcGFja2FnZS5qc29uJykge1xuICAgICAgICAgICAgLy8gTm90IHRoZSBmaWxlIHdlIGFyZSBsb29raW5nIGZvciwgc2tpcCBhaGVhZCAobmV4dCBydW4tbG9vcCB0aWNrKS5cbiAgICAgICAgICAgIHJldHVybiBzZXRJbW1lZGlhdGUobmV4dCk7XG4gICAgICAgICAgfVxuICAgICAgICAgIGNvbnN0IGNodW5rcyA9IG5ldyBBcnJheTxCdWZmZXI+KCk7XG4gICAgICAgICAgcmV0dXJuIHN0cmVhbVxuICAgICAgICAgICAgLm9uKCdkYXRhJywgKGNodW5rKSA9PiBjaHVua3MucHVzaChCdWZmZXIuZnJvbShjaHVuaykpKVxuICAgICAgICAgICAgLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICAgICAgb2soQnVmZmVyLmNvbmNhdChjaHVua3MpKTtcbiAgICAgICAgICAgICAgbmV4dCgpO1xuICAgICAgICAgICAgfSlcbiAgICAgICAgICAgIC5yZXN1bWUoKTtcbiAgICAgICAgfSlcbiAgICAgICAgLm9uY2UoJ2ZpbmlzaCcsICgpID0+IHtcbiAgICAgICAgICBrbyhuZXcgRXJyb3IoJ0NvdWxkIG5vdCBmaW5kIHBhY2thZ2UvcGFja2FnZS5qc29uIGluIHRhcmJhbGwhJykpO1xuICAgICAgICB9KVxuICAgICAgICAud3JpdGUodGFyLCAod3JpdGVFcnIpID0+IHtcbiAgICAgICAgICBpZiAod3JpdGVFcnIpIHtcbiAgICAgICAgICAgIGtvKHdyaXRlRXJyKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcbiAgICAvLyBBZGQgdGhlIFBhY2thZ2VJbmZvIGludG8gdGhlIHdvcmtpbmcgc2V0XG4gIGNvbnN0IHBrZ01ldGFkYXRhID0gSlNPTi5wYXJzZShtYW5pZmVzdC50b1N0cmluZygndXRmLTgnKSk7XG4gIGNvbnN0IG5wbU1ldGFkYXRhID0gSlNPTi5wYXJzZShtZXRhZGF0YVJlc3BvbnNlPy5Cb2R5Py50b1N0cmluZygndXRmLTgnKSA/PyAne30nKTtcbiAgY29uc3QgbWFqb3IgPSBuZXcgU2VtVmVyKHBrZ01ldGFkYXRhLnZlcnNpb24pLm1ham9yO1xuICBpZiAoIXBhY2thZ2VzLmhhcyhwa2dNZXRhZGF0YS5uYW1lKSkge1xuICAgIHBhY2thZ2VzLnNldChwa2dNZXRhZGF0YS5uYW1lLCBuZXcgTWFwKCkpO1xuICB9XG4gIHBhY2thZ2VzLmdldChwa2dNZXRhZGF0YS5uYW1lKSEuc2V0KG1ham9yLCB7XG4gICAgYXV0aG9yOiBwa2dNZXRhZGF0YS5hdXRob3IsXG4gICAgZGVzY3JpcHRpb246IHBrZ01ldGFkYXRhLmRlc2NyaXB0aW9uLFxuICAgIGtleXdvcmRzOiBwa2dNZXRhZGF0YS5rZXl3b3JkcyxcbiAgICBsYW5ndWFnZXM6IHBrZ01ldGFkYXRhLmpzaWkudGFyZ2V0cyxcbiAgICBsaWNlbnNlOiBwa2dNZXRhZGF0YS5saWNlbnNlLFxuICAgIG1ham9yLFxuICAgIG1ldGFkYXRhOiBucG1NZXRhZGF0YSxcbiAgICBuYW1lOiBwa2dNZXRhZGF0YS5uYW1lLFxuICAgIHZlcnNpb246IHBrZ01ldGFkYXRhLnZlcnNpb24sXG4gIH0pO1xuXG59XG5cbmludGVyZmFjZSBQYWNrYWdlSW5mbyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgYXNzZW1ibHkuXG4gICAqL1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBtYWpvciB2ZXJzaW9uIG9mIHRoaXMgYXNzZW1ibHksIGFjY29yZGluZyB0byBTZW1WZXIuXG4gICAqL1xuICByZWFkb25seSBtYWpvcjogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgY29tcGxldGUgU2VtVmVyIHZlcnNpb24gc3RyaW5nIGZvciB0aGlzIHBhY2thZ2UncyBtYWpvciB2ZXJzaW9uIHN0cmVhbSxcbiAgICogaW5jbHVkaW5nIHByZS1yZWxlYXNlIGlkZW50aWZpZXJzLCBidXQgZXhjbHVkaW5nIGFkZGl0aW9uYWwgbWV0YWRhdGFcbiAgICogKGV2ZXJ5dGhpbmcgc3RhcnRpbmcgYXQgYCtgLCBpZiB0aGVyZSBpcyBhbnkpLlxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgU1BEWCBsaWNlbnNlIGlkZW50aWZpZXIgZm9yIHRoZSBwYWNrYWdlJ3MgbGljZW5zZS5cbiAgICovXG4gIHJlYWRvbmx5IGxpY2Vuc2U6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2Yga2V5d29yZHMgY29uZmlndXJlZCBvbiB0aGUgcGFja2FnZS5cbiAgICovXG4gIHJlYWRvbmx5IGtleXdvcmRzOiByZWFkb25seSBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTWV0YWRhdGEgYXNzaWduZWQgYnkgdGhlIGRpc2NvdmVyeSBmdW5jdGlvbiB0byB0aGUgbGF0ZXN0IHJlbGVhc2Ugb2YgdGhpc1xuICAgKiBwYWNrYWdlJ3MgbWFqb3IgdmVyc2lvbiBzdHJlYW0sIGlmIGFueS5cbiAgICovXG4gIHJlYWRvbmx5IG1ldGFkYXRhPzogeyByZWFkb25seSBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogVGhlIGF1dGhvciBvZiB0aGUgcGFja2FnZS5cbiAgICovXG4gIHJlYWRvbmx5IGF1dGhvcjoge1xuICAgIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgICByZWFkb25seSBlbWFpbD86IHN0cmluZztcbiAgICByZWFkb25seSB1cmw/OiBzdHJpbmc7XG4gIH07XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIGxhbmd1YWdlcyBjb25maWd1cmVkIG9uIHRoZSBwYWNrYWdlLCBhbmQgdGhlIGNvcnJlc3BvbmRpbmdcbiAgICogY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGxhbmd1YWdlczogQXNzZW1ibHlUYXJnZXRzO1xuXG4gIC8qKlxuICAgKiBUaGUgdGltZXN0YW1wIGF0IHdoaWNoIHRoaXMgdmVyc2lvbiB3YXMgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHRpbWU6IERhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBkZXNjcmlwdGlvbiBvZiB0aGUgcGFja2FnZS5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xufVxuIl19