"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            // alarms that come from watchful are all considered normal severity
            alarmActionArns: ((_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.normalSeverity) ? [this.alarmActions.normalSeverity] : [],
            alarmActions: ((_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.normalSeverityAction) ? [this.alarmActions.normalSeverityAction] : [],
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'HighSeverityDashboard');
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        var _a, _b;
        const highSeverityActionArn = (_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.highSeverity;
        if (highSeverityActionArn) {
            alarm.addAlarmAction({
                bind: () => ({ alarmActionArn: highSeverityActionArn }),
            });
        }
        const highSeverityAction = (_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.highSeverityAction;
        if (highSeverityAction) {
            alarm.addAlarmAction(highSeverityAction);
        }
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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