"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_lambda_shared_1.MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(constants_lambda_shared_1.MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    ko(new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`));
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibnBtLWpzLWZvbGxvd2VyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wYWNrYWdlLXNvdXJjZXMvbnBtanMvbnBtLWpzLWZvbGxvd2VyLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxtQ0FBbUM7QUFDbkMsK0JBQStCO0FBQy9CLDZCQUEwQjtBQUUxQiwrREFBdUY7QUFDdkYsb0ZBQTZFO0FBRTdFLGlFQUFpRTtBQUNqRSw2QkFBOEI7QUFDOUIsdUZBQThFO0FBQzlFLDBGQUFvRjtBQUNwRiw4REFBOEQ7QUFDOUQsOEVBQW9FO0FBRXBFLDBGQUF5RTtBQUN6RSx1RUFBeUc7QUFDekcsaUVBQWlFO0FBQ2pFLE1BQU0sb0JBQW9CLEdBQUcsT0FBTyxDQUFDLDZCQUE2QixDQUFDLENBQUM7QUFFcEUsTUFBTSxvQkFBb0IsR0FBRyxLQUFNLENBQUM7QUFDcEMsTUFBTSxrQkFBa0IsR0FBd0IsSUFBSSxHQUFHLENBQUMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0FBQzlGLE1BQU0sd0JBQXdCLEdBQUcsOEJBQThCLENBQUM7QUFFaEUsb0NBQW9DO0FBQ3BDLG9DQUFhLENBQUMsbUJBQW1CLEdBQUcsc0JBQVksQ0FBQyxNQUFNLENBQUM7QUFDeEQsb0NBQWEsQ0FBQyxTQUFTLEdBQUcsMkNBQWlCLENBQUM7QUFFNUM7Ozs7Ozs7OztHQVNHO0FBQ0ksS0FBSyxVQUFVLE9BQU8sQ0FBQyxLQUFxQixFQUFFLE9BQWdCO0lBQ25FLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRXhELE1BQU0sYUFBYSxHQUFHLDhCQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEQsTUFBTSxRQUFRLEdBQUcsOEJBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUV6QyxNQUFNLFFBQVEsR0FBRyxNQUFNLHFDQUFjLENBQUMsU0FBUyxFQUFFLENBQUM7SUFDbEQsTUFBTSxXQUFXLEdBQUcsTUFBTSx3Q0FBaUIsQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUV4RCxNQUFNLGFBQWEsR0FBRyxNQUFNLHlCQUF5QixDQUFDLE9BQVMsQ0FBQyxtREFBbUQsQ0FBQyxDQUFDO0lBRXJILE1BQU0sTUFBTSxHQUE4QjtRQUN4QyxXQUFXLEVBQUUsSUFBSTtRQUNqQiw4Q0FBOEM7UUFDOUMsSUFBSSxFQUFFLElBQUk7UUFDVixLQUFLLEVBQUUsYUFBYSxDQUFDLE9BQU8sRUFBRTtRQUM5QixxR0FBcUc7UUFDckcsK0ZBQStGO1FBQy9GLE9BQU8sRUFBRSxvQkFBb0I7UUFDN0IseUJBQXlCO1FBQ3pCLFFBQVEsRUFBRTtZQUNSLElBQUksRUFBRSxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUU7U0FDcEI7UUFDRCxTQUFTLEVBQUUsRUFBRTtLQUNkLENBQUM7SUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUM1QyxNQUFNLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUVuQywyRUFBMkU7SUFDM0UseUJBQXlCO0lBQ3pCLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7UUFDNUIsSUFBSSxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBRWxDLEVBQUUsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQzthQUN6QixFQUFFLENBQUMsT0FBTyxFQUFFLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxLQUF3QixFQUFFLEVBQUU7O1lBQ3ZFLCtIQUErSDtZQUMvSCxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUM7WUFFeEIsT0FBTyxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUUsYUFBYSxDQUFDLENBQUM7WUFDL0MsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBRTdCLDhCQUE4QjtZQUM5QixLQUFLLE1BQU0sRUFBRSxHQUFHLEVBQUUsSUFBSSxLQUFLLEVBQUU7Z0JBQzNCLFVBQUksR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksMENBQUUsUUFBUSxFQUFFO29CQUN2QixPQUFPLENBQUMsU0FBUywwQ0FFZixTQUFTLEdBQUcsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFDakQsMkJBQUksQ0FBQyxZQUFZLENBQ2xCLENBQUM7aUJBQ0g7YUFDRjtZQUVELElBQUk7Z0JBQ0YsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsS0FBSyxDQUFDLE1BQU0sYUFBYSxDQUFDLENBQUM7Z0JBQzlELE9BQU8sQ0FBQyxTQUFTLG1DQUEwQixLQUFLLENBQUMsTUFBTSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBQ3JFLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDL0QsT0FBTyxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsYUFBYSxDQUFDLENBQUM7Z0JBRTdDLHdFQUF3RTtnQkFDeEUseUVBQXlFO2dCQUN6RSxNQUFNLFlBQVksR0FBRyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQztnQkFDcEYsT0FBTyxDQUFDLEdBQUcsQ0FBQyxjQUFjLFlBQVksQ0FBQyxNQUFNLHFDQUFxQyxDQUFDLENBQUM7Z0JBQ3BGLE9BQU8sQ0FBQyxTQUFTLDREQUF1QyxZQUFZLENBQUMsTUFBTSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7Z0JBRXpGLGdDQUFnQztnQkFDaEMsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxFQUFFO29CQUNqRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUM7b0JBQzFCLE1BQU0scUJBQXFCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO29CQUM1QyxPQUFPLENBQUMsU0FBUyxtQ0FBMEIsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLE1BQU0sRUFBRSwyQkFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUNyRixDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUVKLHVDQUF1QztnQkFDdkMsTUFBTSx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDekMsYUFBYSxHQUFHLE9BQU8sQ0FBQztnQkFFeEIsc0dBQXNHO2dCQUN0RywyR0FBMkc7Z0JBQzNHLDJIQUEySDtnQkFDM0gsSUFBSSxPQUFPLENBQUMsd0JBQXdCLEVBQUUsSUFBSSxLQUFNLENBQUMsZ0JBQWdCLEVBQUU7b0JBQ2pFLE9BQU8sQ0FBQyxHQUFHLENBQUMsbURBQW1ELENBQUMsQ0FBQztvQkFDakUsMkZBQTJGO29CQUMxRixFQUFFLENBQUMsYUFBcUIsQ0FBQyxNQUFNLEVBQUUsQ0FBQztpQkFDcEM7cUJBQU07b0JBQ0wsT0FBTyxDQUFDLEdBQUcsQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO29CQUM1RCxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxDQUFDO29CQUN4QixPQUFPLENBQUMsU0FBUyx1Q0FBNEIsT0FBTyxDQUFDLHdCQUF3QixFQUFFLEVBQUUsMkJBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztvQkFDcEcsRUFBRSxDQUFDLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxDQUFDLENBQUM7aUJBQ3RDO2FBQ0Y7WUFBQyxPQUFPLEdBQUcsRUFBRTtnQkFDWiwwRUFBMEU7Z0JBQzFFLE9BQU8sQ0FBQyxLQUFLLENBQUMscUJBQXFCLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQzFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7Z0JBQ3hCLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNUO29CQUFTO2dCQUNSLE9BQU8sQ0FBQyxTQUFTLG9EQUFtQyxJQUFJLENBQUMsR0FBRyxFQUFFLEdBQUcsU0FBUyxFQUFFLDJCQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7YUFDaEc7UUFDSCxDQUFDLENBQUMsQ0FBQzthQUNGLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFO1lBQ2hCLE9BQU8sQ0FBQyxHQUFHLENBQUMsc0NBQXNDLENBQUMsQ0FBQztZQUNwRCxFQUFFLENBQUMsRUFBRSxhQUFhLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQztRQUN2QyxDQUFDLENBQUMsQ0FBQztJQUNQLENBQUMsQ0FBQyxDQUFDO0lBRUgsaUNBQWlDO0lBQ2pDOzs7Ozs7T0FNRztJQUNILEtBQUssVUFBVSx5QkFBeUIsQ0FBQyxZQUFvQjtRQUMzRCxJQUFJO1lBQ0YsTUFBTSxRQUFRLEdBQUcsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDO2dCQUN4QyxNQUFNLEVBQUUsYUFBYTtnQkFDckIsR0FBRyxFQUFFLDBDQUFnQjthQUN0QixDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDYixNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFLLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3JFLE9BQU8sQ0FBQyxHQUFHLENBQUMsaUNBQWlDLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDdkQsT0FBTyxNQUFNLENBQUM7U0FDZjtRQUFDLE9BQU8sS0FBSyxFQUFFO1lBQ2QsSUFBSSxLQUFLLENBQUMsSUFBSSxLQUFLLFdBQVcsRUFBRTtnQkFDOUIsTUFBTSxLQUFLLENBQUM7YUFDYjtZQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsdUJBQXVCLGFBQWEsSUFBSSwwQ0FBZ0IsZ0RBQWdELFlBQVksR0FBRyxDQUFDLENBQUM7WUFDckksT0FBTyxZQUFZLENBQUM7U0FDckI7SUFDSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNILEtBQUssVUFBVSx5QkFBeUIsQ0FBQyxRQUFnQjtRQUN2RCxPQUFPLENBQUMsR0FBRyxDQUFDLHVDQUF1QyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQy9ELE9BQU8sU0FBUyxDQUFDLDBDQUFnQixFQUFFLFFBQVEsQ0FBQyxPQUFPLEVBQUUsRUFBRSxFQUFFLFdBQVcsRUFBRSwyQkFBMkIsRUFBRSxDQUFDLENBQUM7SUFDdkcsQ0FBQztJQUNELFlBQVk7SUFFWix3QkFBd0I7SUFDeEIsS0FBSyxVQUFVLHFCQUFxQixDQUFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxHQUFHLEVBQWtCLEVBQUUsT0FBc0I7UUFDbkcsSUFBSTtZQUNGLHVCQUF1QjtZQUN2QixNQUFNLE9BQU8sR0FBRyxNQUFNLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBRWxELDRDQUE0QztZQUM1Qyw0RkFBNEY7WUFDNUYsNEZBQTRGO1lBQzVGLE1BQU0sVUFBVSxHQUFHLEdBQUcsaUNBQTZCLEdBQUcsSUFBSSxTQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ3JILE1BQU0sU0FBUyxDQUFDLFVBQVUsRUFBRSxPQUFPLEVBQUU7Z0JBQ25DLFdBQVcsRUFBRSwwQkFBMEI7Z0JBQ3ZDLFFBQVEsRUFBRTtvQkFDUixhQUFhLEVBQUUsUUFBUSxDQUFDLFdBQVcsRUFBRTtvQkFDckMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNO29CQUNyQyxZQUFZLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPO29CQUNoQyxVQUFVLEVBQUUsR0FBRyxDQUFDLE9BQU8sRUFBRTtpQkFDMUI7YUFDRixDQUFDLENBQUM7WUFFSCxvQ0FBb0M7WUFDcEMsTUFBTSxXQUFXLEdBQUc7Z0JBQ2xCLFVBQVUsRUFBRSxRQUFRLGFBQWEsSUFBSSxVQUFVLEVBQUU7Z0JBQ2pELFFBQVEsRUFBRTtvQkFDUixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPO29CQUN4QixHQUFHLEVBQUUsR0FBRyxDQUFDLE9BQU8sRUFBRTtpQkFDbkI7Z0JBQ0QsSUFBSSxFQUFFLFFBQVEsQ0FBQyxXQUFXLEVBQUU7YUFDN0IsQ0FBQztZQUNGLE1BQU0sT0FBTyxHQUFtQjtnQkFDOUIsR0FBRyxXQUFXO2dCQUNkLFNBQVMsRUFBRSxtQ0FBUyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUM7YUFDM0MsQ0FBQztZQUVGLDJCQUEyQjtZQUMzQixNQUFNLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUM7Z0JBQzFCLFdBQVcsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2dCQUM3QyxRQUFRLEVBQUUsUUFBUTthQUNuQixDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7WUFFYixPQUFPLENBQUMsU0FBUyxvREFBbUMsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDcEU7UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLDBFQUEwRTtZQUMxRSxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRywrREFBK0QsS0FBSyxDQUFDLElBQUksSUFBSSxLQUFLLENBQUMsT0FBTyxLQUFLLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDM0gsT0FBTyxDQUFDLFNBQVMsb0RBQW1DLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ25FLE1BQU0sU0FBUyxDQUFDLEdBQUcsaUNBQTZCLEdBQUcsR0FBRyxFQUFFLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEdBQUcsS0FBSyxFQUFFLDZCQUE2QixFQUFFLEdBQUcsRUFBRSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRTtnQkFDbkksV0FBVyxFQUFFLGtCQUFrQjtnQkFDL0IsUUFBUSxFQUFFO29CQUNSLGFBQWEsRUFBRSxRQUFRLENBQUMsV0FBVyxFQUFFO2lCQUN0QzthQUNGLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUNELFlBQVk7SUFFWixpQ0FBaUM7SUFDakM7Ozs7OztPQU1HO0lBQ0gsU0FBUyxPQUFPLENBQUMsR0FBVztRQUMxQixPQUFPLElBQUksT0FBTyxDQUFTLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFO1lBQ3BDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLEVBQUU7Z0JBQzFCLElBQUksUUFBUSxDQUFDLFVBQVUsS0FBSyxHQUFHLEVBQUU7b0JBQy9CLEVBQUUsQ0FBQyxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsUUFBUSxDQUFDLFVBQVUsTUFBTSxRQUFRLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDO2lCQUN2RjtnQkFFRCxJQUFJLElBQUksR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUMzQixRQUFRLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsSUFBSSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDakYsUUFBUSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzdCLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSCxTQUFTLFNBQVMsQ0FBQyxHQUFXLEVBQUUsSUFBaUIsRUFBRSxPQUFpRSxFQUFFO1FBQ3BILE9BQU8sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQztZQUN4QixNQUFNLEVBQUUsYUFBYTtZQUNyQixHQUFHLEVBQUUsR0FBRztZQUNSLElBQUksRUFBRSxJQUFJO1lBQ1YsUUFBUSxFQUFFO2dCQUNSLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxZQUFZO2dCQUN4QyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsYUFBYTtnQkFDMUMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxZQUFZO2dCQUNyQyxHQUFHLElBQUksQ0FBQyxRQUFRO2FBQ2pCO1lBQ0QsR0FBRyxJQUFJO1NBQ1IsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQ2YsQ0FBQztJQUNELFlBQVk7QUFDZCxDQUFDO0FBblBELDBCQW1QQztBQUVEOzs7Ozs7Ozs7O0dBVUc7QUFDSCxTQUFTLHVCQUF1QixDQUM5QixPQUEwQixFQUMxQixPQUFzQixFQUN0QixRQUF3QixFQUN4QixXQUE4Qjs7SUFHOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxLQUFLLEVBQWtCLENBQUM7SUFFM0MsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7UUFDNUIsdUVBQXVFO1FBQ3ZFLGtFQUFrRTtRQUNsRSxJQUFJLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxLQUFLLFNBQVMsRUFBRTtZQUNqQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsMENBQTBDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ25GLE9BQU8sQ0FBQyxTQUFTLG1EQUFrQyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNsRSxTQUFTO1NBQ1Y7UUFFRCwyR0FBMkc7UUFDM0csSUFBSSxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ2xELE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyxpRUFBaUUsTUFBTSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7WUFDMUcsT0FBTyxDQUFDLFNBQVMsbURBQWtDLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2xFLFNBQVM7U0FDVjtRQUVELG1FQUFtRTtRQUNuRSxJQUFJLE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxJQUFJLElBQUksRUFBRTtZQUMvQixPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsOENBQThDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3ZGLE9BQU8sQ0FBQyxTQUFTLG1EQUFrQyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNsRSxTQUFTO1NBQ1Y7UUFFRCxzRUFBc0U7UUFDdEUsSUFBSSxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7WUFDM0IsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLDBDQUEwQyxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUNuRixPQUFPLENBQUMsU0FBUyxtREFBa0MsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDbEUsU0FBUztTQUNWO1FBRUQsaURBQWlEO1FBQ2pELE1BQU0sYUFBYSxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7WUFDbkQsZ0RBQWdEO2FBQy9DLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsS0FBSyxTQUFTLElBQUksR0FBRyxLQUFLLFVBQVUsQ0FBQztZQUMzRCxvREFBb0Q7YUFDbkQsR0FBRyxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsT0FBTyxFQUFFLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFVLENBQUM7WUFDbkUsMkJBQTJCO2FBQzFCLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUNyRCxPQUFPLENBQUMsU0FBUyxvREFBbUMsYUFBYSxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXRGLElBQUksY0FBZ0MsQ0FBQztRQUNyQyxLQUFLLE1BQU0sQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLElBQUksYUFBYSxFQUFFO1lBQy9DLElBQUksY0FBYyxJQUFJLElBQUksSUFBSSxjQUFjLENBQUMsT0FBTyxFQUFFLEtBQUssUUFBUSxDQUFDLE9BQU8sRUFBRSxFQUFFO2dCQUM3RSxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDM0MsSUFBSSxLQUFLLElBQUksSUFBSSxFQUFFO29CQUNqQixxREFBcUQ7b0JBQ3JELE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyw4QkFBOEIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksT0FBTyx5QkFBeUIsQ0FBQyxDQUFDO2lCQUM5RztxQkFBTSxJQUFJLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxFQUFFO29CQUVwQyxpQ0FBaUM7b0JBQ2pDLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7b0JBQzFELElBQUksTUFBTSxFQUFFO3dCQUNWLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyxxQkFBcUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7d0JBQ3pFLE9BQU8sQ0FBQyxTQUFTLDRDQUErQixDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQzt3QkFDL0QsU0FBUztxQkFDVjtvQkFFRCxPQUFPLENBQUMsU0FBUyxnREFBaUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxHQUFHLFFBQVEsQ0FBQyxPQUFPLEVBQUUsRUFBRSwyQkFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO29CQUN0RyxNQUFNLFVBQVUsR0FBRyxXQUFXLENBQUMsTUFBTSxPQUFDLEtBQUssQ0FBQyxPQUFPLG1DQUFJLFlBQVksQ0FBQyxJQUFJLElBQUksQ0FBQztvQkFDN0UsT0FBTyxDQUFDLFNBQVMsK0NBQWdDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztvQkFDakYsSUFBSSxVQUFVLEVBQUU7d0JBQ2QsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsR0FBRyxFQUFFLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO3FCQUNuRDt5QkFBTTt3QkFDTCxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsY0FBYyxNQUFNLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSSxPQUFPLHdDQUF3QyxNQUFBLEtBQUssQ0FBQyxPQUFPLG1DQUFJLFlBQVksRUFBRSxDQUFDLENBQUM7cUJBQzVJO2lCQUNGO3FCQUFNO29CQUNMLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyxlQUFlLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLE9BQU8scUNBQXFDLENBQUMsQ0FBQztpQkFDM0c7Z0JBQ0QsY0FBYyxHQUFHLFFBQVEsQ0FBQzthQUMzQjtTQUNGO0tBQ0Y7SUFDRCxPQUFPLE1BQU0sQ0FBQztJQUVkLFNBQVMsa0JBQWtCLENBQUMsS0FBa0I7O1FBQzVDLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7WUFDdEIsT0FBTyxLQUFLLENBQUM7U0FDZDtRQUNELE9BQU8sS0FBSyxDQUFDLElBQUksS0FBSyxXQUFXO2VBQzVCLEtBQUssQ0FBQyxJQUFJLEtBQUssYUFBYTtlQUM1QixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUM7c0JBQ2pDLEtBQUssQ0FBQyxRQUFRLDBDQUFFLElBQUksQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFDLENBQUM7SUFDaEUsQ0FBQztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb25zb2xlIGZyb20gJ2NvbnNvbGUnO1xuaW1wb3J0ICogYXMgaHR0cHMgZnJvbSAnaHR0cHMnO1xuaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcblxuaW1wb3J0IHsgbWV0cmljU2NvcGUsIENvbmZpZ3VyYXRpb24sIE1ldHJpY3NMb2dnZXIsIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgRW52aXJvbm1lbnRzIGZyb20gJ2F3cy1lbWJlZGRlZC1tZXRyaWNzL2xpYi9lbnZpcm9ubWVudC9FbnZpcm9ubWVudHMnO1xuaW1wb3J0IHR5cGUgeyBDb250ZXh0LCBTY2hlZHVsZWRFdmVudCB9IGZyb20gJ2F3cy1sYW1iZGEnO1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIEB0eXBlc2NyaXB0LWVzbGludC9uby1yZXF1aXJlLWltcG9ydHNcbmltcG9ydCBOYW5vID0gcmVxdWlyZSgnbmFubycpO1xuaW1wb3J0IHsgRGVueUxpc3RDbGllbnQgfSBmcm9tICcuLi8uLi9iYWNrZW5kL2RlbnktbGlzdC9jbGllbnQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBMaWNlbnNlTGlzdENsaWVudCB9IGZyb20gJy4uLy4uL2JhY2tlbmQvbGljZW5zZS1saXN0L2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9hd3MubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyByZXF1aXJlRW52IH0gZnJvbSAnLi4vLi4vYmFja2VuZC9zaGFyZWQvZW52LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgSW5nZXN0aW9uSW5wdXQgfSBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9pbmdlc3Rpb24taW5wdXQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBpbnRlZ3JpdHkgfSBmcm9tICcuLi8uLi9iYWNrZW5kL3NoYXJlZC9pbnRlZ3JpdHkubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBNZXRyaWNOYW1lLCBNQVJLRVJfRklMRV9OQU1FLCBNRVRSSUNTX05BTUVTUEFDRSwgUzNLZXlQcmVmaXggfSBmcm9tICcuL2NvbnN0YW50cy5sYW1iZGEtc2hhcmVkJztcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzXG5jb25zdCBub3JtYWxpemVOUE1NZXRhZGF0YSA9IHJlcXVpcmUoJ25vcm1hbGl6ZS1yZWdpc3RyeS1tZXRhZGF0YScpO1xuXG5jb25zdCBUSU1FT1VUX01JTExJU0VDT05EUyA9IDEwXzAwMDtcbmNvbnN0IENPTlNUUlVDVF9LRVlXT1JEUzogUmVhZG9ubHlTZXQ8c3RyaW5nPiA9IG5ldyBTZXQoWydjZGsnLCAnYXdzLWNkaycsICdjZGs4cycsICdjZGt0ZiddKTtcbmNvbnN0IE5QTV9SRVBMSUNBX1JFR0lTVFJZX1VSTCA9ICdodHRwczovL3JlcGxpY2F0ZS5ucG1qcy5jb20vJztcblxuLy8gQ29uZmlndXJlIGVtYmVkZGVkIG1ldHJpY3MgZm9ybWF0XG5Db25maWd1cmF0aW9uLmVudmlyb25tZW50T3ZlcnJpZGUgPSBFbnZpcm9ubWVudHMuTGFtYmRhO1xuQ29uZmlndXJhdGlvbi5uYW1lc3BhY2UgPSBNRVRSSUNTX05BTUVTUEFDRTtcblxuLyoqXG4gKiBUaGlzIGZ1bmN0aW9uIHRyaWdnZXJzIG9uIGEgZml4ZWQgc2NoZWR1bGUgYW5kIHJlYWRzIGEgc3RyZWFtIG9mIGNoYW5nZXMgZnJvbSBucG1qcyBjb3VjaGRiIF9jaGFuZ2VzIGVuZHBvaW50LlxuICogVXBvbiBpbnZvY2F0aW9uIHRoZSBmdW5jdGlvbiBzdGFydHMgcmVhZGluZyBmcm9tIGEgc2VxdWVuY2Ugc3RvcmVkIGluIGFuIHMzIG9iamVjdCAtIHRoZSBgbWFya2VyYC5cbiAqIElmIHRoZSBtYXJrZXIgZmFpbHMgdG8gbG9hZCAob3IgZG8gbm90IGV4aXN0KSwgdGhlIHN0cmVhbSB3aWxsIHN0YXJ0IGZyb20gYG5vd2AgLSB0aGUgbGF0ZXN0IGNoYW5nZS5cbiAqIEZvciBlYWNoIGNoYW5nZTpcbiAqICAtIHRoZSBwYWNrYWdlIHZlcnNpb24gdGFyYmFsbCB3aWxsIGJlIGNvcGllZCBmcm9tIHRoZSBucG0gcmVnaXN0cnkgdG8gYSBzdGF0aW5nIGJ1Y2tldC5cbiAqICAtIGEgbWVzc2FnZSB3aWxsIGJlIHNlbnQgdG8gYW4gc3FzIHF1ZXVlXG4gKiBucG0gcmVnaXN0cnkgQVBJIGRvY3M6IGh0dHBzOi8vZ2l0aHViLmNvbS9ucG0vcmVnaXN0cnkvYmxvYi9tYXN0ZXIvZG9jcy9SRUdJU1RSWS1BUEkubWRcbiAqIEBwYXJhbSBjb250ZXh0IGEgTGFtYmRhIGV4ZWN1dGlvbiBjb250ZXh0XG4gKi9cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBoYW5kbGVyKGV2ZW50OiBTY2hlZHVsZWRFdmVudCwgY29udGV4dDogQ29udGV4dCkge1xuICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gIGNvbnN0IHN0YWdpbmdCdWNrZXQgPSByZXF1aXJlRW52KCdCVUNLRVRfTkFNRScpO1xuICBjb25zdCBxdWV1ZVVybCA9IHJlcXVpcmVFbnYoJ1FVRVVFX1VSTCcpO1xuXG4gIGNvbnN0IGRlbnlMaXN0ID0gYXdhaXQgRGVueUxpc3RDbGllbnQubmV3Q2xpZW50KCk7XG4gIGNvbnN0IGxpY2Vuc2VMaXN0ID0gYXdhaXQgTGljZW5zZUxpc3RDbGllbnQubmV3Q2xpZW50KCk7XG5cbiAgY29uc3QgaW5pdGlhbE1hcmtlciA9IGF3YWl0IGxvYWRMYXN0VHJhbnNhY3Rpb25NYXJrZXIoMV84MDBfMDAwIC8qIEBhd3MtY2RrL2NkayBpbml0aWFsIHJlbGVhc2Ugd2FzIGF0IDFfODQ2XzcwOSAqLyk7XG5cbiAgY29uc3QgY29uZmlnOiBOYW5vLkNoYW5nZXNSZWFkZXJPcHRpb25zID0ge1xuICAgIGluY2x1ZGVEb2NzOiB0cnVlLFxuICAgIC8vIHBhdXNlIHRoZSBjaGFuZ2VzIHJlYWRlciBhZnRlciBlYWNoIHJlcXVlc3RcbiAgICB3YWl0OiB0cnVlLFxuICAgIHNpbmNlOiBpbml0aWFsTWFya2VyLnRvRml4ZWQoKSxcbiAgICAvLyBgY2hhbmdlc1JlYWRlci5nZXRgIHN0b3BzIG9uY2UgYSByZXNwb25zZSB3aXRoIHplcm8gY2hhbmdlcyBpcyByZWNlaXZlZCwgaG93ZXZlciBpdCB3YWl0cyB0b28gbG9uZ1xuICAgIC8vICBzaW5jZSB3ZSB3YW50IHRvIHRlcm1pbmF0ZSB0aGUgTGFtYmRhIGZ1bmN0aW9uIHdlIGRlZmluZSBhIHRpbWVvdXQgc2hvcnRlciB0aGFuIHRoZSBkZWZhdWx0XG4gICAgdGltZW91dDogVElNRU9VVF9NSUxMSVNFQ09ORFMsXG4gICAgLy8gT25seSBpdGVtcyB3aXRoIGEgbmFtZVxuICAgIHNlbGVjdG9yOiB7XG4gICAgICBuYW1lOiB7ICRndDogbnVsbCB9LFxuICAgIH0sXG4gICAgYmF0Y2hTaXplOiAzMCxcbiAgfTtcblxuICBjb25zdCBuYW5vID0gTmFubyhOUE1fUkVQTElDQV9SRUdJU1RSWV9VUkwpO1xuICBjb25zdCBkYiA9IG5hbm8uZGIudXNlKCdyZWdpc3RyeScpO1xuXG4gIC8vIFdlIG5lZWQgdG8gbWFrZSBhbiBleHBsaWNpdCBQcm9taXNlIGhlcmUsIGJlY2F1c2Ugb3RoZXJ3aXNlIExhbWJkYSB3b24ndFxuICAvLyBrbm93IHdoZW4gaXQncyBkb25lLi4uXG4gIHJldHVybiBuZXcgUHJvbWlzZSgob2ssIGtvKSA9PiB7XG4gICAgbGV0IHVwZGF0ZWRNYXJrZXIgPSBpbml0aWFsTWFya2VyO1xuXG4gICAgZGIuY2hhbmdlc1JlYWRlci5nZXQoY29uZmlnKVxuICAgICAgLm9uKCdiYXRjaCcsIG1ldHJpY1Njb3BlKChtZXRyaWNzKSA9PiBhc3luYyAoYmF0Y2g6IHJlYWRvbmx5IENoYW5nZVtdKSA9PiB7XG4gICAgICAgIC8vIENsZWFyIGF1dG9tYXRpY2FsbHkgc2V0IGRpbWVuc2lvbnMgLSB3ZSBkb24ndCBuZWVkIHRoZW0gKHNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzbGFicy9hd3MtZW1iZWRkZWQtbWV0cmljcy1ub2RlL2lzc3Vlcy83MylcbiAgICAgICAgbWV0cmljcy5zZXREaW1lbnNpb25zKCk7XG5cbiAgICAgICAgbWV0cmljcy5zZXRQcm9wZXJ0eSgnU3RhcnRTZXEnLCB1cGRhdGVkTWFya2VyKTtcbiAgICAgICAgY29uc3Qgc3RhcnRUaW1lID0gRGF0ZS5ub3coKTtcblxuICAgICAgICAvLyBFbWl0IG5wbS5qcyByZXBsaWNhdGlvbiBsYWdcbiAgICAgICAgZm9yIChjb25zdCB7IGRvYyB9IG9mIGJhdGNoKSB7XG4gICAgICAgICAgaWYgKGRvYz8udGltZT8ubW9kaWZpZWQpIHtcbiAgICAgICAgICAgIG1ldHJpY3MucHV0TWV0cmljKFxuICAgICAgICAgICAgICBNZXRyaWNOYW1lLk5QTUpTX0NIQU5HRV9BR0UsXG4gICAgICAgICAgICAgIHN0YXJ0VGltZSAtIG5ldyBEYXRlKGRvYy50aW1lLm1vZGlmaWVkKS5nZXRUaW1lKCksXG4gICAgICAgICAgICAgIFVuaXQuTWlsbGlzZWNvbmRzLFxuICAgICAgICAgICAgKTtcbiAgICAgICAgICB9XG4gICAgICAgIH1cblxuICAgICAgICB0cnkge1xuICAgICAgICAgIGNvbnNvbGUubG9nKGBSZWNlaXZlZCBhIGJhdGNoIG9mICR7YmF0Y2gubGVuZ3RofSBlbGVtZW50KHMpYCk7XG4gICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5DSEFOR0VfQ09VTlQsIGJhdGNoLmxlbmd0aCwgVW5pdC5Db3VudCk7XG4gICAgICAgICAgY29uc3QgbGFzdFNlcSA9IE1hdGgubWF4KC4uLmJhdGNoLm1hcCgoY2hhbmdlKSA9PiBjaGFuZ2Uuc2VxKSk7XG4gICAgICAgICAgbWV0cmljcy5zZXRQcm9wZXJ0eSgnRW5kU2VxJywgdXBkYXRlZE1hcmtlcik7XG5cbiAgICAgICAgICAvLyBPYnRhaW4gdGhlIG1vZGlmaWVkIHBhY2thZ2UgdmVyc2lvbiBmcm9tIHRoZSB1cGRhdGUgZXZlbnQsIGFuZCBmaWx0ZXJcbiAgICAgICAgICAvLyBvdXQgcGFja2FnZXMgdGhhdCBhcmUgbm90IG9mIGludGVyZXN0IHRvIHVzIChub3QgY29uc3RydWN0IGxpYnJhcmllcykuXG4gICAgICAgICAgY29uc3QgdmVyc2lvbkluZm9zID0gZ2V0UmVsZXZhbnRWZXJzaW9uSW5mb3MoYmF0Y2gsIG1ldHJpY3MsIGRlbnlMaXN0LCBsaWNlbnNlTGlzdCk7XG4gICAgICAgICAgY29uc29sZS5sb2coYElkZW50aWZpZWQgJHt2ZXJzaW9uSW5mb3MubGVuZ3RofSByZWxldmFudCBwYWNrYWdlIHZlcnNpb24gdXBkYXRlKHMpYCk7XG4gICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5SRUxFVkFOVF9QQUNLQUdFX1ZFUlNJT05TLCB2ZXJzaW9uSW5mb3MubGVuZ3RoLCBVbml0LkNvdW50KTtcblxuICAgICAgICAgIC8vIFByb2Nlc3MgYWxsIHJlbWFpbmluZyB1cGRhdGVzXG4gICAgICAgICAgYXdhaXQgUHJvbWlzZS5hbGwodmVyc2lvbkluZm9zLm1hcChhc3luYyAoaW5mb3MpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGJlZm9yZSA9IERhdGUubm93KCk7XG4gICAgICAgICAgICBhd2FpdCBwcm9jZXNzVXBkYXRlZFZlcnNpb24oaW5mb3MsIG1ldHJpY3MpO1xuICAgICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5TVEFHSU5HX1RJTUUsIERhdGUubm93KCkgLSBiZWZvcmUsIFVuaXQuTWlsbGlzZWNvbmRzKTtcbiAgICAgICAgICB9KSk7XG5cbiAgICAgICAgICAvLyBVcGRhdGUgdGhlIHRyYW5zYWN0aW9uIG1hcmtlciBpbiBTMy5cbiAgICAgICAgICBhd2FpdCBzYXZlTGFzdFRyYW5zYWN0aW9uTWFya2VyKGxhc3RTZXEpO1xuICAgICAgICAgIHVwZGF0ZWRNYXJrZXIgPSBsYXN0U2VxO1xuXG4gICAgICAgICAgLy8gSWYgd2UgaGF2ZSBlbm91Z2ggdGltZSBsZWZ0IGJlZm9yZSB0aW1lb3V0LCBwcm9jZWVkIHdpdGggdGhlIG5leHQgYmF0Y2gsIG90aGVyd2lzZSB3ZSdyZSBkb25lIGhlcmUuXG4gICAgICAgICAgLy8gU2luY2UgdGhlIGRpc3RyaWJ1dGlvbiBvZiB0aGUgdGltZSBpdCB0YWtlcyB0byBwcm9jZXNzIGVhY2ggcGFja2FnZS9iYXRjaCBpcyBub24gdW5pZm9ybSwgdGhpcyBpcyBhIGJlc3RcbiAgICAgICAgICAvLyBlZmZvcnQsIGFuZCB3ZSBleHBlY3QgdGhlIGZ1bmN0aW9uIHRvIHRpbWVvdXQgaW4gc29tZSBpbnZvY2F0aW9ucywgd2UgcmVseSBvbiB0aGUgZG93bnN0cmVhbSBpZGVtcG90ZW5jeSB0byBoYW5kbGUgdGhpcy5cbiAgICAgICAgICBpZiAoY29udGV4dC5nZXRSZW1haW5pbmdUaW1lSW5NaWxsaXMoKSA+PSAzMF8wMDAgLyogMzAgc2Vjb25kcyAqLykge1xuICAgICAgICAgICAgY29uc29sZS5sb2coJ1RoZXJlIGlzIHN0aWxsIHRpbWUsIHJlcXVlc3RpbmcgdGhlIG5leHQgYmF0Y2guLi4nKTtcbiAgICAgICAgICAgIC8vIE5vdGU6IHRoZSBgcmVzdW1lYCBmdW5jdGlvbiBpcyBtaXNzaW5nIGZyb20gdGhlIGBuYW5vYCB0eXBlIGRlZmluaXRpb25zLCBidXQgaXMgdGhlcmUuLi5cbiAgICAgICAgICAgIChkYi5jaGFuZ2VzUmVhZGVyIGFzIGFueSkucmVzdW1lKCk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKCdXZSBhcmUgYWxtb3N0IG91dCBvZiB0aW1lLCBzbyBzdG9wcGluZyBoZXJlLicpO1xuICAgICAgICAgICAgZGIuY2hhbmdlc1JlYWRlci5zdG9wKCk7XG4gICAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlJFTUFJTklOR19USU1FLCBjb250ZXh0LmdldFJlbWFpbmluZ1RpbWVJbk1pbGxpcygpLCBVbml0Lk1pbGxpc2Vjb25kcyk7XG4gICAgICAgICAgICBvayh7IGluaXRpYWxNYXJrZXIsIHVwZGF0ZWRNYXJrZXIgfSk7XG4gICAgICAgICAgfVxuICAgICAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgICAgICAvLyBBbiBleGNlcHRpb24gYnViYmxlZCBvdXQsIHdoaWNoIG1lYW5zIHRoaXMgTGFtYmRhIGV4ZWN1dGlvbiBoYXMgZmFpbGVkLlxuICAgICAgICAgIGNvbnNvbGUuZXJyb3IoYFVuZXhwZWN0ZWQgZXJyb3I6ICR7ZXJyfWApO1xuICAgICAgICAgIGRiLmNoYW5nZXNSZWFkZXIuc3RvcCgpO1xuICAgICAgICAgIGtvKGVycik7XG4gICAgICAgIH0gZmluYWxseSB7XG4gICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5CQVRDSF9QUk9DRVNTSU5HX1RJTUUsIERhdGUubm93KCkgLSBzdGFydFRpbWUsIFVuaXQuTWlsbGlzZWNvbmRzKTtcbiAgICAgICAgfVxuICAgICAgfSkpXG4gICAgICAub25jZSgnZW5kJywgKCkgPT4ge1xuICAgICAgICBjb25zb2xlLmxvZygnTm8gbW9yZSB1cGRhdGVzIHRvIHByb2Nlc3MsIGV4aXRpbmcuJyk7XG4gICAgICAgIG9rKHsgaW5pdGlhbE1hcmtlciwgdXBkYXRlZE1hcmtlciB9KTtcbiAgICAgIH0pO1xuICB9KTtcblxuICAvLyNyZWdpb24gTGFzdCB0cmFuc2FjdGlvbiBtYXJrZXJcbiAgLyoqXG4gICAqIExvYWRzIHRoZSBsYXN0IHRyYW5zYWN0aW9uIG1hcmtlciBmcm9tIFMzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVmYXVsdFZhbHVlIHRoZSB2YWx1ZSB0byByZXR1cm4gaW4gY2FzZSB0aGUgbWFya2VyIGRvZXMgbm90IGV4aXN0XG4gICAqXG4gICAqIEByZXR1cm5zIHRoZSB2YWx1ZSBvZiB0aGUgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXIuXG4gICAqL1xuICBhc3luYyBmdW5jdGlvbiBsb2FkTGFzdFRyYW5zYWN0aW9uTWFya2VyKGRlZmF1bHRWYWx1ZTogbnVtYmVyKTogUHJvbWlzZTxudW1iZXI+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBhd3MuczMoKS5nZXRPYmplY3Qoe1xuICAgICAgICBCdWNrZXQ6IHN0YWdpbmdCdWNrZXQsXG4gICAgICAgIEtleTogTUFSS0VSX0ZJTEVfTkFNRSxcbiAgICAgIH0pLnByb21pc2UoKTtcbiAgICAgIGNvbnN0IG1hcmtlciA9IE51bWJlci5wYXJzZUludChyZXNwb25zZS5Cb2R5IS50b1N0cmluZygndXRmLTgnKSwgMTApO1xuICAgICAgY29uc29sZS5sb2coYFJlYWQgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXI6ICR7bWFya2VyfWApO1xuICAgICAgcmV0dXJuIG1hcmtlcjtcbiAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgaWYgKGVycm9yLmNvZGUgIT09ICdOb1N1Y2hLZXknKSB7XG4gICAgICAgIHRocm93IGVycm9yO1xuICAgICAgfVxuICAgICAgY29uc29sZS5sb2coYE1hcmtlciBvYmplY3QgKHMzOi8vJHtzdGFnaW5nQnVja2V0fS8ke01BUktFUl9GSUxFX05BTUV9KSBkb2VzIG5vdCBleGlzdCwgc3RhcnRpbmcgZnJvbSB0aGUgZGVmYXVsdCAoJHtkZWZhdWx0VmFsdWV9KWApO1xuICAgICAgcmV0dXJuIGRlZmF1bHRWYWx1ZTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXIgaW4gUzMuXG4gICAqXG4gICAqIEBwYXJhbSBzZXF1ZW5jZSB0aGUgbGFzdCB0cmFuc2FjdGlvbiBtYXJrZXIgdmFsdWVcbiAgICovXG4gIGFzeW5jIGZ1bmN0aW9uIHNhdmVMYXN0VHJhbnNhY3Rpb25NYXJrZXIoc2VxdWVuY2U6IE51bWJlcikge1xuICAgIGNvbnNvbGUubG9nKGBVcGRhdGluZyBsYXN0IHRyYW5zYWN0aW9uIG1hcmtlciB0byAke3NlcXVlbmNlfWApO1xuICAgIHJldHVybiBwdXRPYmplY3QoTUFSS0VSX0ZJTEVfTkFNRSwgc2VxdWVuY2UudG9GaXhlZCgpLCB7IENvbnRlbnRUeXBlOiAndGV4dC9wbGFpbjsgY2hhcnNldD1VVEYtOCcgfSk7XG4gIH1cbiAgLy8jZW5kcmVnaW9uXG5cbiAgLy8jcmVnaW9uIEJ1c2luZXNzIExvZ2ljXG4gIGFzeW5jIGZ1bmN0aW9uIHByb2Nlc3NVcGRhdGVkVmVyc2lvbih7IGluZm9zLCBtb2RpZmllZCwgc2VxIH06IFVwZGF0ZWRWZXJzaW9uLCBtZXRyaWNzOiBNZXRyaWNzTG9nZ2VyKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgdHJ5IHtcbiAgICAgIC8vIERvd25sb2FkIHRoZSB0YXJiYWxsXG4gICAgICBjb25zdCB0YXJiYWxsID0gYXdhaXQgaHR0cEdldChpbmZvcy5kaXN0LnRhcmJhbGwpO1xuXG4gICAgICAvLyBTdG9yZSB0aGUgdGFyYmFsbCBpbnRvIHRoZSBzdGFnaW5nIGJ1Y2tldFxuICAgICAgLy8gLSBpbmZvcy5kaXN0LnRhcmJhbGwgPT4gaHR0cHM6Ly9yZWdpc3RyeS5ucG1qcy5vcmcvPEBzY29wZT4vPG5hbWU+Ly0vPG5hbWU+LTx2ZXJzaW9uPi50Z3pcbiAgICAgIC8vIC0gc3RhZ2luZ0tleSAgICAgICAgID0+ICAgICAgICAgICAgICAgICAgICAgc3RhZ2VkLzxAc2NvcGU+LzxuYW1lPi8tLzxuYW1lPi08dmVyc2lvbj4udGd6XG4gICAgICBjb25zdCBzdGFnaW5nS2V5ID0gYCR7UzNLZXlQcmVmaXguU1RBR0VEX0tFWV9QUkVGSVh9JHtuZXcgVVJMKGluZm9zLmRpc3QudGFyYmFsbCkucGF0aG5hbWV9YC5yZXBsYWNlKC9cXC97Mix9L2csICcvJyk7XG4gICAgICBhd2FpdCBwdXRPYmplY3Qoc3RhZ2luZ0tleSwgdGFyYmFsbCwge1xuICAgICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL29jdGV0LXN0cmVhbScsXG4gICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgJ01vZGlmaWVkLUF0JzogbW9kaWZpZWQudG9JU09TdHJpbmcoKSxcbiAgICAgICAgICAnT3JpZ2luLUludGVncml0eSc6IGluZm9zLmRpc3Quc2hhc3VtLFxuICAgICAgICAgICdPcmlnaW4tVVJJJzogaW5mb3MuZGlzdC50YXJiYWxsLFxuICAgICAgICAgICdTZXF1ZW5jZSc6IHNlcS50b0ZpeGVkKCksXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gUHJlcGFyZSBTUVMgbWVzc2FnZSBmb3IgaW5nZXN0aW9uXG4gICAgICBjb25zdCBtZXNzYWdlQmFzZSA9IHtcbiAgICAgICAgdGFyYmFsbFVyaTogYHMzOi8vJHtzdGFnaW5nQnVja2V0fS8ke3N0YWdpbmdLZXl9YCxcbiAgICAgICAgbWV0YWRhdGE6IHtcbiAgICAgICAgICBkaXN0OiBpbmZvcy5kaXN0LnRhcmJhbGwsXG4gICAgICAgICAgc2VxOiBzZXEudG9GaXhlZCgpLFxuICAgICAgICB9LFxuICAgICAgICB0aW1lOiBtb2RpZmllZC50b1VUQ1N0cmluZygpLFxuICAgICAgfTtcbiAgICAgIGNvbnN0IG1lc3NhZ2U6IEluZ2VzdGlvbklucHV0ID0ge1xuICAgICAgICAuLi5tZXNzYWdlQmFzZSxcbiAgICAgICAgaW50ZWdyaXR5OiBpbnRlZ3JpdHkobWVzc2FnZUJhc2UsIHRhcmJhbGwpLFxuICAgICAgfTtcblxuICAgICAgLy8gU2VuZCB0aGUgU1FTIG1lc3NhZ2Ugb3V0XG4gICAgICBhd2FpdCBhd3Muc3FzKCkuc2VuZE1lc3NhZ2Uoe1xuICAgICAgICBNZXNzYWdlQm9keTogSlNPTi5zdHJpbmdpZnkobWVzc2FnZSwgbnVsbCwgMiksXG4gICAgICAgIFF1ZXVlVXJsOiBxdWV1ZVVybCxcbiAgICAgIH0pLnByb21pc2UoKTtcblxuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5TVEFHSU5HX0ZBSUxVUkVfQ09VTlQsIDAsIFVuaXQuQ291bnQpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgLy8gU29tZXRoaW5nIGZhaWxlZCwgc3RvcmUgdGhlIHBheWxvYWQgaW4gdGhlIHByb2JsZW0gcHJlZml4LCBhbmQgbW92ZSBvbi5cbiAgICAgIGNvbnNvbGUuZXJyb3IoYFske3NlcX1dIEZhaWxlZCBwcm9jZXNzaW5nLCBsb2dnaW5nIGVycm9yIHRvIFMzIGFuZCByZXN1bWluZyB3b3JrLiAke2luZm9zLm5hbWV9QCR7aW5mb3MudmVyc2lvbn06ICR7ZXJyfWApO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5TVEFHSU5HX0ZBSUxVUkVfQ09VTlQsIDEsIFVuaXQuQ291bnQpO1xuICAgICAgYXdhaXQgcHV0T2JqZWN0KGAke1MzS2V5UHJlZml4LkZBSUxFRF9LRVlfUFJFRklYfSR7c2VxfWAsIEpTT04uc3RyaW5naWZ5KHsgLi4uaW5mb3MsIF9jb25zdHJ1Y3RfaHViX2ZhaWx1cmVfcmVhc29uOiBlcnIgfSwgbnVsbCwgMiksIHtcbiAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICAgICAgTWV0YWRhdGE6IHtcbiAgICAgICAgICAnTW9kaWZpZWQtQXQnOiBtb2RpZmllZC50b0lTT1N0cmluZygpLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9XG4gIC8vI2VuZHJlZ2lvblxuXG4gIC8vI3JlZ2lvbiBBc3luY2hyb25vdXMgUHJpbWl0aXZlc1xuICAvKipcbiAgICogTWFrZXMgYW4gSFRUUCBHRVQgcmVxdWVzdCwgYW5kIHJldHVybnMgdGhlIHJlc3VsdGluZyBwYXlsb2FkLlxuICAgKlxuICAgKiBAcGFyYW0gdXJsIHRoZSBVUkwgdG8gZ2V0LlxuICAgKlxuICAgKiBAcmV0dXJucyBhIEJ1ZmZlciBjb250YWluaW5nIHRoZSByZWNlaXZlZCBkYXRhLlxuICAgKi9cbiAgZnVuY3Rpb24gaHR0cEdldCh1cmw6IHN0cmluZykge1xuICAgIHJldHVybiBuZXcgUHJvbWlzZTxCdWZmZXI+KChvaywga28pID0+IHtcbiAgICAgIGh0dHBzLmdldCh1cmwsIChyZXNwb25zZSkgPT4ge1xuICAgICAgICBpZiAocmVzcG9uc2Uuc3RhdHVzQ29kZSAhPT0gMjAwKSB7XG4gICAgICAgICAga28obmV3IEVycm9yKGBVbnN1Y2Nlc3NmdWwgR0VUOiAke3Jlc3BvbnNlLnN0YXR1c0NvZGV9IC0gJHtyZXNwb25zZS5zdGF0dXNNZXNzYWdlfWApKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGxldCBib2R5ID0gQnVmZmVyLmFsbG9jKDApO1xuICAgICAgICByZXNwb25zZS5vbignZGF0YScsIChjaHVuaykgPT4gYm9keSA9IEJ1ZmZlci5jb25jYXQoW2JvZHksIEJ1ZmZlci5mcm9tKGNodW5rKV0pKTtcbiAgICAgICAgcmVzcG9uc2Uub25jZSgnY2xvc2UnLCAoKSA9PiBvayhib2R5KSk7XG4gICAgICAgIHJlc3BvbnNlLm9uY2UoJ2Vycm9yJywga28pO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUHV0cyBhbiBvYmplY3QgaW4gdGhlIHN0YWdpbmcgYnVja2V0LCB3aXRoIHN0YW5kYXJkaXplZCBvYmplY3QgbWV0YWRhdGEuXG4gICAqXG4gICAqIEBwYXJhbSBrZXkgIHRoZSBrZXkgZm9yIHRoZSBvYmplY3QgdG8gYmUgcHV0LlxuICAgKiBAcGFyYW0gYm9keSB0aGUgYm9keSBvZiB0aGUgb2JqZWN0IHRvIGJlIHB1dC5cbiAgICogQHBhcmFtIG9wdHMgYW55IG90aGVyIG9wdGlvbnMgdG8gdXNlIHdoZW4gc2VuZGluZyB0aGUgUzMgcmVxdWVzdC5cbiAgICpcbiAgICogQHJldHVybnMgdGhlIHJlc3VsdCBvZiB0aGUgUzMgcmVxdWVzdC5cbiAgICovXG4gIGZ1bmN0aW9uIHB1dE9iamVjdChrZXk6IHN0cmluZywgYm9keTogQVdTLlMzLkJvZHksIG9wdHM6IE9taXQ8QVdTLlMzLlB1dE9iamVjdFJlcXVlc3QsICdCdWNrZXQnIHwgJ0tleScgfCAnQm9keSc+ID0ge30pIHtcbiAgICByZXR1cm4gYXdzLnMzKCkucHV0T2JqZWN0KHtcbiAgICAgIEJ1Y2tldDogc3RhZ2luZ0J1Y2tldCxcbiAgICAgIEtleToga2V5LFxuICAgICAgQm9keTogYm9keSxcbiAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICdMYW1iZGEtTG9nLUdyb3VwJzogY29udGV4dC5sb2dHcm91cE5hbWUsXG4gICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICAgLi4ub3B0cy5NZXRhZGF0YSxcbiAgICAgIH0sXG4gICAgICAuLi5vcHRzLFxuICAgIH0pLnByb21pc2UoKTtcbiAgfVxuICAvLyNlbmRyZWdpb25cbn1cblxuLyoqXG4gKiBPYnRhaW5zIHRoZSBgVmVyc2lvbkluZm9gIGNvcnJlc3BvbmRpbmcgdG8gdGhlIG1vZGlmaWVkIHZlcnNpb24ocykgaW4gdGhlXG4gKiBwcm92aWRlZCBgQ2hhbmdlYCBvYmplY3RzLCBlbnN1cmVzIHRoZXkgYXJlIHJlbGV2YW50IChjb25zdHJ1Y3QgbGlicmFyaWVzKSxcbiAqIGFuZCByZXR1cm5zIHRob3NlIG9ubHkuXG4gKlxuICogQHBhcmFtIGNoYW5nZXMgdGhlIGNoYW5nZXMgdG8gYmUgcHJvY2Vzc2VkLlxuICogQHBhcmFtIG1ldHJpY3MgdGhlIG1ldHJpY3MgbG9nZ2VyIHRvIHVzZS5cbiAqIEBwYXJhbSBkZW55TGlzdCBkZW55IGxpc3QgY2xpZW50XG4gKlxuICogQHJldHVybnMgYSBsaXN0IG9mIGBWZXJzaW9uSW5mb2Agb2JqZWN0c1xuICovXG5mdW5jdGlvbiBnZXRSZWxldmFudFZlcnNpb25JbmZvcyhcbiAgY2hhbmdlczogcmVhZG9ubHkgQ2hhbmdlW10sXG4gIG1ldHJpY3M6IE1ldHJpY3NMb2dnZXIsXG4gIGRlbnlMaXN0OiBEZW55TGlzdENsaWVudCxcbiAgbGljZW5zZUxpc3Q6IExpY2Vuc2VMaXN0Q2xpZW50LFxuKTogcmVhZG9ubHkgVXBkYXRlZFZlcnNpb25bXSB7XG5cbiAgY29uc3QgcmVzdWx0ID0gbmV3IEFycmF5PFVwZGF0ZWRWZXJzaW9uPigpO1xuXG4gIGZvciAoY29uc3QgY2hhbmdlIG9mIGNoYW5nZXMpIHtcbiAgICAvLyBGaWx0ZXIgb3V0IGFsbCBlbGVtZW50cyB0aGF0IGRvbid0IGhhdmUgYSBcIm5hbWVcIiBpbiB0aGUgZG9jdW1lbnQsIGFzXG4gICAgLy8gdGhlc2UgYXJlIHNjaGVtYXMsIHdoaWNoIGFyZSBub3QgcmVsZXZhbnQgdG8gb3VyIGJ1c2luZXNzIGhlcmUuXG4gICAgaWYgKGNoYW5nZS5kb2MubmFtZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBjb25zb2xlLmVycm9yKGBbJHtjaGFuZ2Uuc2VxfV0gQ2hhbmdlZCBkb2N1bWVudCBjb250YWlucyBubyAnbmFtZSc6ICR7Y2hhbmdlLmlkfWApO1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5VTlBST0NFU1NBQkxFX0VOVElUWSwgMSwgVW5pdC5Db3VudCk7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG5cbiAgICAvLyBUaGUgbm9ybWFsaXplIGZ1bmN0aW9uIGNoYW5nZSB0aGUgb2JqZWN0IGluIHBsYWNlLCBpZiB0aGUgZG9jIG9iamVjdCBpcyBpbnZhbGlkIGl0IHdpbGwgcmV0dXJuIHVuZGVmaW5lZFxuICAgIGlmIChub3JtYWxpemVOUE1NZXRhZGF0YShjaGFuZ2UuZG9jKSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBjb25zb2xlLmVycm9yKGBbJHtjaGFuZ2Uuc2VxfV0gQ2hhbmdlZCBkb2N1bWVudCBpbnZhbGlkLCBucG0gbm9ybWFsaXplIHJldHVybmVkIHVuZGVmaW5lZDogJHtjaGFuZ2UuaWR9YCk7XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlVOUFJPQ0VTU0FCTEVfRU5USVRZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cblxuICAgIC8vIFNvbWV0aW1lcywgdGhlcmUgYXJlIG5vIHZlcnNpb25zIGluIHRoZSBkb2N1bWVudC4gV2Ugc2tpcCB0aG9zZS5cbiAgICBpZiAoY2hhbmdlLmRvYy52ZXJzaW9ucyA9PSBudWxsKSB7XG4gICAgICBjb25zb2xlLmVycm9yKGBbJHtjaGFuZ2Uuc2VxfV0gQ2hhbmdlZCBkb2N1bWVudCBjb250YWlucyBubyAndmVyc2lvbnMnOiAke2NoYW5nZS5pZH1gKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuVU5QUk9DRVNTQUJMRV9FTlRJVFksIDEsIFVuaXQuQ291bnQpO1xuICAgICAgY29udGludWU7XG4gICAgfVxuXG4gICAgLy8gU29tZXRpbWVzLCB0aGVyZSBpcyBubyAndGltZScgZW50cnkgaW4gdGhlIGRvY3VtZW50LiBXZSBza2lwIHRob3NlLlxuICAgIGlmIChjaGFuZ2UuZG9jLnRpbWUgPT0gbnVsbCkge1xuICAgICAgY29uc29sZS5lcnJvcihgWyR7Y2hhbmdlLnNlcX1dIENoYW5nZWQgZG9jdW1lbnQgY29udGFpbnMgbm8gJ3RpbWUnOiAke2NoYW5nZS5pZH1gKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuVU5QUk9DRVNTQUJMRV9FTlRJVFksIDEsIFVuaXQuQ291bnQpO1xuICAgICAgY29udGludWU7XG4gICAgfVxuXG4gICAgLy8gR2V0IHRoZSBsYXN0IG1vZGlmaWNhdGlvbiBkYXRlIGZyb20gdGhlIGNoYW5nZVxuICAgIGNvbnN0IHNvcnRlZFVwZGF0ZXMgPSBPYmplY3QuZW50cmllcyhjaGFuZ2UuZG9jLnRpbWUpXG4gICAgICAvLyBJZ25vcmUgdGhlIFwiY3JlYXRlZFwiIGFuZCBcIm1vZGlmaWVkXCIga2V5cyBoZXJlXG4gICAgICAuZmlsdGVyKChba2V5XSkgPT4ga2V5ICE9PSAnY3JlYXRlZCcgJiYga2V5ICE9PSAnbW9kaWZpZWQnKVxuICAgICAgLy8gUGFyc2UgYWxsIHRoZSBkYXRlcyB0byBlbnN1cmUgdGhleSBhcmUgY29tcGFyYWJsZVxuICAgICAgLm1hcCgoW3ZlcnNpb24sIGlzb0RhdGVdKSA9PiBbdmVyc2lvbiwgbmV3IERhdGUoaXNvRGF0ZSldIGFzIGNvbnN0KVxuICAgICAgLy8gU29ydCBieSBkYXRlLCBkZXNjZW5kaW5nXG4gICAgICAuc29ydCgoWywgbF0sIFssIHJdKSA9PiByLmdldFRpbWUoKSAtIGwuZ2V0VGltZSgpKTtcbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlBBQ0tBR0VfVkVSU0lPTl9DT1VOVCwgc29ydGVkVXBkYXRlcy5sZW5ndGgsIFVuaXQuQ291bnQpO1xuXG4gICAgbGV0IGxhdGVzdE1vZGlmaWVkOiBEYXRlIHwgdW5kZWZpbmVkO1xuICAgIGZvciAoY29uc3QgW3ZlcnNpb24sIG1vZGlmaWVkXSBvZiBzb3J0ZWRVcGRhdGVzKSB7XG4gICAgICBpZiAobGF0ZXN0TW9kaWZpZWQgPT0gbnVsbCB8fCBsYXRlc3RNb2RpZmllZC5nZXRUaW1lKCkgPT09IG1vZGlmaWVkLmdldFRpbWUoKSkge1xuICAgICAgICBjb25zdCBpbmZvcyA9IGNoYW5nZS5kb2MudmVyc2lvbnNbdmVyc2lvbl07XG4gICAgICAgIGlmIChpbmZvcyA9PSBudWxsKSB7XG4gICAgICAgICAgLy8gQ291bGQgYmUgdGhlIHZlcnNpb24gaW4gcXVlc3Rpb24gd2FzIHVuLXB1Ymxpc2hlZC5cbiAgICAgICAgICBjb25zb2xlLmxvZyhgWyR7Y2hhbmdlLnNlcX1dIENvdWxkIG5vdCBmaW5kIGluZm8gZm9yIFwiJHtjaGFuZ2UuZG9jLm5hbWV9QCR7dmVyc2lvbn1cIi4gV2FzIGl0IHVuLXB1Ymxpc2hlZD9gKTtcbiAgICAgICAgfSBlbHNlIGlmIChpc0NvbnN0cnVjdExpYnJhcnkoaW5mb3MpKSB7XG5cbiAgICAgICAgICAvLyBza2lwIGlmIHRoaXMgcGFja2FnZSBpcyBkZW5pZWRcbiAgICAgICAgICBjb25zdCBkZW5pZWQgPSBkZW55TGlzdC5sb29rdXAoaW5mb3MubmFtZSwgaW5mb3MudmVyc2lvbik7XG4gICAgICAgICAgaWYgKGRlbmllZCkge1xuICAgICAgICAgICAgY29uc29sZS5sb2coYFske2NoYW5nZS5zZXF9XSBQYWNrYWdlIGRlbmllZDogJHtKU09OLnN0cmluZ2lmeShkZW5pZWQpfWApO1xuICAgICAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5ERU5ZX0xJU1RFRF9DT1VOVCwgMSwgVW5pdC5Db3VudCk7XG4gICAgICAgICAgICBjb250aW51ZTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlBBQ0tBR0VfVkVSU0lPTl9BR0UsIERhdGUubm93KCkgLSBtb2RpZmllZC5nZXRUaW1lKCksIFVuaXQuTWlsbGlzZWNvbmRzKTtcbiAgICAgICAgICBjb25zdCBpc0VsaWdpYmxlID0gbGljZW5zZUxpc3QubG9va3VwKGluZm9zLmxpY2Vuc2UgPz8gJ1VOTElDRU5TRUQnKSAhPSBudWxsO1xuICAgICAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuSU5FTElHSUJMRV9MSUNFTlNFLCBpc0VsaWdpYmxlID8gMCA6IDEsIFVuaXQuQ291bnQpO1xuICAgICAgICAgIGlmIChpc0VsaWdpYmxlKSB7XG4gICAgICAgICAgICByZXN1bHQucHVzaCh7IGluZm9zLCBtb2RpZmllZCwgc2VxOiBjaGFuZ2Uuc2VxIH0pO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBjb25zb2xlLmxvZyhgWyR7Y2hhbmdlLnNlcX1dIFBhY2thZ2UgXCIke2NoYW5nZS5kb2MubmFtZX1AJHt2ZXJzaW9ufVwiIGRvZXMgbm90IHVzZSBhbGxvdy1saXN0ZWQgbGljZW5zZTogJHtpbmZvcy5saWNlbnNlID8/ICdVTkxJQ0VOU0VEJ31gKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29uc29sZS5sb2coYFske2NoYW5nZS5zZXF9XSBJZ25vcmluZyBcIiR7Y2hhbmdlLmRvYy5uYW1lfUAke3ZlcnNpb259XCIgYXMgaXQgaXMgbm90IGEgY29uc3RydWN0IGxpYnJhcnkuYCk7XG4gICAgICAgIH1cbiAgICAgICAgbGF0ZXN0TW9kaWZpZWQgPSBtb2RpZmllZDtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgcmV0dXJuIHJlc3VsdDtcblxuICBmdW5jdGlvbiBpc0NvbnN0cnVjdExpYnJhcnkoaW5mb3M6IFZlcnNpb25JbmZvKTogYm9vbGVhbiB7XG4gICAgaWYgKGluZm9zLmpzaWkgPT0gbnVsbCkge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cbiAgICByZXR1cm4gaW5mb3MubmFtZSA9PT0gJ2NvbnN0cnVjdCdcbiAgICAgIHx8IGluZm9zLm5hbWUgPT09ICdhd3MtY2RrLWxpYidcbiAgICAgIHx8IGluZm9zLm5hbWUuc3RhcnRzV2l0aCgnQGF3cy1jZGsnKVxuICAgICAgfHwgaW5mb3Mua2V5d29yZHM/LnNvbWUoKGt3KSA9PiBDT05TVFJVQ1RfS0VZV09SRFMuaGFzKGt3KSk7XG4gIH1cbn1cblxuLyoqXG4gICogVGhlIHNjaGVtZSBvZiBhIHBhY2thZ2UgdmVyc2lvbiBpbiB0aGUgdXBkYXRlLiBJbmNsdWRlcyB0aGUgcGFja2FnZS5qc29uIGtleXMsIGFzIHdlbGwgYXMgc29tZSBhZGRpdGlvbmFsIG5wbSBtZXRhZGF0YVxuICAqIEBzZWUgaHR0cHM6Ly9naXRodWIuY29tL25wbS9yZWdpc3RyeS9ibG9iL21hc3Rlci9kb2NzL1JFR0lTVFJZLUFQSS5tZCN2ZXJzaW9uXG4gICovXG5pbnRlcmZhY2UgVmVyc2lvbkluZm8ge1xuICByZWFkb25seSBkZXZEZXBlbmRlbmNpZXM6IHsgcmVhZG9ubHkgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICByZWFkb25seSBkZXBlbmRlbmNpZXM6IHsgcmVhZG9ubHkgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICByZWFkb25seSBqc2lpOiB1bmtub3duO1xuICByZWFkb25seSBsaWNlbnNlPzogc3RyaW5nO1xuICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IFtrZXk6IHN0cmluZ106IHVua25vd247XG4gIHJlYWRvbmx5IGtleXdvcmRzOiBzdHJpbmdbXTtcbiAgcmVhZG9ubHkgZGlzdDoge1xuICAgIHJlYWRvbmx5IHNoYXN1bTogc3RyaW5nO1xuICAgIHJlYWRvbmx5IHRhcmJhbGw6IHN0cmluZztcbiAgfTtcbiAgcmVhZG9ubHkgdmVyc2lvbjogc3RyaW5nO1xufVxuXG5pbnRlcmZhY2UgVXBkYXRlZFZlcnNpb24ge1xuICAvKipcbiAgICogVGhlIGBWZXJzaW9uSW5mb2AgZm9yIHRoZSBtb2RpZmllZCBwYWNrYWdlIHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSBpbmZvczogVmVyc2lvbkluZm87XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lIGF0IHdoaWNoIHRoZSBgVmVyc2lvbkluZm9gIHdhcyBsYXN0IG1vZGlmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9kaWZpZWQ6IERhdGU7XG5cbiAgLyoqXG4gICAqIFRoZSBDb3VjaERCIHRyYW5zYWN0aW9uIG51bWJlciBmb3IgdGhlIHVwZGF0ZS5cbiAgICovXG4gIHJlYWRvbmx5IHNlcTogbnVtYmVyO1xufVxuXG5pbnRlcmZhY2UgRG9jdW1lbnQge1xuXG4gIC8qKlxuICAgKiBhIExpc3Qgb2YgYWxsIFZlcnNpb24gb2JqZWN0cyBmb3IgdGhlIHBhY2thZ2VcbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb25zOiB7IFtrZXk6c3RyaW5nXTogVmVyc2lvbkluZm8gfCB1bmRlZmluZWQgfTtcblxuICAvKipcbiAgICogVGhlIHBhY2thZ2UncyBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaW1lc3RhbXBzIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGRvY3VtZW50LiBUaGUgdmFsdWVzIGFyZSBJU08tODYwMSBlbmNvZGVkXG4gICAqIHRpbWVzdGFtcHMuXG4gICAqL1xuICByZWFkb25seSB0aW1lOiB7XG4gICAgcmVhZG9ubHkgY3JlYXRlZDogc3RyaW5nO1xuICAgIHJlYWRvbmx5IG1vZGlmaWVkOiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgW3ZlcnNpb246IHN0cmluZ106IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIENoYW5nZSB7XG4gIHJlYWRvbmx5IHNlcTogbnVtYmVyO1xuICByZWFkb25seSBkb2M6IERvY3VtZW50O1xuICByZWFkb25seSBpZDogc3RyaW5nO1xuICByZWFkb25seSBkZWxldGVkOiBib29sZWFuO1xufVxuIl19