"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./private/cross-region-support-stack");
const full_action_descriptor_1 = require("./private/full-action-descriptor");
const rich_action_1 = require("./private/rich-action");
const stage_1 = require("./private/stage");
const validation_1 = require("./private/validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     * @stability stable
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
    /**
     * Returns a source configuration for notification rule.
     *
     * @stability stable
     */
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.pipelineArn,
        };
    }
    /**
     * Defines a CodeStar notification rule triggered when the pipeline events emitted by you specified, it very similar to `onEvent` API.
     *
     * You can also use the methods `notifyOnExecutionStateChange`, `notifyOnAnyStageStateChange`,
     * `notifyOnAnyActionStateChange` and `notifyOnAnyManualApprovalStateChange`
     * to define rules for these specific event emitted.
     *
     * @stability stable
     */
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    /**
     * Define an notification rule triggered by the set of the "Pipeline execution" events emitted from this pipeline.
     *
     * @stability stable
     */
    notifyOnExecutionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUCCEEDED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUPERSEDED,
            ],
        });
    }
    /**
     * Define an notification rule triggered by the set of the "Stage execution" events emitted from this pipeline.
     *
     * @stability stable
     */
    notifyOnAnyStageStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_SUCCEEDED,
            ],
        });
    }
    /**
     * Define an notification rule triggered by the set of the "Action execution" events emitted from this pipeline.
     *
     * @stability stable
     */
    notifyOnAnyActionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_SUCCEEDED,
            ],
        });
    }
    /**
     * Define an notification rule triggered by the set of the "Manual approval" events emitted from this pipeline.
     *
     * @stability stable
     */
    notifyOnAnyManualApprovalStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_FAILED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_NEEDED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_SUCCEEDED,
            ],
        });
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @stability stable
 * @example
 *
 * // create a pipeline
 * import * as codecommit from '@aws-cdk/aws-codecommit';
 *
 * const pipeline = new codepipeline.Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * declare const repo: codecommit.Repository;
 * declare const sourceArtifact: codepipeline.Artifact;
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 * actionName: 'Source',
 * output: sourceArtifact,
 * repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_PipelineProps(props);
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // @deprecated(v2): switch to default false
        this.crossAccountKeys = (_b = props.crossAccountKeys) !== null && _b !== void 0 ? _b : true;
        this.enableKeyRotation = props.enableKeyRotation;
        // Cross account keys must be set for key rotation to be enabled
        if (this.enableKeyRotation && !this.crossAccountKeys) {
            throw new Error("Setting 'enableKeyRotation' to true also requires 'crossAccountKeys' to be enabled");
        }
        this.reuseCrossRegionSupportStacks = (_c = props.reuseCrossRegionSupportStacks) !== null && _c !== void 0 ? _c : true;
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            let encryptionKey;
            if (this.crossAccountKeys) {
                encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                    // remove the key - there is a grace period of a few days before it's gone for good,
                    // that should be enough for any emergency access to the bucket artifacts
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                    enableKeyRotation: this.enableKeyRotation,
                });
                // add an alias to make finding the key in the console easier
                new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                    aliasName: this.generateNameForDefaultBucketKeyAlias(),
                    targetKey: encryptionKey,
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
            }
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: encryptionKey ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
                enforceSSL: true,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.any({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.any({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.any({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline.
     * @param id the logical ID of the returned pipeline construct.
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`).
     * @stability stable
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).splitArn(pipelineArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage.
     * @returns the newly created Stage
     * @stability stable
     */
    addStage(props) {
        jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_StageOptions(props);
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     *
     * @stability stable
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     *
     * @stability stable
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name.
     *
     * @stability stable
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @stability stable
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        const richAction = new rich_action_1.RichAction(action, this);
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(richAction);
        // get the role for the given action, handling if it's cross-account
        const actionRole = this.getRoleForAction(stage, richAction, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(richAction.actionProperties.variablesNamespace);
        // bind the Action (type h4x)
        const actionConfig = richAction.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            // must be 'action', not 'richAction',
            // as those are returned by the IStage.actions property,
            // and it's important customers of Pipeline get the same instance
            // back as they added to the pipeline
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure.
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     *
     * @stability stable
     * @override true
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        if (!action.isCrossRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // The action has a specific region,
        // require the pipeline to have a known region as well.
        this.requireRegion();
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        const crossRegionSupport = this.obtainCrossRegionSupportFor(action);
        // the stack containing the replication bucket must be deployed before the pipeline
        core_1.Stack.of(this).addDependency(crossRegionSupport.stack);
        // The Pipeline role must be able to replicate to that bucket
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: action.effectiveRegion,
        };
    }
    /**
     * Get or create the cross-region support construct for the given action
     */
    obtainCrossRegionSupportFor(action) {
        // this method is never called for non cross-region actions
        const actionRegion = action.effectiveRegion;
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            const otherStack = action.resourceStack;
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        return crossRegionSupport;
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id, {
                    createKmsKey: this.crossAccountKeys,
                    enableKeyRotation: this.enableKeyRotation,
                });
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.supportScope();
        const supportStackId = `cross-region-stack-${this.reuseCrossRegionSupportStacks ? pipelineAccount : pipelineStack.stackName}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
                createKmsKey: this.crossAccountKeys,
                enableKeyRotation: this.enableKeyRotation,
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = core_1.Names.uniqueId(this);
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        // (otherwise we can't configure cross-account trust policies)
        if (action.isCrossAccount) {
            const artifactBucket = this.ensureReplicationResourcesExistFor(action).artifactBucket;
            if (!artifactBucket.encryptionKey) {
                throw new Error(`Artifact Bucket must have a KMS Key to add cross-account action '${action.actionProperties.actionName}' ` +
                    `(pipeline account: '${renderEnvDimension(this.env.account)}', action account: '${renderEnvDimension(action.effectiveAccount)}'). ` +
                    'Create Pipeline with \'crossAccountKeys: true\' (or pass an existing Bucket with a key)');
            }
        }
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${core_1.Names.uniqueId(this)}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const targetAccount = action.actionProperties.resource
            ? action.actionProperties.resource.env.account
            : action.actionProperties.account;
        if (targetAccount === undefined) {
            // if the account of the Action is not specified,
            // then it defaults to the same account the pipeline itself is in
            return undefined;
        }
        // check whether the action's account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            if (core_1.Token.isUnresolved(this.env.account)) {
                // the pipeline is also env-agnostic, so that's fine
                return undefined;
            }
            else {
                throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
            }
        }
        // At this point, we know that the action's account is a static string.
        // In this case, the pipeline's account must also be a static string.
        if (core_1.Token.isUnresolved(this.env.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        // at this point, we know that both the Pipeline's account,
        // and the action-backing resource's account are static strings
        // if they are identical - nothing to do (the action is not cross-account)
        if (this.env.account === targetAccount) {
            return undefined;
        }
        // at this point, we know that the action is certainly cross-account,
        // so we need to return a Stack in its account to create the helper Role in
        const candidateActionResourceStack = action.actionProperties.resource
            ? core_1.Stack.of(action.actionProperties.resource)
            : undefined;
        if ((candidateActionResourceStack === null || candidateActionResourceStack === void 0 ? void 0 : candidateActionResourceStack.account) === targetAccount) {
            // we always use the "latest" action-backing resource's Stack for this account,
            // even if a different one was used earlier
            this._crossAccountSupport[targetAccount] = candidateActionResourceStack;
            return candidateActionResourceStack;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.supportScope();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                const actionRegion = action.actionProperties.resource
                    ? action.actionProperties.resource.env.region
                    : action.actionProperties.region;
                const pipelineStack = core_1.Stack.of(this);
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: actionRegion !== null && actionRegion !== void 0 ? actionRegion : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    supportScope() {
        const scope = core_1.Stage.of(this);
        if (!scope) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK App or Stage');
        }
        return scope;
    }
}
exports.Pipeline = Pipeline;
_a = JSII_RTTI_SYMBOL_1;
Pipeline[_a] = { fqn: "@aws-cdk/aws-codepipeline.Pipeline", version: "1.144.0" };
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
/**
 * Render an env dimension without showing the ugly stringified tokens
 */
function renderEnvDimension(s) {
    return core_1.Token.isUnresolved(s) ? '(current)' : s;
}
//# sourceMappingURL=data:application/json;base64,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