"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.detectConstructFramework = void 0;
const semver_1 = require("semver");
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a, _b;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    // exception: we assume all @cdktf/ libraries are cdktf, even if they
    // also take other CDK types as dependencies
    if (assembly.name.startsWith('@cdktf/')) {
        name = "cdktf" /* CDKTF */;
        if ('cdktf' in ((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
            detectConstructFrameworkPackage('cdktf');
        }
        return { name, majorVersion };
    }
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_b = assembly.dependencyClosure) !== null && _b !== void 0 ? _b : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf') {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
exports.detectConstructFramework = detectConstructFramework;
//# sourceMappingURL=data:application/json;base64,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