"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const storage_1 = require("../../s3/storage");
/**
 * A list of licenses, which can be used to control filter packages before
 * indexing.
 */
class LicenseList extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The object key in which the license list is stored.
         */
        this.objectKey = 'allowed-licenses.json';
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.RETAIN,
            versioned: true,
        });
        this.upload = new s3deploy.BucketDeployment(this, 'Resource', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: true,
            sources: [this.createAsset(props.licenses)],
        });
    }
    /**
     * Grants an AWS Lambda function permissions to read the license allow list,
     * and adds the relevant environment variables expected by the
     * `LicenseListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment("LICENSE_LIST_BUCKET_NAME" /* BUCKET_NAME */, this.bucket.bucketName);
        handler.addEnvironment("LICENSE_LIST_OBJECT_KEY" /* OBJECT_KEY */, this.objectKey);
        this.bucket.grantRead(handler);
        // The handler now depends on the deny-list having been uploaded
        handler.node.addDependency(this.upload);
    }
    createAsset(licenses) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'license-list-'));
        fs.writeFileSync(path.join(tmpdir, this.objectKey), JSON.stringify(licenses.map((l) => l.id), null, 2), 'utf-8');
        return s3deploy.Source.asset(tmpdir);
    }
}
exports.LicenseList = LicenseList;
//# sourceMappingURL=data:application/json;base64,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