"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isTagApplicable = void 0;
const package_tag_1 = require("../../package-tag");
/**
 * Extract value from tag condition source.
 */
function getNestedField(config, artifacts) {
    switch (config === null || config === void 0 ? void 0 : config.source) {
        case package_tag_1.TagConditionSource.PACKAGE_JSON:
            const path = config === null || config === void 0 ? void 0 : config.key;
            return path === null || path === void 0 ? void 0 : path.reduce((accum, key) => (accum ? accum[key] : undefined), artifacts.pkg);
        case package_tag_1.TagConditionSource.README:
            return artifacts.readme;
        default:
            throw new Error(`Unexpected TagConditionConfig source: "${config.source}"`);
    }
}
/**
 * Create a function that combines conditions using && or || logic.
 */
const createBoolCombinator = (combine, initial) => (config, artifacts) => { var _a, _b; return (_b = (_a = config === null || config === void 0 ? void 0 : config.children) === null || _a === void 0 ? void 0 : _a.reduce((accum, cond) => combine(accum, isTagApplicable(cond, artifacts)), initial)) !== null && _b !== void 0 ? _b : true; };
/**
 * Create a function that checks nested value against predicate.
 */
const createFieldComparator = (compare) => (config, artifacts) => compare(getNestedField(config, artifacts), config === null || config === void 0 ? void 0 : config.value, config === null || config === void 0 ? void 0 : config.options);
/**
 * Negate and recurse for `not` functionality.
 */
const not = (config, artifacts) => {
    var _a;
    const cond = (_a = config === null || config === void 0 ? void 0 : config.children) === null || _a === void 0 ? void 0 : _a[0];
    if (!cond) {
        throw new Error('NOT logical operator requires a single condition');
    }
    return !isTagApplicable(cond, artifacts);
};
const checkIncludes = (haystack, needle, options = {}) => {
    var _a, _b, _c;
    const atLeast = (_a = options.atLeast) !== null && _a !== void 0 ? _a : 1;
    const caseSensitive = (_b = options.caseSensitive) !== null && _b !== void 0 ? _b : false;
    if (typeof haystack === 'string') {
        if (typeof needle !== 'string') {
            return false;
        }
        if (!caseSensitive) {
            needle = needle.toLowerCase();
            haystack = haystack.toLowerCase();
        }
        const matches = (_c = haystack.match(new RegExp(needle, 'g'))) !== null && _c !== void 0 ? _c : [];
        return matches.length >= atLeast;
    }
    if (Array.isArray(haystack)) {
        let matches = 0;
        for (const item of haystack) {
            // TODO: add deep equality checking for comparing objects? ¯\_(ツ)_/¯
            if (item === needle) {
                matches += 1;
            }
        }
        return matches >= atLeast;
    }
    return false;
};
/**
 * Checks whether a tag's condition applies to a package by computing declared
 * logic and running against the package.json and README. Recursively constructs
 * chains of `&&` and `||` conditions to allow arbitrary combinations.
 */
function isTagApplicable(config, artifacts) {
    var _a;
    const fnMap = {
        [package_tag_1.TagConditionLogicType.AND]: createBoolCombinator((field, val) => field && val, true),
        [package_tag_1.TagConditionLogicType.OR]: createBoolCombinator((field, val) => field || val, false),
        [package_tag_1.TagConditionLogicType.EQUALS]: createFieldComparator((field, val) => field === val),
        [package_tag_1.TagConditionLogicType.INCLUDES]: createFieldComparator((field, val) => checkIncludes(field, val, config.options)),
        [package_tag_1.TagConditionLogicType.STARTS_WITH]: createFieldComparator((field, val) => field.startsWith(val)),
        [package_tag_1.TagConditionLogicType.NOT]: not,
    };
    const fn = (_a = fnMap[config.type]) !== null && _a !== void 0 ? _a : (() => false);
    return fn(config, artifacts);
}
exports.isTagApplicable = isTagApplicable;
//# sourceMappingURL=data:application/json;base64,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