"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const tls = require("tls");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const env_lambda_shared_1 = require("../backend/shared/env.lambda-shared");
/**
 * Obtains the TLS certificate used by the HTTPS endpoint designated by the
 * `HTTPS_ENDPOINT` environment variable, checks how many validity days remain
 * for this endpoint, and produces a CloudWatch metric under the namespace
 * designated by `METRIC_NAMESPACE`, named as specified by `METRIC_NAME`.
 *
 * If the certificate is past expiration, the metric will be trimmed to `0`
 * instead of turning into negative values.
 */
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, _context) => {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const endpoint = env_lambda_shared_1.requireEnv('HTTPS_ENDPOINT');
    const metricNamespace = env_lambda_shared_1.requireEnv('METRIC_NAMESPACE');
    const metricName = env_lambda_shared_1.requireEnv('METRIC_NAME');
    const now = new Date();
    const daysRemaining = await tlsValidDaysRemaining(endpoint, now);
    console.log(`The certificate has ${daysRemaining} remaining validity days`);
    metrics.setDimensions({ DomainName: endpoint });
    metrics.setNamespace(metricNamespace);
    metrics.putMetric(metricName, daysRemaining, aws_embedded_metrics_1.Unit.Count);
});
/**
 * Obtains the remaining validity days from the provided `endpoint`. This
 * establishes a TLS connection to the `endpoint` on port `443`, and assesses
 * the `valid_to` value therein. This function also works with endpoints that
 * require SNI in order to present a certificate (as is the case for API Gateway
 * and CloudFront, for example). It does however not bother with ALPN protocols.
 *
 * @param endpoint the domain name for which a certificate is needed.
 * @param now      the date to use as the `now` reference point.
 *
 * @returns the amount of days the certificate is still valid for, or `0` if the
 *          certificate has already expired.
 */
function tlsValidDaysRemaining(endpoint, now) {
    console.log(`Checking remaining validity time for certificate on ${endpoint}:443`);
    const sock = tls.connect({
        host: endpoint,
        port: 443,
        servername: endpoint,
    });
    // "Cleanly" terminates the socket connection, so it does not leak
    const closeThen = (cb) => sock.end(() => {
        sock.destroy();
        sock.unref();
        cb();
    });
    return new Promise((resolve, reject) => {
        const ok = (val) => closeThen(() => resolve(val));
        const ko = (reason) => closeThen(() => reject(reason));
        try {
            sock.once('error', (err) => err.code === 'CERT_HAS_EXPIRED' ? ok(0) : ko(err));
            sock.once('secureConnect', () => {
                const cert = sock.getPeerCertificate();
                console.log(`Secure connection established: ${cert.fingerprint256} is valid from ${cert.valid_from} until ${cert.valid_to}`);
                const remainingMillis = new Date(cert.valid_to).getTime() - now.getTime();
                if (remainingMillis < 0) {
                    console.error(`The certificate expired ${remainingMillis}ms ago!`);
                    ok(0);
                    return;
                }
                // Converting to days (there are 86,400,000 milliseconds in a day)
                ok(remainingMillis / 86400000);
            });
        }
        catch (e) {
            ko(e);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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