"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmJs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const runbook_url_1 = require("../runbook-url");
const storage_1 = require("../s3/storage");
const canary_1 = require("./npmjs/canary");
const constants_lambda_shared_1 = require("./npmjs/constants.lambda-shared");
const npm_js_follower_1 = require("./npmjs/npm-js-follower");
const stage_and_notify_1 = require("./npmjs/stage-and-notify");
/**
 * The periodicity at which the NpmJs follower will run. This MUST be a valid
 * CloudWatch Metric grain, as this will also be the period of the CloudWatch
 * alarm that montiors the health of the follower.
 */
const FOLLOWER_RUN_RATE = core_1.Duration.minutes(5);
/**
 * (experimental) A package source that gets package data from the npmjs.com package registry.
 *
 * @experimental
 */
class NpmJs {
    /**
     * @experimental
     */
    constructor(props = {}) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { baseUrl, denyList, ingestion, licenseList, monitoring, queue, repository }) {
        var _b, _c, _d;
        repository === null || repository === void 0 ? void 0 : repository.addExternalConnection('public:npmjs');
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(scope);
        const bucket = this.props.stagingBucket || storageFactory.newBucket(scope, 'NpmJs/StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ prefix: "staged/" /* STAGED_KEY_PREFIX */, expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const stager = new stage_and_notify_1.StageAndNotify(scope, 'NpmJs-StageAndNotify', {
            deadLetterQueue: new aws_sqs_1.Queue(scope, 'StagerDLQ', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                retentionPeriod: core_1.Duration.days(14),
                visibilityTimeout: core_1.Duration.minutes(15),
            }),
            description: `[${scope.node.path}/NpmJS-StageAndNotify] Stages tarballs to S3 and notifies ConstructHub`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 10024,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(stager);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(stager);
        queue.grantSendMessages(stager);
        stager.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(stager.deadLetterQueue, { batchSize: 1, enabled: false }));
        const follower = new npm_js_follower_1.NpmJsFollower(scope, 'NpmJs', {
            description: `[${scope.node.path}/NpmJs] Periodically query npmjs.com index for new packages`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                FUNCTION_NAME: stager.functionName,
            },
            memorySize: 10024,
            reservedConcurrentExecutions: 1,
            timeout: FOLLOWER_RUN_RATE,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(follower, constants_lambda_shared_1.MARKER_FILE_NAME);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(follower);
        licenseList.grantRead(follower);
        stager.grantInvoke(follower);
        const rule = new aws_events_1.Rule(scope, 'NpmJs/Schedule', {
            description: `${scope.node.path}/NpmJs/Schedule`,
            schedule: aws_events_1.Schedule.rate(FOLLOWER_RUN_RATE),
            targets: [new aws_events_targets_1.LambdaFunction(follower)],
        });
        this.registerAlarms(scope, follower, stager, monitoring, rule);
        return {
            name: follower.node.path,
            links: [
                { name: 'NpmJs Follower', url: deep_link_1.lambdaFunctionUrl(follower), primary: true },
                { name: 'Marker Object', url: deep_link_1.s3ObjectUrl(bucket, constants_lambda_shared_1.MARKER_FILE_NAME) },
                { name: 'Stager', url: deep_link_1.lambdaFunctionUrl(stager) },
                { name: 'Stager DLQ', url: deep_link_1.sqsQueueUrl(stager.deadLetterQueue) },
            ],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Follower Health',
                        left: [
                            metric_utils_1.fillMetric(follower.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(follower.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            this.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Health',
                        left: [
                            metric_utils_1.fillMetric(stager.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(stager.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ], [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            metric_utils_1.fillMetric(this.metricChangeCount({ label: 'Change Count' }), 0),
                            metric_utils_1.fillMetric(this.metricUnprocessableEntity({ label: 'Unprocessable' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            metric_utils_1.fillMetric(this.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            metric_utils_1.fillMetric(this.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Changes',
                        left: [
                            metric_utils_1.fillMetric(this.metricLastSeq({ label: 'Last Sequence Number' }), 'REPEAT'),
                        ],
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Dead-Letter Queue',
                        left: [
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }), 0),
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    ...(((_b = this.props.enableCanary) !== null && _b !== void 0 ? _b : true)
                        ? this.registerCanary(follower, (_c = this.props.canaryPackage) !== null && _c !== void 0 ? _c : 'construct-hub-probe', (_d = this.props.canarySla) !== null && _d !== void 0 ? _d : core_1.Duration.minutes(5), bucket, baseUrl, monitoring)
                        : []),
                ],
            ],
        };
    }
    /**
     * (experimental) The average time it took to process a changes batch.
     *
     * @experimental
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of changes that were processed.
     *
     * @experimental
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The last sequence number that was processed.
     *
     * This metric can be used to
     * discover when a sequence reset has happened in the CouchDB instance.
     *
     * @experimental
     */
    metricLastSeq(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "LastSeq" /* LAST_SEQ */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * @experimental
     */
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The age of the oldest package version that was processed.
     *
     * @experimental
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were inspected.
     *
     * @experimental
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were deemed relevant.
     *
     * @experimental
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of time that was remaining when the lambda returned in order to avoid hitting a timeout.
     *
     * @experimental
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of changes that were not processed due to having an invalid format.
     *
     * @experimental
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    registerAlarms(scope, follower, stager, monitoring, schedule) {
        const failureAlarm = follower.metricErrors().createAlarm(scope, 'NpmJs/Follower/Failures', {
            alarmName: `${scope.node.path}/NpmJs/Follower/Failures`,
            alarmDescription: [
                'The NpmJs follower function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Failures', failureAlarm);
        const notRunningAlarm = follower.metricInvocations({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NotRunning', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NotRunning`,
            alarmDescription: [
                'The NpmJs follower function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Not Running', notRunningAlarm);
        // The period for this alarm needs to match the scheduling interval of the
        // follower, otherwise the metric will be too sparse to properly detect
        // problems.
        const noChangeAlarm = this.metricChangeCount({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NoChanges', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NoChanges`,
            alarmDescription: [
                'The NpmJs follower function is no discovering any changes from CouchDB!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // If the metric is not emitted, it can be assumed to be zero.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addLowSeverityAlarm('Np npmjs.com changes discovered', noChangeAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`,
            alarmDescription: [
                'The NpmJS package stager is failing - its dead letter queue is not empty',
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(stager)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(stager.deadLetterQueue)}`,
                '',
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addLowSeverityAlarm('NpmJs/Stager DLQ Not Empty', dlqNotEmptyAlarm);
        // Finally - the "not running" alarm depends on the schedule (it won't run until the schedule
        // exists!), and the schedule depends on the failure alarm existing (we don't want it to run
        // before we can know it is failing). This means the returned `IDependable` effectively ensures
        // all alarms have been provisionned already! Isn't it nice!
        notRunningAlarm.node.addDependency(schedule);
        schedule.node.addDependency(failureAlarm);
    }
    registerCanary(scope, packageName, visibilitySla, bucket, constructHubBaseUrl, monitoring) {
        const canary = new canary_1.NpmJsPackageCanary(scope, 'Canary', { bucket, constructHubBaseUrl, packageName });
        const alarm = new aws_cloudwatch_1.MathExpression({
            expression: 'MAX([mDwell, mTTC])',
            period: core_1.Duration.minutes(1),
            usingMetrics: {
                mDwell: canary.metricDwellTime(),
                mTTC: canary.metricTimeToCatalog(),
            },
        }).createAlarm(canary, 'Alarm', {
            alarmName: `${canary.node.path}/SLA-Breached`,
            alarmDescription: [
                `New versions of ${packageName} have been published over ${visibilitySla.toHumanString()} ago and are still not visible in construct hub`,
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 2,
            // If there is no data, the canary might not be running, so... *Chuckles* we're in danger!
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
            threshold: visibilitySla.toSeconds(),
        });
        monitoring.addHighSeverityAlarm('New version visibility SLA breached', alarm);
        return [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Package Canary',
                left: [
                    canary.metricDwellTime({ label: 'Dwell Time' }),
                    canary.metricTimeToCatalog({ label: 'Time to Catalog' }),
                ],
                leftAnnotations: [{
                        color: '#ff0000',
                        label: `SLA (${visibilitySla.toHumanString()})`,
                        value: visibilitySla.toSeconds(),
                    }],
                leftYAxis: { min: 0 },
                right: [
                    canary.metricTrackedVersionCount({ label: 'Tracked Version Count' }),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
    }
}
exports.NpmJs = NpmJs;
_a = JSII_RTTI_SYMBOL_1;
NpmJs[_a] = { fqn: "construct-hub.sources.NpmJs", version: "0.3.201" };
//# sourceMappingURL=data:application/json;base64,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