"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.throughVpcEndpoint = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
/**
 * Decorates an S3 Bucket so that grants are made including a VPC endpoint
 * policy.
 *
 * This currently only supports the `gratRead` and `grantWrite` APIs.
 *
 * @param bucket      the bucket to be wrapped.
 * @param vpcEndpoint the VPC Endpoint for S3 to be used.
 *
 * @returns a decorated S3 Bucket.
 */
function throughVpcEndpoint(bucket, vpcEndpoint) {
    return new Proxy(bucket, {
        get(target, property, _receiver) {
            switch (property) {
                case 'grantDelete':
                    return decoratedGrantDelete.bind(target);
                case 'grantRead':
                    return decoratedGrantRead.bind(target);
                case 'grantWrite':
                    return decoratedGrantWrite.bind(target);
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        console.warn(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return target[property];
            }
        },
        getOwnPropertyDescriptor(target, property) {
            const realDescriptor = Object.getOwnPropertyDescriptor(target, property);
            switch (property) {
                case 'grantDelete':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantDelete,
                        get: undefined,
                        set: undefined,
                    };
                case 'grantRead':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantRead,
                        get: undefined,
                        set: undefined,
                    };
                case 'grantWrite':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantWrite,
                        get: undefined,
                        set: undefined,
                    };
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        console.warn(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return realDescriptor;
            }
        },
    });
    function decoratedGrantDelete(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantDelete(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:DeleteObject*'],
                resources: [this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
    function decoratedGrantRead(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantRead(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
    function decoratedGrantWrite(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantWrite(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:Abort*', 's3:DeleteObject*', 's3:PutObject*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
}
exports.throughVpcEndpoint = throughVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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