"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const events_1 = require("events");
const fs_1 = require("fs");
const path = require("path");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const jsii_rosetta_1 = require("jsii-rosetta");
const constants = require("../../../backend/shared/constants");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-rosetta/lib/commands/transliterate');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    transliterator_lambda_1.reset();
    done();
});
test('scoped package', async () => {
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/'));
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', `@${packageScope}`, packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', `@${packageScope}`, packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key
        // The key is not URI-encoded when it's sent to the S3 SDK...
        .split('/').map((comp) => decodeURIComponent(comp)).join('/')
        .replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
test('unscoped package', async () => {
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key);
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key.replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
class MockChildProcess extends events_1.EventEmitter {
    constructor(spawnfile, spawnargs, promise) {
        super();
        this.spawnfile = spawnfile;
        this.spawnargs = spawnargs;
        this.stdin = null;
        this.stdout = null;
        this.stderr = null;
        this.stdio = [this.stdin, this.stdout, this.stderr, null, null];
        this.exitCode = 0;
        this.killed = false;
        this.signalCode = null;
        this.pid = -1; // Obviously fake
        promise.then(() => this.emit('close', this.exitCode, this.signalCode), (err) => this.emit('error', err));
    }
    get connected() {
        throw new Error('Not Implemented');
    }
    disconnect() {
        throw new Error('Not Implemented');
    }
    kill() {
        throw new Error('Not Implemented');
    }
    ref() {
        throw new Error('Not Implemented');
    }
    send() {
        throw new Error('Not Implemented');
    }
    unref() {
        throw new Error('Not Implemented');
    }
}
//# sourceMappingURL=data:application/json;base64,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