"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    console.log('Loading existing catalog...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (!data.Body) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const object of relevantObjects(BUCKET_NAME)) {
            const assemblyKey = object.Key;
            try {
                await appendPackage(packages, assemblyKey, BUCKET_NAME);
            }
            catch (e) {
                failures[assemblyKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        for (const record of event.Records) {
            // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
            // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
            // that would not replace `@` in the position where it is in the keys... So we have to work on
            // the URI components instead.
            const key = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
            // note that we intentionally don't catch errors here to let these
            // event go to the DLQ for manual inspection.
            await appendPackage(packages, key, BUCKET_NAME);
        }
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'text/json',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.ASSEMBLY_KEY_SUFFIX))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, assemblyKey, bucketName) {
    var _a, _b, _c;
    console.log(`Processing key: ${assemblyKey}`);
    const pkgKey = assemblyKey.replace(constants.ASSEMBLY_KEY_SUFFIX, constants.PACKAGE_KEY_SUFFIX);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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