"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogBuilder = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants = require("../shared/constants");
const catalog_builder_1 = require("./catalog-builder");
/**
 * Builds or re-builds the `catalog.json` object in the designated bucket.
 */
class CatalogBuilder extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const handler = new catalog_builder_1.CatalogBuilder(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: `Creates the catalog.json object in ${props.bucket.bucketName}`,
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantReadWrite(handler);
        handler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED],
            filters: [{ prefix: constants.STORAGE_KEY_PREFIX, suffix: constants.ASSEMBLY_KEY_SUFFIX }],
        }));
        props.monitoring.watchful.watchLambdaFunction('Catalog Builder Function', handler);
        this.alarmDeadLetterQueueNotEmpty = handler.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The catalog builder function failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.CatalogBuilder = CatalogBuilder;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYmFja2VuZC9jYXRhbG9nLWJ1aWxkZXIvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNERBQXFFO0FBQ3JFLGdGQUFrRTtBQUNsRSxnREFBa0Q7QUFDbEQsNENBQW9EO0FBQ3BELHdDQUFvRDtBQUdwRCxpREFBaUQ7QUFDakQsdURBQThEO0FBcUI5RDs7R0FFRztBQUNILE1BQWEsY0FBZSxTQUFRLGdCQUFTO0lBUTNDLFlBQW1CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCOztRQUN6RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sT0FBTyxHQUFHLElBQUksZ0NBQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzNDLHNCQUFzQixFQUFFLElBQUk7WUFDNUIsV0FBVyxFQUFFLHNDQUFzQyxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUM1RSxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTthQUNyQztZQUNELFlBQVksUUFBRSxLQUFLLENBQUMsWUFBWSxtQ0FBSSx3QkFBYSxDQUFDLFNBQVM7WUFDM0QsVUFBVSxFQUFFLEtBQU07WUFDbEIsNEJBQTRCLEVBQUUsQ0FBQztZQUMvQixPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFckMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLHdDQUFhLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNyRCxNQUFNLEVBQUUsQ0FBQyxrQkFBUyxDQUFDLGNBQWMsQ0FBQztZQUNsQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUMsa0JBQWtCLEVBQUUsTUFBTSxFQUFFLFNBQVMsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1NBQzNGLENBQUMsQ0FBQyxDQUFDO1FBRUosS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsMEJBQTBCLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDbkYsSUFBSSxDQUFDLDRCQUE0QixHQUFHLE9BQU8sQ0FBQyxlQUFnQixDQUFDLHdDQUF3QyxFQUFFO2FBQ3BHLFdBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzdCLGdCQUFnQixFQUFFLDRDQUE0QztZQUM5RCxrQkFBa0IsRUFBRSxtQ0FBa0IsQ0FBQyxrQ0FBa0M7WUFDekUsaUJBQWlCLEVBQUUsQ0FBQztZQUNwQixTQUFTLEVBQUUsQ0FBQztTQUNiLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDRjtBQXZDRCx3Q0F1Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb21wYXJpc29uT3BlcmF0b3IsIElBbGFybSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IFMzRXZlbnRTb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhLWV2ZW50LXNvdXJjZXMnO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCB7IEJ1Y2tldCwgRXZlbnRUeXBlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHsgTW9uaXRvcmluZyB9IGZyb20gJy4uLy4uL21vbml0b3JpbmcnO1xuaW1wb3J0ICogYXMgY29uc3RhbnRzIGZyb20gJy4uL3NoYXJlZC9jb25zdGFudHMnO1xuaW1wb3J0IHsgQ2F0YWxvZ0J1aWxkZXIgYXMgSGFuZGxlciB9IGZyb20gJy4vY2F0YWxvZy1idWlsZGVyJztcblxuZXhwb3J0IGludGVyZmFjZSBDYXRhbG9nQnVpbGRlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBwYWNrYWdlIHN0b3JlIGJ1Y2tldC5cbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldDogQnVja2V0O1xuXG4gIC8qKlxuICAgKiBUaGUgbW9uaXRvcmluZyBoYW5kbGVyIHRvIHJlZ2lzdGVyIGFsYXJtcyB3aXRoLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9uaXRvcmluZzogTW9uaXRvcmluZztcblxuICAvKipcbiAgICogSG93IGxvbmcgc2hvdWxkIGV4ZWN1dGlvbiBsb2dzIGJlIHJldGFpbmVkP1xuICAgKlxuICAgKiBAZGVmYXVsdCBSZXRlbnRpb25EYXlzLlRFTl9ZRUFSU1xuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogUmV0ZW50aW9uRGF5cztcbn1cblxuLyoqXG4gKiBCdWlsZHMgb3IgcmUtYnVpbGRzIHRoZSBgY2F0YWxvZy5qc29uYCBvYmplY3QgaW4gdGhlIGRlc2lnbmF0ZWQgYnVja2V0LlxuICovXG5leHBvcnQgY2xhc3MgQ2F0YWxvZ0J1aWxkZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogQWxhcm1zIHdoZW4gdGhlIGRlYWQtbGV0dGVyLXF1ZXVlIGFzc29jaWF0ZWQgd2l0aCB0aGUgY2F0YWxvZyBidWlsZGVyXG4gICAqIGZ1bmN0aW9uIGlzIG5vdCBlbXB0eSwgbWVhbmluZyB0aGUgY2F0YWxvZyBidWlsZGVyIGZhaWxlZCB0byBydW4gYW5kXG4gICAqIHJlcXVpcmVzIG9wZXJhdG9yIGF0dGVudGlvbi5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhbGFybURlYWRMZXR0ZXJRdWV1ZU5vdEVtcHR5OiBJQWxhcm07XG5cbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDYXRhbG9nQnVpbGRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgSGFuZGxlcih0aGlzLCAnRGVmYXVsdCcsIHtcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQ6IHRydWUsXG4gICAgICBkZXNjcmlwdGlvbjogYENyZWF0ZXMgdGhlIGNhdGFsb2cuanNvbiBvYmplY3QgaW4gJHtwcm9wcy5idWNrZXQuYnVja2V0TmFtZX1gLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQlVDS0VUX05BTUU6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgfSxcbiAgICAgIGxvZ1JldGVudGlvbjogcHJvcHMubG9nUmV0ZW50aW9uID8/IFJldGVudGlvbkRheXMuVEVOX1lFQVJTLFxuICAgICAgbWVtb3J5U2l6ZTogMTBfMjQwLCAvLyBDdXJyZW50bHkgdGhlIG1heGltdW0gcG9zc2libGUgc2V0dGluZ1xuICAgICAgcmVzZXJ2ZWRDb25jdXJyZW50RXhlY3V0aW9uczogMSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgIH0pO1xuXG4gICAgcHJvcHMuYnVja2V0LmdyYW50UmVhZFdyaXRlKGhhbmRsZXIpO1xuXG4gICAgaGFuZGxlci5hZGRFdmVudFNvdXJjZShuZXcgUzNFdmVudFNvdXJjZShwcm9wcy5idWNrZXQsIHtcbiAgICAgIGV2ZW50czogW0V2ZW50VHlwZS5PQkpFQ1RfQ1JFQVRFRF0sXG4gICAgICBmaWx0ZXJzOiBbeyBwcmVmaXg6IGNvbnN0YW50cy5TVE9SQUdFX0tFWV9QUkVGSVgsIHN1ZmZpeDogY29uc3RhbnRzLkFTU0VNQkxZX0tFWV9TVUZGSVggfV0sXG4gICAgfSkpO1xuXG4gICAgcHJvcHMubW9uaXRvcmluZy53YXRjaGZ1bC53YXRjaExhbWJkYUZ1bmN0aW9uKCdDYXRhbG9nIEJ1aWxkZXIgRnVuY3Rpb24nLCBoYW5kbGVyKTtcbiAgICB0aGlzLmFsYXJtRGVhZExldHRlclF1ZXVlTm90RW1wdHkgPSBoYW5kbGVyLmRlYWRMZXR0ZXJRdWV1ZSEubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSgpXG4gICAgICAuY3JlYXRlQWxhcm0odGhpcywgJ0RMUUFsYXJtJywge1xuICAgICAgICBhbGFybURlc2NyaXB0aW9uOiAnVGhlIGNhdGFsb2cgYnVpbGRlciBmdW5jdGlvbiBmYWlsZWQgdG8gcnVuJyxcbiAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgIH0pO1xuICB9XG59XG4iXX0=