"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(15);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 2 minutes ahead of the timeout.
        const lambda = new discovery_1.Discovery(this, 'Default', {
            description: 'Periodically query npm.js index for new construct libraries',
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
        });
        this.bucket.grantReadWrite(lambda);
        props.queue.grantSendMessages(lambda);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(lambda)],
        });
        props.monitoring.watchful.watchLambdaFunction('Discovery Function', lambda);
        this.alarmErrors = lambda.metricErrors({ period: core_1.Duration.minutes(15) }).createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = lambda.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) is not running as scheduled',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
        });
    }
    /**
     * The total count of staging failures.
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRIC_NAMESPACE,
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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