"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const ingestion_1 = require("./ingestion");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: 'Ingests new package versions into the Construct Hub',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
            },
            memorySize: 10240,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantWrite(handler);
        handler.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        this.grantPrincipal = handler.grantPrincipal;
        props.monitoring.watchful.watchLambdaFunction('Ingestion Function', handler);
        this.alarmDeadLetterQueueNotEmpty = handler.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The ingestion function failed for one or more packages',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.Ingestion = Ingestion;
//# sourceMappingURL=data:application/json;base64,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