"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const path_1 = require("path");
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
async function handler(event, context) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        const { dotJsii, licenseText } = await new Promise((ok, ko) => {
            let dotJsiiBuffer;
            let licenseTextBuffer;
            tar_stream_1.extract()
                .on('entry', (headers, stream, next) => {
                const chunks = new Array();
                if (headers.name === 'package/.jsii') {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        dotJsiiBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                else if (isLicenseFile(headers.name)) {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        licenseTextBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                // Skip on next runLoop iteration so we avoid filling the stack.
                return setImmediate(next);
            })
                .once('error', ko)
                .once('close', () => {
                if (dotJsiiBuffer == null) {
                    ko(new Error('No .jsii file found in tarball!'));
                }
                else {
                    ok({ dotJsii: dotJsiiBuffer, licenseText: licenseTextBuffer });
                }
            })
                .write(tar, (err) => {
                if (err != null) {
                    ko(err);
                }
            });
        });
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        const { license, name: packageName, version: packageVersion } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
        // Re-check that the license in `.jsii` is eligible. We don't blindly expect it matches that in package.json!
        if (!constants.ELIGIBLE_LICENSES.has(license.toUpperCase())) {
            console.log(`Ignoring package with ineligible license (SPDX identifier "${license}")`);
            continue;
        }
        await aws_embedded_metrics_1.metricScope((metrics) => () => {
            metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        })();
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const metadataKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.METADATA_KEY_SUFFIX}`;
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        result.push(created);
    }
    return result;
}
exports.handler = handler;
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
//# sourceMappingURL=data:application/json;base64,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