"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const lambda = new transliterator_1.Transliterator(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: 'Creates transliterated assemblies from jsii-enabled npm packages',
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(15),
        });
        // The handler reads & writes to this bucket.
        props.bucket.grantRead(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        props.bucket.grantWrite(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.assemblyKeySuffix('*')}`);
        // Creating the event chaining
        lambda.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED],
            filters: [{ prefix: constants.STORAGE_KEY_PREFIX, suffix: constants.PACKAGE_KEY_SUFFIX }],
        }));
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
        this.alarmDeadLetterQueueNotEmpty = lambda.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The transliteration function failed for one or more packages',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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