"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const config_file_1 = require("../../config-file");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const configFilename = 'config.json';
        const config = new config_file_1.ConfigFile(configFilename, JSON.stringify({
            packageLinks: (_a = props.packageLinks) !== null && _a !== void 0 ? _a : [],
            packageTags: (_b = props.packageTags) !== null && _b !== void 0 ? _b : [],
        }));
        const configBucket = new aws_s3_1.Bucket(this, 'ConfigBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'DeployIngestionConfiguration', {
            sources: [aws_s3_deployment_1.Source.asset(config.dir)],
            destinationBucket: configBucket,
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
                CONFIG_BUCKET_NAME: configBucket.bucketName,
                CONFIG_FILE_KEY: configFilename,
            },
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        configBucket.grantRead(handler);
        props.bucket.grantWrite(this.function);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a NextContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                MaxKeys: 250,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        listBucket.next(process.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), listBucket)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done'))));
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(stateMachine);
        props.queue.grantSendMessages(stateMachine);
    }
}
//# sourceMappingURL=data:application/json;base64,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