"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const assembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            constructFramework = detectConstructFramework(assembly);
            const { license, name, version } = assembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public, max-age=86400, must-revalidate, s-maxage=300, proxy-revalidate',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public, max-age: 86400, must-revalidate, s-maxage=300, proxy-revalidate',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf' || packageName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5nZXN0aW9uLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2luZ2VzdGlvbi9pbmdlc3Rpb24ubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1DQUFvQztBQUNwQywrQkFBeUM7QUFDekMsNkJBQTBCO0FBRTFCLHFDQUF3RDtBQUN4RCwrREFBd0U7QUFFeEUsbUNBQW9DO0FBSXBDLG1EQUFtRDtBQUNuRCxpREFBaUQ7QUFDakQsbUVBQXlEO0FBRXpELCtFQUE4RDtBQUM5RCx5Q0FBaUQ7QUFDakQsMkVBQWlFO0FBQ2pFLDJDQUE0RDtBQUU1RCxvQ0FBYSxDQUFDLFNBQVMsR0FBRyw2QkFBaUIsQ0FBQztBQUUvQixRQUFBLE9BQU8sR0FBRyxrQ0FBVyxDQUNoQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQWUsRUFBRSxPQUFnQixFQUFFLEVBQUU7O0lBQ3ZELE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRXhELHdEQUF3RDtJQUN4RCxPQUFPLENBQUMsYUFBYSxFQUFFLENBQUM7SUFFeEIsTUFBTSxXQUFXLEdBQUcsOEJBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM5QyxNQUFNLGlCQUFpQixHQUFHLDhCQUFVLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUMxRCxNQUFNLGtCQUFrQixHQUFHLDhCQUFVLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUM1RCxNQUFNLGVBQWUsR0FBRyw4QkFBVSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFFdEQscUJBQXFCO0lBQ3JCLE1BQU0sRUFBRSxXQUFXLEVBQUUsaUJBQWlCLEVBQUUsWUFBWSxFQUFFLFlBQVksRUFBRSxHQUFXLE1BQU0sU0FBUyxDQUFDLGtCQUFrQixFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBRXBJLE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7SUFFbkMsS0FBSyxNQUFNLE1BQU0sVUFBSSxLQUFLLENBQUMsT0FBTyxtQ0FBSSxFQUFFLEVBQUU7UUFDeEMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFtQixDQUFDO1FBRTFELE1BQU0sVUFBVSxHQUFHLElBQUksU0FBRyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvQyxJQUFJLFVBQVUsQ0FBQyxRQUFRLEtBQUssS0FBSyxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0NBQWdDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDL0Q7UUFDRCxNQUFNLE9BQU8sR0FBRyxNQUFNLEdBQUc7YUFDdEIsRUFBRSxFQUFFO2FBQ0osU0FBUyxDQUFDO1lBQ1QsaUdBQWlHO1lBQ2pHLE1BQU0sRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDckMsb0VBQW9FO1lBQ3BFLEdBQUcsRUFBRSxVQUFVLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDO1lBQzNDLFNBQVMsUUFBRSxVQUFVLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsbUNBQUksU0FBUztTQUNqRSxDQUFDO2FBQ0QsT0FBTyxFQUFFLENBQUM7UUFFYixNQUFNLEVBQUUsU0FBUyxFQUFFLGNBQWMsRUFBRSxHQUFHLG1DQUFTLENBQUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUssQ0FBQyxDQUFDLENBQUM7UUFDckYsSUFBSSxPQUFPLENBQUMsU0FBUyxLQUFLLGNBQWMsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUNiLDJCQUEyQixPQUFPLENBQUMsU0FBUyxRQUFRLGNBQWMsRUFBRSxDQUNyRSxDQUFDO1NBQ0g7UUFFRCxJQUFJLE9BQWUsQ0FBQztRQUNwQixJQUFJLFdBQW1CLENBQUM7UUFDeEIsSUFBSSxXQUErQixDQUFDO1FBQ3BDLElBQUk7WUFDRixDQUFDLEVBQUUsT0FBTyxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsR0FBRyxNQUFNLHNDQUFjLENBQzNELE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUssQ0FBQyxFQUMxQjtnQkFDRSxPQUFPLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUU7Z0JBQ2xELFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxzQkFBc0IsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFO2dCQUM3RCxXQUFXLEVBQUUsRUFBRSxNQUFNLEVBQUUsYUFBYSxFQUFFO2FBQ3ZDLENBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFBQyxPQUFPLEdBQUcsRUFBRTtZQUNaLE9BQU8sQ0FBQyxLQUFLLENBQUMsNEJBQTRCLEdBQUcsRUFBRSxDQUFDLENBQUM7WUFDakQsT0FBTyxDQUFDLFNBQVMseUNBQTZCLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzdELE9BQU87U0FDUjtRQUVELElBQUksa0JBQWtELENBQUM7UUFDdkQsSUFBSSxjQUFzQixDQUFDO1FBQzNCLElBQUksV0FBbUIsQ0FBQztRQUN4QixJQUFJLGNBQXNCLENBQUM7UUFDM0IsSUFBSTtZQUNGLE1BQU0sUUFBUSxHQUFHLHVCQUFnQixDQUMvQixJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FDdEMsQ0FBQztZQUNGLGtCQUFrQixHQUFHLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3hELE1BQU0sRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxHQUFHLFFBQVEsQ0FBQztZQUM1QyxjQUFjLEdBQUcsT0FBTyxDQUFDO1lBQ3pCLFdBQVcsR0FBRyxJQUFJLENBQUM7WUFDbkIsY0FBYyxHQUFHLE9BQU8sQ0FBQztZQUN6QixPQUFPLENBQUMsU0FBUywyQ0FBOEIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDL0Q7UUFBQyxPQUFPLEVBQUUsRUFBRTtZQUNYLE9BQU8sQ0FBQyxLQUFLLENBQ1gsMERBQTBELEVBQUUsRUFBRSxDQUMvRCxDQUFDO1lBQ0YsT0FBTyxDQUFDLFNBQVMsMkNBQThCLENBQUMsRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzlELE9BQU87U0FDUjtRQUVELG9GQUFvRjtRQUNwRixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUNqRSxNQUFNLEVBQ0osSUFBSSxFQUFFLGVBQWUsRUFDckIsT0FBTyxFQUFFLGtCQUFrQixFQUMzQixPQUFPLEVBQUUsa0JBQWtCLEVBQzNCLFlBQVksR0FDYixHQUFHLGNBQWMsQ0FBQztRQUNuQixJQUNFLGVBQWUsS0FBSyxXQUFXO1lBQy9CLGtCQUFrQixLQUFLLGNBQWM7WUFDckMsa0JBQWtCLEtBQUssY0FBYyxFQUNyQztZQUNBLE9BQU8sQ0FBQyxHQUFHLENBQ1QsbUVBQW1FLGVBQWUsSUFBSSxrQkFBa0IsT0FBTyxrQkFBa0IsUUFBUSxXQUFXLElBQUksY0FBYyxPQUFPLGNBQWMsR0FBRyxDQUMvTCxDQUFDO1lBQ0YsT0FBTyxDQUFDLFNBQVMsc0VBRWYsQ0FBQyxFQUNELDJCQUFJLENBQUMsS0FBSyxDQUNYLENBQUM7WUFDRixTQUFTO1NBQ1Y7UUFDRCxPQUFPLENBQUMsU0FBUyxzRUFFZixDQUFDLEVBQ0QsMkJBQUksQ0FBQyxLQUFLLENBQ1gsQ0FBQztRQUVGLHlDQUF5QztRQUN6QyxPQUFPLENBQUMsU0FBUyw4Q0FFZixXQUFXLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFDM0IsMkJBQUksQ0FBQyxLQUFLLENBQ1gsQ0FBQztRQUVGLE1BQU0sWUFBWSxHQUFHLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxTQUFTLEVBQUUsY0FBYyxFQUFFLEVBQUUsRUFBRTtZQUNoRixNQUFNLFFBQVEsR0FBRyxZQUFZLGFBQVosWUFBWSx1QkFBWixZQUFZLENBQUUsWUFBWSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBRXZELElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ2IsT0FBTyxLQUFLLENBQUM7YUFDZDtZQUVELDRDQUE0QztZQUM1QyxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUM5QixJQUFJLENBQUEsY0FBYyxhQUFkLGNBQWMsdUJBQWQsY0FBYyxDQUFFLE1BQU0sS0FBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFO2dCQUNoRSxPQUFPLEtBQUssQ0FBQzthQUNkO1lBRUQsK0JBQStCO1lBQy9CLE9BQU8sRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxDQUFDO1FBQzdDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVQLE1BQU0sV0FBVyxHQUFHLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQWlELEVBQUUsU0FBUyxFQUFFLEVBQUU7WUFDNUcsTUFBTSxFQUFFLFNBQVMsRUFBRSxHQUFHLE9BQU8sRUFBRSxHQUFHLFNBQVMsQ0FBQztZQUM1QyxJQUFJLHNCQUFlLENBQUMsU0FBUyxFQUFFLGNBQWMsQ0FBQyxFQUFFO2dCQUM5QyxPQUFPLENBQUMsR0FBRyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7YUFDNUI7WUFFRCxPQUFPLEtBQUssQ0FBQztRQUNmLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVQLE1BQU0sUUFBUSxHQUFHO1lBQ2Ysa0JBQWtCO1lBQ2xCLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtZQUNsQixXQUFXLEVBQUUsV0FBVyxhQUFYLFdBQVcsdUJBQVgsV0FBVyxDQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUM7WUFDM0MsWUFBWTtZQUNaLFdBQVc7U0FDWixDQUFDO1FBRUYsTUFBTSxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLEdBQUcsU0FBUyxDQUFDLGFBQWEsQ0FDdEUsV0FBVyxFQUNYLGNBQWMsQ0FDZixDQUFDO1FBQ0YsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUNsRCxPQUFPLENBQUMsR0FBRyxDQUFDLHVCQUF1QixVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sQ0FBQyxHQUFHLENBQUMsd0JBQXdCLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFbkQsMkVBQTJFO1FBQzNFLG1EQUFtRDtRQUNuRCxPQUFPLENBQUMsR0FBRyxDQUNULEdBQUcsV0FBVyxJQUFJLGNBQWMseUNBQXlDLENBQzFFLENBQUM7UUFDRixNQUFNLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUM5QyxHQUFHO2lCQUNBLEVBQUUsRUFBRTtpQkFDSixTQUFTLENBQUM7Z0JBQ1QsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxVQUFVO2dCQUNmLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsWUFBWSxFQUFFLHdFQUF3RTtnQkFDdEYsV0FBVyxFQUFFLDBCQUEwQjtnQkFDdkMsUUFBUSxFQUFFO29CQUNSLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxZQUFZO29CQUN4QyxtQkFBbUIsRUFBRSxPQUFPLENBQUMsYUFBYTtvQkFDMUMsZUFBZSxFQUFFLE9BQU8sQ0FBQyxZQUFZO2lCQUN0QzthQUNGLENBQUM7aUJBQ0QsT0FBTyxFQUFFO1lBQ1osR0FBRztpQkFDQSxFQUFFLEVBQUU7aUJBQ0osU0FBUyxDQUFDO2dCQUNULE1BQU0sRUFBRSxXQUFXO2dCQUNuQixHQUFHLEVBQUUsV0FBVztnQkFDaEIsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDO2dCQUM5QixZQUFZLEVBQUUsd0RBQXdEO2dCQUN0RSxXQUFXLEVBQUUsa0JBQWtCO2dCQUMvQixRQUFRLEVBQUU7b0JBQ1Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLFlBQVk7b0JBQ3hDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxhQUFhO29CQUMxQyxlQUFlLEVBQUUsT0FBTyxDQUFDLFlBQVk7aUJBQ3RDO2FBQ0YsQ0FBQztpQkFDRCxPQUFPLEVBQUU7U0FDYixDQUFDLENBQUM7UUFFSCxrQ0FBa0M7UUFDbEMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLFdBQVcsSUFBSSxjQUFjLDRCQUE0QixDQUFDLENBQUM7UUFDMUUsTUFBTSxRQUFRLEdBQUcsTUFBTSxHQUFHO2FBQ3ZCLEVBQUUsRUFBRTthQUNKLFNBQVMsQ0FBQztZQUNULE1BQU0sRUFBRSxXQUFXO1lBQ25CLEdBQUcsRUFBRSxXQUFXO1lBQ2hCLElBQUksRUFBRSxPQUFPO1lBQ2IsWUFBWSxFQUFFLHlFQUF5RTtZQUN2RixXQUFXLEVBQUUsa0JBQWtCO1lBQy9CLFFBQVEsRUFBRTtnQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTthQUN0QztTQUNGLENBQUM7YUFDRCxPQUFPLEVBQUUsQ0FBQztRQUViLE1BQU0sT0FBTyxHQUFzQjtZQUNqQyxNQUFNLEVBQUUsV0FBVztZQUNuQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUzthQUM5QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxHQUFHLEVBQUUsVUFBVTtnQkFDZixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7YUFDekI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxjQUFjLENBQUMsU0FBUzthQUNwQztTQUNGLENBQUM7UUFDRixPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sR0FBRyxHQUFHLE1BQU0sR0FBRzthQUNsQixhQUFhLEVBQUU7YUFDZixjQUFjLENBQUM7WUFDZCxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7WUFDOUIsSUFBSSxFQUFFLHlCQUF5QixDQUM3QixXQUFXLEVBQ1gsSUFBSSxjQUFjLEVBQUUsRUFDcEIsT0FBTyxDQUFDLFlBQVksQ0FDckI7WUFDRCxlQUFlLEVBQUUsaUJBQWlCO1NBQ25DLENBQUM7YUFDRCxPQUFPLEVBQUUsQ0FBQztRQUNiLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUNBQW1DLEdBQUcsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxDQUFDO0tBQy9CO0lBRUQsT0FBTyxNQUFNLENBQUM7QUFDaEIsQ0FBQyxDQUNGLENBQUM7QUFxQkY7Ozs7Ozs7R0FPRztBQUNILFNBQVMsd0JBQXdCLENBQUMsUUFBa0I7O0lBQ2xELElBQUksSUFBNEMsQ0FBQztJQUNqRCxJQUFJLGFBQWEsR0FBRyxLQUFLLENBQUM7SUFDMUIsSUFBSSxZQUFnQyxDQUFDO0lBQ3JDLElBQUkscUJBQXFCLEdBQUcsS0FBSyxDQUFDO0lBQ2xDLCtCQUErQixDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ2pFLEtBQUssTUFBTSxPQUFPLElBQUksTUFBTSxDQUFDLElBQUksT0FBQyxRQUFRLENBQUMsaUJBQWlCLG1DQUFJLEVBQUUsQ0FBQyxFQUFFO1FBQ25FLCtCQUErQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3pDLElBQUksYUFBYSxFQUFFO1lBQ2pCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO0tBQ0Y7SUFDRCxPQUFPLElBQUksSUFBSSxFQUFFLElBQUksRUFBRSxZQUFZLEVBQUUscUJBQXFCLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7SUFFeEYsU0FBUywrQkFBK0IsQ0FBQyxXQUFtQixFQUFFLFlBQW1EOztxQ0FBbkQsRUFBQSxxQkFBZSxRQUFRLENBQUMsWUFBWSwwQ0FBRyxXQUFXLENBQUM7UUFDL0csSUFBSSxXQUFXLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFdBQVcsS0FBSyxhQUFhLElBQUksV0FBVyxLQUFLLFNBQVMsRUFBRTtZQUNyRyxJQUFJLElBQUksSUFBSSxJQUFJLDRCQUFtQyxFQUFFO2dCQUNuRCw0REFBNEQ7Z0JBQzVELGFBQWEsR0FBRyxJQUFJLENBQUM7Z0JBQ3JCLE9BQU87YUFDUjtZQUNELElBQUksMEJBQWlDLENBQUM7U0FDdkM7YUFBTSxJQUFJLFdBQVcsS0FBSyxPQUFPLElBQUksV0FBVyxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUN2RSxJQUFJLElBQUksSUFBSSxJQUFJLHdCQUFpQyxFQUFFO2dCQUNqRCw0REFBNEQ7Z0JBQzVELGFBQWEsR0FBRyxJQUFJLENBQUM7Z0JBQ3JCLE9BQU87YUFDUjtZQUNELElBQUksc0JBQStCLENBQUM7U0FDckM7YUFBTSxJQUFJLFdBQVcsS0FBSyxPQUFPLElBQUksbUNBQW1DLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQzNGLElBQUksSUFBSSxJQUFJLElBQUksd0JBQWlDLEVBQUU7Z0JBQ2pELDREQUE0RDtnQkFDNUQsYUFBYSxHQUFHLElBQUksQ0FBQztnQkFDckIsT0FBTzthQUNSO1lBQ0QsSUFBSSxzQkFBK0IsQ0FBQztTQUNyQzthQUFNO1lBQ0wsT0FBTztTQUNSO1FBQ0QsSUFBSSxZQUFZLEVBQUU7WUFDaEIsTUFBTSxLQUFLLFNBQUcsbUJBQVUsQ0FBQyxZQUFZLENBQUMsMENBQUUsS0FBSyxDQUFDO1lBQzlDLElBQUksWUFBWSxJQUFJLElBQUksSUFBSSxZQUFZLEtBQUssS0FBSyxFQUFFO2dCQUNsRCwwREFBMEQ7Z0JBQzFELHFCQUFxQixHQUFHLElBQUksQ0FBQzthQUM5QjtZQUNELFlBQVksR0FBRyxLQUFLLENBQUM7U0FDdEI7UUFDRCxPQUFPO0lBQ1QsQ0FBQztBQUNILENBQUM7QUFFRDs7Ozs7OztHQU9HO0FBQ0gsU0FBUyxhQUFhLENBQUMsUUFBZ0I7SUFDckMsTUFBTSxHQUFHLEdBQUcsY0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzlCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7SUFDeEQsT0FBTyxDQUNMLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDekMsZUFBUSxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsS0FBSyxTQUFTLENBQ3BELENBQUM7QUFDSixDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsU0FBUyx5QkFBeUIsQ0FDaEMsS0FBYSxFQUNiLEdBQUcsSUFBdUI7SUFFMUIsTUFBTSxLQUFLLEdBQUcsQ0FBQyxLQUFLLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUMvQixNQUFNLElBQUksR0FBRyxLQUFLO1NBQ2YsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBQ2xELElBQUksQ0FBQyxHQUFHLENBQUM7U0FDVCxPQUFPLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztTQUNsQixPQUFPLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzFCLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxFQUFFLEVBQUU7UUFDckIsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUNELE1BQU0sTUFBTSxHQUFHLG1CQUFVLENBQUMsUUFBUSxDQUFDO1FBQ2pDLHFFQUFxRTtTQUNwRSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN2QixNQUFNLENBQUMsS0FBSyxDQUFDO1NBQ2IsU0FBUyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNuQixPQUFPLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxHQUFHLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLElBQUksTUFBTSxFQUFFLENBQUM7QUFDbEUsQ0FBQztBQVVEOzs7R0FHRztBQUNILEtBQUssVUFBVSxTQUFTLENBQUMsTUFBYyxFQUFFLEdBQVc7O0lBQ2xELE1BQU0sYUFBYSxHQUFHO1FBQ3BCLFdBQVcsRUFBRSxFQUFFO1FBQ2YsWUFBWSxFQUFFLEVBQUU7S0FDakIsQ0FBQztJQUNGLElBQUk7UUFDRixNQUFNLEdBQUcsR0FBRyxNQUFNLEdBQUcsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUM7WUFDbkMsTUFBTSxFQUFFLE1BQU07WUFDZCxHQUFHLEVBQUUsR0FBRztTQUNULENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNiLE1BQU0sSUFBSSxTQUFHLEdBQUcsYUFBSCxHQUFHLHVCQUFILEdBQUcsQ0FBRSxJQUFJLDBDQUFFLFFBQVEsRUFBRSxDQUFDO1FBQ25DLElBQUksSUFBSSxFQUFFO1lBQ1IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ3pCO1FBQ0QsT0FBTyxhQUFhLENBQUM7S0FDdEI7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakIsT0FBTyxhQUFhLENBQUM7S0FDdEI7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY3JlYXRlSGFzaCB9IGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgeyBiYXNlbmFtZSwgZXh0bmFtZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcblxuaW1wb3J0IHsgQXNzZW1ibHksIHZhbGlkYXRlQXNzZW1ibHkgfSBmcm9tICdAanNpaS9zcGVjJztcbmltcG9ydCB7IG1ldHJpY1Njb3BlLCBDb25maWd1cmF0aW9uLCBVbml0IH0gZnJvbSAnYXdzLWVtYmVkZGVkLW1ldHJpY3MnO1xuaW1wb3J0IHR5cGUgeyBDb250ZXh0LCBTUVNFdmVudCB9IGZyb20gJ2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgbWluVmVyc2lvbiB9IGZyb20gJ3NlbXZlcic7XG5pbXBvcnQgdHlwZSB7IFBhY2thZ2VUYWdDb25maWcgfSBmcm9tICcuLi8uLi9wYWNrYWdlLXRhZyc7XG5pbXBvcnQgdHlwZSB7IFBhY2thZ2VMaW5rQ29uZmlnIH0gZnJvbSAnLi4vLi4vd2ViYXBwJztcbmltcG9ydCB0eXBlIHsgU3RhdGVNYWNoaW5lSW5wdXQgfSBmcm9tICcuLi9wYXlsb2FkLXNjaGVtYSc7XG5pbXBvcnQgKiBhcyBhd3MgZnJvbSAnLi4vc2hhcmVkL2F3cy5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCAqIGFzIGNvbnN0YW50cyBmcm9tICcuLi9zaGFyZWQvY29uc3RhbnRzJztcbmltcG9ydCB7IHJlcXVpcmVFbnYgfSBmcm9tICcuLi9zaGFyZWQvZW52LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgSW5nZXN0aW9uSW5wdXQgfSBmcm9tICcuLi9zaGFyZWQvaW5nZXN0aW9uLWlucHV0LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgaW50ZWdyaXR5IH0gZnJvbSAnLi4vc2hhcmVkL2ludGVncml0eS5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IGlzVGFnQXBwbGljYWJsZSB9IGZyb20gJy4uL3NoYXJlZC90YWdzJztcbmltcG9ydCB7IGV4dHJhY3RPYmplY3RzIH0gZnJvbSAnLi4vc2hhcmVkL3RhcmJhbGwubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBNZXRyaWNOYW1lLCBNRVRSSUNTX05BTUVTUEFDRSB9IGZyb20gJy4vY29uc3RhbnRzJztcblxuQ29uZmlndXJhdGlvbi5uYW1lc3BhY2UgPSBNRVRSSUNTX05BTUVTUEFDRTtcblxuZXhwb3J0IGNvbnN0IGhhbmRsZXIgPSBtZXRyaWNTY29wZShcbiAgKG1ldHJpY3MpID0+IGFzeW5jIChldmVudDogU1FTRXZlbnQsIGNvbnRleHQ6IENvbnRleHQpID0+IHtcbiAgICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gICAgLy8gQ2xlYXIgb3V0IHRoZSBkZWZhdWx0IGRpbWVuc2lvbnMsIHdlIHdvbid0IG5lZWQgdGhlbS5cbiAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcblxuICAgIGNvbnN0IEJVQ0tFVF9OQU1FID0gcmVxdWlyZUVudignQlVDS0VUX05BTUUnKTtcbiAgICBjb25zdCBTVEFURV9NQUNISU5FX0FSTiA9IHJlcXVpcmVFbnYoJ1NUQVRFX01BQ0hJTkVfQVJOJyk7XG4gICAgY29uc3QgQ09ORklHX0JVQ0tFVF9OQU1FID0gcmVxdWlyZUVudignQ09ORklHX0JVQ0tFVF9OQU1FJyk7XG4gICAgY29uc3QgQ09ORklHX0ZJTEVfS0VZID0gcmVxdWlyZUVudignQ09ORklHX0ZJTEVfS0VZJyk7XG5cbiAgICAvLyBMb2FkIGNvbmZpZ3VyYXRpb25cbiAgICBjb25zdCB7IHBhY2thZ2VUYWdzOiBwYWNrYWdlVGFnc0NvbmZpZywgcGFja2FnZUxpbmtzOiBhbGxvd2VkTGlua3MgfTogQ29uZmlnID0gYXdhaXQgZ2V0Q29uZmlnKENPTkZJR19CVUNLRVRfTkFNRSwgQ09ORklHX0ZJTEVfS0VZKTtcblxuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG5cbiAgICBmb3IgKGNvbnN0IHJlY29yZCBvZiBldmVudC5SZWNvcmRzID8/IFtdKSB7XG4gICAgICBjb25zdCBwYXlsb2FkID0gSlNPTi5wYXJzZShyZWNvcmQuYm9keSkgYXMgSW5nZXN0aW9uSW5wdXQ7XG5cbiAgICAgIGNvbnN0IHRhcmJhbGxVcmkgPSBuZXcgVVJMKHBheWxvYWQudGFyYmFsbFVyaSk7XG4gICAgICBpZiAodGFyYmFsbFVyaS5wcm90b2NvbCAhPT0gJ3MzOicpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBwcm90b2NvbCBpbiBVUkk6ICR7dGFyYmFsbFVyaX1gKTtcbiAgICAgIH1cbiAgICAgIGNvbnN0IHRhcmJhbGwgPSBhd2FpdCBhd3NcbiAgICAgICAgLnMzKClcbiAgICAgICAgLmdldE9iamVjdCh7XG4gICAgICAgICAgLy8gTm90ZTogd2UgZHJvcCBhbnl0aGluZyBhZnRlciB0aGUgZmlyc3QgYC5gIGluIHRoZSBob3N0LCBhcyB3ZSBvbmx5IGNhcmUgYWJvdXQgdGhlIGJ1Y2tldCBuYW1lLlxuICAgICAgICAgIEJ1Y2tldDogdGFyYmFsbFVyaS5ob3N0LnNwbGl0KCcuJylbMF0sXG4gICAgICAgICAgLy8gTm90ZTogdGhlIHBhdGhuYW1lIHBhcnQgaXMgYWJzb2x1dGUsIHNvIHdlIHN0cmlwIHRoZSBsZWFkaW5nIGAvYC5cbiAgICAgICAgICBLZXk6IHRhcmJhbGxVcmkucGF0aG5hbWUucmVwbGFjZSgvXlxcLy8sICcnKSxcbiAgICAgICAgICBWZXJzaW9uSWQ6IHRhcmJhbGxVcmkuc2VhcmNoUGFyYW1zLmdldCgndmVyc2lvbklkJykgPz8gdW5kZWZpbmVkLFxuICAgICAgICB9KVxuICAgICAgICAucHJvbWlzZSgpO1xuXG4gICAgICBjb25zdCB7IGludGVncml0eTogaW50ZWdyaXR5Q2hlY2sgfSA9IGludGVncml0eShwYXlsb2FkLCBCdWZmZXIuZnJvbSh0YXJiYWxsLkJvZHkhKSk7XG4gICAgICBpZiAocGF5bG9hZC5pbnRlZ3JpdHkgIT09IGludGVncml0eUNoZWNrKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgSW50ZWdyaXR5IGNoZWNrIGZhaWxlZDogJHtwYXlsb2FkLmludGVncml0eX0gIT09ICR7aW50ZWdyaXR5Q2hlY2t9YCxcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgbGV0IGRvdEpzaWk6IEJ1ZmZlcjtcbiAgICAgIGxldCBwYWNrYWdlSnNvbjogQnVmZmVyO1xuICAgICAgbGV0IGxpY2Vuc2VUZXh0OiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgICB0cnkge1xuICAgICAgICAoeyBkb3RKc2lpLCBwYWNrYWdlSnNvbiwgbGljZW5zZVRleHQgfSA9IGF3YWl0IGV4dHJhY3RPYmplY3RzKFxuICAgICAgICAgIEJ1ZmZlci5mcm9tKHRhcmJhbGwuQm9keSEpLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGRvdEpzaWk6IHsgcGF0aDogJ3BhY2thZ2UvLmpzaWknLCByZXF1aXJlZDogdHJ1ZSB9LFxuICAgICAgICAgICAgcGFja2FnZUpzb246IHsgcGF0aDogJ3BhY2thZ2UvcGFja2FnZS5qc29uJywgcmVxdWlyZWQ6IHRydWUgfSxcbiAgICAgICAgICAgIGxpY2Vuc2VUZXh0OiB7IGZpbHRlcjogaXNMaWNlbnNlRmlsZSB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICkpO1xuICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYEludmFsaWQgdGFyYmFsbCBjb250ZW50OiAke2Vycn1gKTtcbiAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX1RBUkJBTEwsIDEsIFVuaXQuQ291bnQpO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIGxldCBjb25zdHJ1Y3RGcmFtZXdvcms6IENvbnN0cnVjdEZyYW1ld29yayB8IHVuZGVmaW5lZDtcbiAgICAgIGxldCBwYWNrYWdlTGljZW5zZTogc3RyaW5nO1xuICAgICAgbGV0IHBhY2thZ2VOYW1lOiBzdHJpbmc7XG4gICAgICBsZXQgcGFja2FnZVZlcnNpb246IHN0cmluZztcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGFzc2VtYmx5ID0gdmFsaWRhdGVBc3NlbWJseShcbiAgICAgICAgICBKU09OLnBhcnNlKGRvdEpzaWkudG9TdHJpbmcoJ3V0Zi04JykpLFxuICAgICAgICApO1xuICAgICAgICBjb25zdHJ1Y3RGcmFtZXdvcmsgPSBkZXRlY3RDb25zdHJ1Y3RGcmFtZXdvcmsoYXNzZW1ibHkpO1xuICAgICAgICBjb25zdCB7IGxpY2Vuc2UsIG5hbWUsIHZlcnNpb24gfSA9IGFzc2VtYmx5O1xuICAgICAgICBwYWNrYWdlTGljZW5zZSA9IGxpY2Vuc2U7XG4gICAgICAgIHBhY2thZ2VOYW1lID0gbmFtZTtcbiAgICAgICAgcGFja2FnZVZlcnNpb24gPSB2ZXJzaW9uO1xuICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLklOVkFMSURfQVNTRU1CTFksIDAsIFVuaXQuQ291bnQpO1xuICAgICAgfSBjYXRjaCAoZXgpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihcbiAgICAgICAgICBgUGFja2FnZSBkb2VzIG5vdCBjb250YWluIGEgdmFsaWQgYXNzZW1ibHkgLS0gaWdub3Jpbmc6ICR7ZXh9YCxcbiAgICAgICAgKTtcbiAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX0FTU0VNQkxZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICAvLyBFbnN1cmUgdGhlIGAuanNpaWAgbmFtZSwgdmVyc2lvbiAmIGxpY2Vuc2UgY29ycmVzcG9uZHMgdG8gdGhvc2UgaW4gYHBhY2thZ2UuanNvbmBcbiAgICAgIGNvbnN0IHBhY2thZ2VKc29uT2JqID0gSlNPTi5wYXJzZShwYWNrYWdlSnNvbi50b1N0cmluZygndXRmLTgnKSk7XG4gICAgICBjb25zdCB7XG4gICAgICAgIG5hbWU6IHBhY2thZ2VKc29uTmFtZSxcbiAgICAgICAgdmVyc2lvbjogcGFja2FnZUpzb25WZXJzaW9uLFxuICAgICAgICBsaWNlbnNlOiBwYWNrYWdlSnNvbkxpY2Vuc2UsXG4gICAgICAgIGNvbnN0cnVjdEh1YixcbiAgICAgIH0gPSBwYWNrYWdlSnNvbk9iajtcbiAgICAgIGlmIChcbiAgICAgICAgcGFja2FnZUpzb25OYW1lICE9PSBwYWNrYWdlTmFtZSB8fFxuICAgICAgICBwYWNrYWdlSnNvblZlcnNpb24gIT09IHBhY2thZ2VWZXJzaW9uIHx8XG4gICAgICAgIHBhY2thZ2VKc29uTGljZW5zZSAhPT0gcGFja2FnZUxpY2Vuc2VcbiAgICAgICkge1xuICAgICAgICBjb25zb2xlLmxvZyhcbiAgICAgICAgICBgSWdub3JpbmcgcGFja2FnZSB3aXRoIG1pc21hdGNoZWQgbmFtZSwgdmVyc2lvbiwgYW5kL29yIGxpY2Vuc2UgKCR7cGFja2FnZUpzb25OYW1lfUAke3BhY2thZ2VKc29uVmVyc2lvbn0gaXMgJHtwYWNrYWdlSnNvbkxpY2Vuc2V9ICE9PSAke3BhY2thZ2VOYW1lfUAke3BhY2thZ2VWZXJzaW9ufSBpcyAke3BhY2thZ2VMaWNlbnNlfSlgLFxuICAgICAgICApO1xuICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhcbiAgICAgICAgICBNZXRyaWNOYW1lLk1JU01BVENIRURfSURFTlRJVFlfUkVKRUNUSU9OUyxcbiAgICAgICAgICAxLFxuICAgICAgICAgIFVuaXQuQ291bnQsXG4gICAgICAgICk7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoXG4gICAgICAgIE1ldHJpY05hbWUuTUlTTUFUQ0hFRF9JREVOVElUWV9SRUpFQ1RJT05TLFxuICAgICAgICAwLFxuICAgICAgICBVbml0LkNvdW50LFxuICAgICAgKTtcblxuICAgICAgLy8gRGlkIHdlIGlkZW50aWZ5IGEgbGljZW5zZSBmaWxlIG9yIG5vdD9cbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKFxuICAgICAgICBNZXRyaWNOYW1lLkZPVU5EX0xJQ0VOU0VfRklMRSxcbiAgICAgICAgbGljZW5zZVRleHQgIT0gbnVsbCA/IDEgOiAwLFxuICAgICAgICBVbml0LkNvdW50LFxuICAgICAgKTtcblxuICAgICAgY29uc3QgcGFja2FnZUxpbmtzID0gYWxsb3dlZExpbmtzLnJlZHVjZSgoYWNjdW0sIHsgY29uZmlnS2V5LCBhbGxvd2VkRG9tYWlucyB9KSA9PiB7XG4gICAgICAgIGNvbnN0IHBrZ1ZhbHVlID0gY29uc3RydWN0SHViPy5wYWNrYWdlTGlua3NbY29uZmlnS2V5XTtcblxuICAgICAgICBpZiAoIXBrZ1ZhbHVlKSB7XG4gICAgICAgICAgcmV0dXJuIGFjY3VtO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gY2hlY2sgaWYgdmFsdWUgaXMgaW4gYWxsb3dlZCBkb21haW5zIGxpc3RcbiAgICAgICAgY29uc3QgdXJsID0gbmV3IFVSTChwa2dWYWx1ZSk7XG4gICAgICAgIGlmIChhbGxvd2VkRG9tYWlucz8ubGVuZ3RoICYmICFhbGxvd2VkRG9tYWlucy5pbmNsdWRlcyh1cmwuaG9zdCkpIHtcbiAgICAgICAgICByZXR1cm4gYWNjdW07XG4gICAgICAgIH1cblxuICAgICAgICAvLyBpZiBubyBhbGxvdyBsaXN0IGlzIHByb3ZpZGVkXG4gICAgICAgIHJldHVybiB7IC4uLmFjY3VtLCBbY29uZmlnS2V5XTogcGtnVmFsdWUgfTtcbiAgICAgIH0sIHt9KTtcblxuICAgICAgY29uc3QgcGFja2FnZVRhZ3MgPSBwYWNrYWdlVGFnc0NvbmZpZy5yZWR1Y2UoKGFjY3VtOiBBcnJheTxPbWl0PFBhY2thZ2VUYWdDb25maWcsICdjb25kaXRpb24nPj4sIHRhZ0NvbmZpZykgPT4ge1xuICAgICAgICBjb25zdCB7IGNvbmRpdGlvbiwgLi4udGFnRGF0YSB9ID0gdGFnQ29uZmlnO1xuICAgICAgICBpZiAoaXNUYWdBcHBsaWNhYmxlKGNvbmRpdGlvbiwgcGFja2FnZUpzb25PYmopKSB7XG4gICAgICAgICAgcmV0dXJuIFsuLi5hY2N1bSwgdGFnRGF0YV07XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4gYWNjdW07XG4gICAgICB9LCBbXSk7XG5cbiAgICAgIGNvbnN0IG1ldGFkYXRhID0ge1xuICAgICAgICBjb25zdHJ1Y3RGcmFtZXdvcmssXG4gICAgICAgIGRhdGU6IHBheWxvYWQudGltZSxcbiAgICAgICAgbGljZW5zZVRleHQ6IGxpY2Vuc2VUZXh0Py50b1N0cmluZygndXRmLTgnKSxcbiAgICAgICAgcGFja2FnZUxpbmtzLFxuICAgICAgICBwYWNrYWdlVGFncyxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHsgYXNzZW1ibHlLZXksIG1ldGFkYXRhS2V5LCBwYWNrYWdlS2V5IH0gPSBjb25zdGFudHMuZ2V0T2JqZWN0S2V5cyhcbiAgICAgICAgcGFja2FnZU5hbWUsXG4gICAgICAgIHBhY2thZ2VWZXJzaW9uLFxuICAgICAgKTtcbiAgICAgIGNvbnNvbGUubG9nKGBXcml0aW5nIGFzc2VtYmx5IGF0ICR7YXNzZW1ibHlLZXl9YCk7XG4gICAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBwYWNrYWdlIGF0ICAke3BhY2thZ2VLZXl9YCk7XG4gICAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBtZXRhZGF0YSBhdCAgJHttZXRhZGF0YUtleX1gKTtcblxuICAgICAgLy8gd2UgdXBsb2FkIHRoZSBtZXRhZGF0YSBmaWxlIGZpcnN0IGJlY2F1c2UgdGhlIGNhdGFsb2cgYnVpbGRlciBkZXBlbmRzIG9uXG4gICAgICAvLyBpdCBhbmQgaXMgdHJpZ2dlcmVkIGJ5IHRoZSBhc3NlbWJseSBmaWxlIHVwbG9hZC5cbiAgICAgIGNvbnNvbGUubG9nKFxuICAgICAgICBgJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gfCBVcGxvYWRpbmcgcGFja2FnZSBhbmQgbWV0YWRhdGEgZmlsZXNgLFxuICAgICAgKTtcbiAgICAgIGNvbnN0IFtwa2csIHN0b3JlZE1ldGFkYXRhXSA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgICAgYXdzXG4gICAgICAgICAgLnMzKClcbiAgICAgICAgICAucHV0T2JqZWN0KHtcbiAgICAgICAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgICAgICBLZXk6IHBhY2thZ2VLZXksXG4gICAgICAgICAgICBCb2R5OiB0YXJiYWxsLkJvZHksXG4gICAgICAgICAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMsIG1heC1hZ2U9ODY0MDAsIG11c3QtcmV2YWxpZGF0ZSwgcy1tYXhhZ2U9MzAwLCBwcm94eS1yZXZhbGlkYXRlJyxcbiAgICAgICAgICAgIENvbnRlbnRUeXBlOiAnYXBwbGljYXRpb24vb2N0ZXQtc3RyZWFtJyxcbiAgICAgICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgICAgICdMYW1iZGEtTG9nLUdyb3VwJzogY29udGV4dC5sb2dHcm91cE5hbWUsXG4gICAgICAgICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSlcbiAgICAgICAgICAucHJvbWlzZSgpLFxuICAgICAgICBhd3NcbiAgICAgICAgICAuczMoKVxuICAgICAgICAgIC5wdXRPYmplY3Qoe1xuICAgICAgICAgICAgQnVja2V0OiBCVUNLRVRfTkFNRSxcbiAgICAgICAgICAgIEtleTogbWV0YWRhdGFLZXksXG4gICAgICAgICAgICBCb2R5OiBKU09OLnN0cmluZ2lmeShtZXRhZGF0YSksXG4gICAgICAgICAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMsIG1heC1hZ2U9MzAwLCBtdXN0LXJldmFsaWRhdGUsIHByb3h5LXJldmFsaWRhdGUnLFxuICAgICAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICAgICAgICAgIE1ldGFkYXRhOiB7XG4gICAgICAgICAgICAgICdMYW1iZGEtTG9nLUdyb3VwJzogY29udGV4dC5sb2dHcm91cE5hbWUsXG4gICAgICAgICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgICAgICAgJ0xhbWJkYS1SdW4tSWQnOiBjb250ZXh0LmF3c1JlcXVlc3RJZCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSlcbiAgICAgICAgICAucHJvbWlzZSgpLFxuICAgICAgXSk7XG5cbiAgICAgIC8vIG5vdyB3ZSBjYW4gdXBsb2FkIHRoZSBhc3NlbWJseS5cbiAgICAgIGNvbnNvbGUubG9nKGAke3BhY2thZ2VOYW1lfUAke3BhY2thZ2VWZXJzaW9ufSB8IFVwbG9hZGluZyBhc3NlbWJseSBmaWxlYCk7XG4gICAgICBjb25zdCBhc3NlbWJseSA9IGF3YWl0IGF3c1xuICAgICAgICAuczMoKVxuICAgICAgICAucHV0T2JqZWN0KHtcbiAgICAgICAgICBCdWNrZXQ6IEJVQ0tFVF9OQU1FLFxuICAgICAgICAgIEtleTogYXNzZW1ibHlLZXksXG4gICAgICAgICAgQm9keTogZG90SnNpaSxcbiAgICAgICAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMsIG1heC1hZ2U6IDg2NDAwLCBtdXN0LXJldmFsaWRhdGUsIHMtbWF4YWdlPTMwMCwgcHJveHktcmV2YWxpZGF0ZScsXG4gICAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICAgICAgICBNZXRhZGF0YToge1xuICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSlcbiAgICAgICAgLnByb21pc2UoKTtcblxuICAgICAgY29uc3QgY3JlYXRlZDogU3RhdGVNYWNoaW5lSW5wdXQgPSB7XG4gICAgICAgIGJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgIGFzc2VtYmx5OiB7XG4gICAgICAgICAga2V5OiBhc3NlbWJseUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IGFzc2VtYmx5LlZlcnNpb25JZCxcbiAgICAgICAgfSxcbiAgICAgICAgcGFja2FnZToge1xuICAgICAgICAgIGtleTogcGFja2FnZUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IHBrZy5WZXJzaW9uSWQsXG4gICAgICAgIH0sXG4gICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAga2V5OiBtZXRhZGF0YUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IHN0b3JlZE1ldGFkYXRhLlZlcnNpb25JZCxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgICBjb25zb2xlLmxvZyhgQ3JlYXRlZCBvYmplY3RzOiAke0pTT04uc3RyaW5naWZ5KGNyZWF0ZWQsIG51bGwsIDIpfWApO1xuXG4gICAgICBjb25zdCBzZm4gPSBhd2FpdCBhd3NcbiAgICAgICAgLnN0ZXBGdW5jdGlvbnMoKVxuICAgICAgICAuc3RhcnRFeGVjdXRpb24oe1xuICAgICAgICAgIGlucHV0OiBKU09OLnN0cmluZ2lmeShjcmVhdGVkKSxcbiAgICAgICAgICBuYW1lOiBzZm5FeGVjdXRpb25OYW1lRnJvbVBhcnRzKFxuICAgICAgICAgICAgcGFja2FnZU5hbWUsXG4gICAgICAgICAgICBgdiR7cGFja2FnZVZlcnNpb259YCxcbiAgICAgICAgICAgIGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgICAgICksXG4gICAgICAgICAgc3RhdGVNYWNoaW5lQXJuOiBTVEFURV9NQUNISU5FX0FSTixcbiAgICAgICAgfSlcbiAgICAgICAgLnByb21pc2UoKTtcbiAgICAgIGNvbnNvbGUubG9nKGBTdGFydGVkIFN0YXRlTWFjaGluZSBleGVjdXRpb246ICR7c2ZuLmV4ZWN1dGlvbkFybn1gKTtcbiAgICAgIHJlc3VsdC5wdXNoKHNmbi5leGVjdXRpb25Bcm4pO1xuICAgIH1cblxuICAgIHJldHVybiByZXN1bHQ7XG4gIH0sXG4pO1xuXG5jb25zdCBlbnVtIENvbnN0cnVjdEZyYW1ld29ya05hbWUge1xuICBBV1NfQ0RLID0gJ2F3cy1jZGsnLFxuICBDREs4UyA9ICdjZGs4cycsXG4gIENES1RGID0gJ2Nka3RmJyxcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDb25zdHJ1Y3RGcmFtZXdvcmsge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGNvbnN0cnVjdCBmcmFtZXdvcmsuXG4gICAqL1xuICByZWFkb25seSBuYW1lOiBDb25zdHJ1Y3RGcmFtZXdvcmtOYW1lO1xuXG4gIC8qKlxuICAgKiBUaGUgbWFqb3IgdmVyc2lvbiBvZiB0aGUgY29uc3RydWN0IGZyYW1ld29yayB0aGF0IGlzIHVzZWQsIGlmIGl0IGNvdWxkIGJlXG4gICAqIGlkZW50aWZpZWQuXG4gICAqL1xuICByZWFkb25seSBtYWpvclZlcnNpb24/OiBudW1iZXI7XG59XG5cbi8qKlxuICogRGV0ZXJtaW5lcyB0aGUgQ29uc3RydWN0IGZyYW1ld29yayB1c2VkIGJ5IHRoZSBwcm92aWRlZCBhc3NlbWJseS5cbiAqXG4gKiBAcGFyYW0gYXNzZW1ibHkgdGhlIGFzc2VtYmx5IGZvciB3aGljaCBhIGNvbnN0cnVjdCBmcmFtZXdvcmsgc2hvdWxkIGJlXG4gKiAgICAgICAgICAgICAgICAgaWRlbnRpZmllZC5cbiAqXG4gKiBAcmV0dXJucyBhIGNvbnN0cnVjdCBmcmFtZXdvcmsgaWYgb25lIGNvdWxkIGJlIGlkZW50aWZpZWQuXG4gKi9cbmZ1bmN0aW9uIGRldGVjdENvbnN0cnVjdEZyYW1ld29yayhhc3NlbWJseTogQXNzZW1ibHkpOiBDb25zdHJ1Y3RGcmFtZXdvcmsgfCB1bmRlZmluZWQge1xuICBsZXQgbmFtZTogQ29uc3RydWN0RnJhbWV3b3JrWyduYW1lJ10gfCB1bmRlZmluZWQ7XG4gIGxldCBuYW1lQW1iaWd1b3VzID0gZmFsc2U7XG4gIGxldCBtYWpvclZlcnNpb246IG51bWJlciB8IHVuZGVmaW5lZDtcbiAgbGV0IG1ham9yVmVyc2lvbkFtYmlndW91cyA9IGZhbHNlO1xuICBkZXRlY3RDb25zdHJ1Y3RGcmFtZXdvcmtQYWNrYWdlKGFzc2VtYmx5Lm5hbWUsIGFzc2VtYmx5LnZlcnNpb24pO1xuICBmb3IgKGNvbnN0IGRlcE5hbWUgb2YgT2JqZWN0LmtleXMoYXNzZW1ibHkuZGVwZW5kZW5jeUNsb3N1cmUgPz8ge30pKSB7XG4gICAgZGV0ZWN0Q29uc3RydWN0RnJhbWV3b3JrUGFja2FnZShkZXBOYW1lKTtcbiAgICBpZiAobmFtZUFtYmlndW91cykge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gIH1cbiAgcmV0dXJuIG5hbWUgJiYgeyBuYW1lLCBtYWpvclZlcnNpb246IG1ham9yVmVyc2lvbkFtYmlndW91cyA/IHVuZGVmaW5lZCA6IG1ham9yVmVyc2lvbiB9O1xuXG4gIGZ1bmN0aW9uIGRldGVjdENvbnN0cnVjdEZyYW1ld29ya1BhY2thZ2UocGFja2FnZU5hbWU6IHN0cmluZywgdmVyc2lvblJhbmdlID0gYXNzZW1ibHkuZGVwZW5kZW5jaWVzPy5bcGFja2FnZU5hbWVdKTogdm9pZCB7XG4gICAgaWYgKHBhY2thZ2VOYW1lLnN0YXJ0c1dpdGgoJ0Bhd3MtY2RrLycpIHx8IHBhY2thZ2VOYW1lID09PSAnYXdzLWNkay1saWInIHx8IHBhY2thZ2VOYW1lID09PSAnbW9ub2NkaycpIHtcbiAgICAgIGlmIChuYW1lICYmIG5hbWUgIT09IENvbnN0cnVjdEZyYW1ld29ya05hbWUuQVdTX0NESykge1xuICAgICAgICAvLyBJZGVudGlmaWVkIG11bHRpcGxlIGNhbmRpZGF0ZXMsIHNvIHJldHVybmluZyBhbWJpZ3VvdXMuLi5cbiAgICAgICAgbmFtZUFtYmlndW91cyA9IHRydWU7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cbiAgICAgIG5hbWUgPSBDb25zdHJ1Y3RGcmFtZXdvcmtOYW1lLkFXU19DREs7XG4gICAgfSBlbHNlIGlmIChwYWNrYWdlTmFtZSA9PT0gJ2Nka3RmJyB8fCBwYWNrYWdlTmFtZS5zdGFydHNXaXRoKCdAY2RrdGYvJykpIHtcbiAgICAgIGlmIChuYW1lICYmIG5hbWUgIT09IENvbnN0cnVjdEZyYW1ld29ya05hbWUuQ0RLVEYpIHtcbiAgICAgICAgLy8gSWRlbnRpZmllZCBtdWx0aXBsZSBjYW5kaWRhdGVzLCBzbyByZXR1cm5pbmcgYW1iaWd1b3VzLi4uXG4gICAgICAgIG5hbWVBbWJpZ3VvdXMgPSB0cnVlO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgICBuYW1lID0gQ29uc3RydWN0RnJhbWV3b3JrTmFtZS5DREtURjtcbiAgICB9IGVsc2UgaWYgKHBhY2thZ2VOYW1lID09PSAnY2RrOHMnIHx8IC9eY2RrOHMtcGx1cyg/Oi0oPzoxN3wyMHwyMXwyMikpPyQvLnRlc3QocGFja2FnZU5hbWUpKSB7XG4gICAgICBpZiAobmFtZSAmJiBuYW1lICE9PSBDb25zdHJ1Y3RGcmFtZXdvcmtOYW1lLkNESzhTKSB7XG4gICAgICAgIC8vIElkZW50aWZpZWQgbXVsdGlwbGUgY2FuZGlkYXRlcywgc28gcmV0dXJuaW5nIGFtYmlndW91cy4uLlxuICAgICAgICBuYW1lQW1iaWd1b3VzID0gdHJ1ZTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuICAgICAgbmFtZSA9IENvbnN0cnVjdEZyYW1ld29ya05hbWUuQ0RLOFM7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgaWYgKHZlcnNpb25SYW5nZSkge1xuICAgICAgY29uc3QgbWFqb3IgPSBtaW5WZXJzaW9uKHZlcnNpb25SYW5nZSk/Lm1ham9yO1xuICAgICAgaWYgKG1ham9yVmVyc2lvbiAhPSBudWxsICYmIG1ham9yVmVyc2lvbiAhPT0gbWFqb3IpIHtcbiAgICAgICAgLy8gSWRlbnRpZmllZCBtdWx0aXBsZSBjYW5kaWRhdGVzLCBzbyB0aGlzIGlzIGFtYmlndW91cy4uLlxuICAgICAgICBtYWpvclZlcnNpb25BbWJpZ3VvdXMgPSB0cnVlO1xuICAgICAgfVxuICAgICAgbWFqb3JWZXJzaW9uID0gbWFqb3I7XG4gICAgfVxuICAgIHJldHVybjtcbiAgfVxufVxuXG4vKipcbiAqIENoZWNrcyB3aGV0aGVyIHRoZSBwcm92aWRlZCBmaWxlIG5hbWUgY29ycmVzcG9uZHMgdG8gYSBsaWNlbnNlIGZpbGUgb3Igbm90LlxuICpcbiAqIEBwYXJhbSBmaWxlTmFtZSB0aGUgZmlsZSBuYW1lIHRvIGJlIGNoZWNrZWQuXG4gKlxuICogQHJldHVybnMgYHRydWVgIElJRiB0aGUgZmlsZSBpcyBuYW1lZCBMSUNFTlNFIGFuZCBoYXMgdGhlIC5NRCBvciAuVFhUXG4gKiAgICAgICAgICBleHRlbnNpb24sIG9yIG5vIGV4dGVuc2lvbiBhdCBhbGwuIFRoZSB0ZXN0IGlzIGNhc2UtaW5zZW5zaXRpdmUuXG4gKi9cbmZ1bmN0aW9uIGlzTGljZW5zZUZpbGUoZmlsZU5hbWU6IHN0cmluZyk6IGJvb2xlYW4ge1xuICBjb25zdCBleHQgPSBleHRuYW1lKGZpbGVOYW1lKTtcbiAgY29uc3QgcG9zc2libGVFeHRlbnNpb25zID0gbmV3IFNldChbJycsICcubWQnLCAnLnR4dCddKTtcbiAgcmV0dXJuIChcbiAgICBwb3NzaWJsZUV4dGVuc2lvbnMuaGFzKGV4dC50b0xvd2VyQ2FzZSgpKSAmJlxuICAgIGJhc2VuYW1lKGZpbGVOYW1lLCBleHQpLnRvVXBwZXJDYXNlKCkgPT09ICdMSUNFTlNFJ1xuICApO1xufVxuXG4vKipcbiAqIENyZWF0ZXMgYSBTdGVwRnVuY3Rpb24gZXhlY3V0aW9uIHJlcXVlc3QgbmFtZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgcGFydHMuXG4gKiBUaGUgcmVzdWx0IGlzIGd1YXJhbnRlZWQgdG8gYmUgODAgY2hhcmFjdGVycyBvciBsZXNzIGFuZCB0byBjb250YWluIG9ubHlcbiAqIGNoYXJhY3RlcnMgdGhhdCBhcmUgdmFsaWQgZm9yIGEgU3RlcEZ1bmN0aW9uIGV4ZWN1dGlvbiByZXF1ZXN0IG5hbWUgZm9yIHdoaWNoXG4gKiBDbG91ZFdhdGNoIExvZ2dpbmcgY2FuIGJlIGVuYWJsZWQuIFRoZSByZXN1bHRpbmcgbmFtZSBpcyB2ZXJ5IGxpa2VseSB0b1xuICogYmUgdW5pcXVlIGZvciBhIGdpdmVuIGlucHV0LlxuICovXG5mdW5jdGlvbiBzZm5FeGVjdXRpb25OYW1lRnJvbVBhcnRzKFxuICBmaXJzdDogc3RyaW5nLFxuICAuLi5yZXN0OiByZWFkb25seSBzdHJpbmdbXVxuKTogc3RyaW5nIHtcbiAgY29uc3QgcGFydHMgPSBbZmlyc3QsIC4uLnJlc3RdO1xuICBjb25zdCBuYW1lID0gcGFydHNcbiAgICAubWFwKChwYXJ0KSA9PiBwYXJ0LnJlcGxhY2UoL1teYS16MC05Xy1dKy9naSwgJ18nKSlcbiAgICAuam9pbignXycpXG4gICAgLnJlcGxhY2UoL15fL2csICcnKVxuICAgIC5yZXBsYWNlKC9fezIsfS9nLCAnXycpO1xuICBpZiAobmFtZS5sZW5ndGggPD0gODApIHtcbiAgICByZXR1cm4gbmFtZTtcbiAgfVxuICBjb25zdCBzdWZmaXggPSBjcmVhdGVIYXNoKCdzaGEyNTYnKVxuICAgIC8vIFRoZSBoYXNoIGlzIGNvbXB1dGVkIGJhc2VkIG9uIGlucHV0IGFyZ3VtZW50cywgdG8gbWF4aW1pemUgdW5pY2l0eVxuICAgIC51cGRhdGUocGFydHMuam9pbignXycpKVxuICAgIC5kaWdlc3QoJ2hleCcpXG4gICAgLnN1YnN0cmluZygwLCA2KTtcbiAgcmV0dXJuIGAke25hbWUuc3Vic3RyaW5nKDAsIDgwIC0gc3VmZml4Lmxlbmd0aCAtIDEpfV8ke3N1ZmZpeH1gO1xufVxuXG4vKipcbiAqIEluZ2VzdGlvbiBjb25maWd1cmF0aW9uIGZvciBwYWNrYWdlIGxpbmtzIGFuZCB0YWdzXG4gKi9cbmludGVyZmFjZSBDb25maWcge1xuICBwYWNrYWdlVGFnczogUGFja2FnZVRhZ0NvbmZpZ1tdO1xuICBwYWNrYWdlTGlua3M6IFBhY2thZ2VMaW5rQ29uZmlnW107XG59XG5cbi8qKlxuICogTG9va3MgZm9yIHRoZSBpbmdlc3Rpb24gY29uZmlndXJhdGlvbiBmaWxlIGluIHRoZSBwYXNzZWQgYnVja2V0IGFuZCBwYXJzZXNcbiAqIGl0LiBJZiBpdCBpcyBub3QgZm91bmQgb3IgaW52YWxpZCB0aGVuIGEgZGVmYXVsdCBpcyByZXR1cm5lZC5cbiAqL1xuYXN5bmMgZnVuY3Rpb24gZ2V0Q29uZmlnKGJ1Y2tldDogc3RyaW5nLCBrZXk6IHN0cmluZyk6IFByb21pc2U8Q29uZmlnPiB7XG4gIGNvbnN0IGRlZmF1bHRDb25maWcgPSB7XG4gICAgcGFja2FnZVRhZ3M6IFtdLFxuICAgIHBhY2thZ2VMaW5rczogW10sXG4gIH07XG4gIHRyeSB7XG4gICAgY29uc3QgcmVxID0gYXdhaXQgYXdzLnMzKCkuZ2V0T2JqZWN0KHtcbiAgICAgIEJ1Y2tldDogYnVja2V0LFxuICAgICAgS2V5OiBrZXksXG4gICAgfSkucHJvbWlzZSgpO1xuICAgIGNvbnN0IGJvZHkgPSByZXE/LkJvZHk/LnRvU3RyaW5nKCk7XG4gICAgaWYgKGJvZHkpIHtcbiAgICAgIHJldHVybiBKU09OLnBhcnNlKGJvZHkpO1xuICAgIH1cbiAgICByZXR1cm4gZGVmYXVsdENvbmZpZztcbiAgfSBjYXRjaCAoZSkge1xuICAgIGNvbnNvbGUuZXJyb3IoZSk7XG4gICAgcmV0dXJuIGRlZmF1bHRDb25maWc7XG4gIH1cbn1cbiJdfQ==