"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnsupportedLanguageError = exports.DocumentationLanguage = void 0;
/**
 * Supported languages to generate documentation in.
 */
class DocumentationLanguage {
    constructor(lang) {
        this.lang = lang;
    }
    /**
     * Transform a literal string to the `DocumentationLanguage` object.
     *
     * Throws an `UnsupportedLanguageError` if the language is not supported.
     */
    static fromString(lang) {
        switch (lang) {
            case DocumentationLanguage.TYPESCRIPT.toString():
                return DocumentationLanguage.TYPESCRIPT;
            case DocumentationLanguage.PYTHON.toString():
                return DocumentationLanguage.PYTHON;
            case DocumentationLanguage.JAVA.toString():
                return DocumentationLanguage.JAVA;
            case DocumentationLanguage.CSHARP.toString():
                return DocumentationLanguage.CSHARP;
            default:
                throw new UnsupportedLanguageError(lang, DocumentationLanguage.ALL);
        }
    }
    toString() {
        return this.lang;
    }
}
exports.DocumentationLanguage = DocumentationLanguage;
/**
 * TypeScript.
 */
DocumentationLanguage.TYPESCRIPT = new DocumentationLanguage('typescript');
/**
 * Python.
 */
DocumentationLanguage.PYTHON = new DocumentationLanguage('python');
/**
 * Java.
 */
DocumentationLanguage.JAVA = new DocumentationLanguage('java');
/**
 * CSharp.
 */
DocumentationLanguage.CSHARP = new DocumentationLanguage('csharp');
/**
 * All supported languages.
 */
DocumentationLanguage.ALL = [
    DocumentationLanguage.TYPESCRIPT,
    DocumentationLanguage.PYTHON,
    DocumentationLanguage.JAVA,
    DocumentationLanguage.CSHARP,
];
class UnsupportedLanguageError extends Error {
    constructor(lang, supported) {
        super(`Unsupported language: ${lang}. Supported languages are: [${supported}]`);
    }
}
exports.UnsupportedLanguageError = UnsupportedLanguageError;
//# sourceMappingURL=data:application/json;base64,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