"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            // Set embedded metrics format environment to "Local", to have a consistent experience.
            AWS_EMF_ENVIRONMENT: 'Local',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', { retention: props.logRetention });
        this.containerDefinition = new transliterator_1.Transliterator(this, 'Resource', {
            environment,
            logging: aws_ecs_1.LogDrivers.awsLogs({ logGroup: this.logGroup, streamPrefix: 'transliterator' }),
            taskDefinition: new aws_ecs_1.FargateTaskDefinition(this, 'TaskDefinition', {
                cpu: 4096,
                memoryLimitMiB: 8192,
            }),
        });
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.taskDefinition.taskRole);
        // The task handler reads & writes to this bucket.
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
        }
        const executionRole = this.taskDefinition.obtainExecutionRole();
        (_b = props.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.ecrApi.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
            ],
            resources: ['*'],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_c = props.vpcEndpoints) === null || _c === void 0 ? void 0 : _c.ecr.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
            // We cannot get the ECR repository info from an asset... So scoping down to same-account repositories instead...
            resources: [core_1.Stack.of(this).formatArn({ service: 'ecr', resource: 'repository', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' })],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_d = props.vpcEndpoints) === null || _d === void 0 ? void 0 : _d.cloudWatchLogs.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: this.logGroup.logGroupName }),
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: `${this.logGroup.logGroupName}:log-stream:*` }),
            ],
            principals: [executionRole],
            sid: 'Allow-Logging',
        }));
        (_e = props.vpcEndpoints) === null || _e === void 0 ? void 0 : _e.stepFunctions.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
                'states:SendTaskSuccess',
            ],
            resources: ['*'],
            principals: [this.taskDefinition.taskRole],
            sid: 'Allow-StepFunctions-Callbacks',
        }));
    }
    get taskDefinition() {
        return this.containerDefinition.taskDefinition;
    }
    createEcsRunTask(scope, id, opts) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(scope, id, {
            // The container sends heartbeats every minute, but when the runloop will
            // actually get to submitting it is fairly dependent on the async
            // workload's nature... so we don't rely on it all too strongly, and
            // default to a 5 minutes timeout here as a minimal protection. Options
            // may override this liberally if they know better.
            heartbeat: core_1.Duration.minutes(5),
            ...opts,
            containerOverrides: [{
                    containerDefinition: this.containerDefinition,
                    command: aws_stepfunctions_1.JsonPath.listAt('$'),
                    environment: [
                        { name: 'TARGET_LANGUAGE', value: opts.language.toString() },
                        { name: 'SFN_TASK_TOKEN', value: aws_stepfunctions_1.JsonPath.taskToken },
                    ],
                }],
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({ platformVersion: aws_ecs_1.FargatePlatformVersion.VERSION1_4 }),
            subnets: opts.vpcSubnets,
            taskDefinition: this.taskDefinition,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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