"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        console.log(`TARGET_LANGUAGE=${language}`);
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tarball = path.join(fs.mkdtempSync(path.join(os.tmpdir(), 'packages-')), 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const isCSharpAndSupported = language === 'csharp' && assembly.targets.dotnet;
        const isOtherwiseSupported = language === 'typescript' || assembly.targets[language];
        if (!isCSharpAndSupported
            && !isOtherwiseSupported) {
            console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
            console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
            for (const submodule of [undefined, ...submodules]) {
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                const response = await uploadFile(event.bucket, key, event.assembly.versionId);
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            }
            return created;
        }
        const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
            metrics.setDimensions();
            metrics.setNamespace(constants_1.METRICS_NAMESPACE);
            const uploads = new Map();
            const docs = await docgen.Documentation.forPackage(tarball, { name: packageName, language: docgen.Language.fromString(lang) });
            function renderAndDispatch(submodule) {
                console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                const page = Buffer.from(docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render());
                metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                const { buffer: body, contentEncoding } = compressContent(page);
                metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                console.log(`Uploading ${key}`);
                const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                uploads.set(key, upload);
            }
            renderAndDispatch();
            for (const submodule of submodules) {
                renderAndDispatch(submodule);
            }
            for (const [key, upload] of uploads.entries()) {
                const response = await upload;
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
            }
        });
        await generateDocs(language);
        return created;
    });
}
exports.handler = handler;
function compressContent(buffer) {
    if (buffer.length < 1024) {
        return { buffer };
    }
    const gz = zlib_1.gzipSync(buffer, { level: 9 });
    // If it did not compress well, we'll keep the un-compressed original...
    if (gz.length >= buffer.length) {
        return { buffer };
    }
    return { buffer: gz, contentEncoding: 'gzip' };
}
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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