"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = exports.SampleLanguage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported languages to generate sample code.
 */
var SampleLanguage;
(function (SampleLanguage) {
    SampleLanguage["TYPESCRIPT"] = "ts";
    SampleLanguage["PYTHON"] = "py";
})(SampleLanguage = exports.SampleLanguage || (exports.SampleLanguage = {}));
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            projenrcTsOptions: {
                projenCodeDir: "",
            },
            release: false,
            sampleCode: false,
            name: "monorepo",
            defaultReleaseBranch: "mainline",
        });
        this.implicitDependencies = {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("aws-prototyping-sdk", "@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(options.nxIgnorePatterns || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs
                .readFileSync("./node_modules/aws-prototyping-sdk/lib/pdk_projen/nx_monorepo/plugin/nx-monorepo-plugin.js")
                .toString("utf-8")
                .split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(options.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
        if (options.sampleLanguage) {
            if (fs.existsSync(path.join(this.outdir, ".projenrc.ts.bk"))) {
                console.log("Ignoring generation of sample code as this is a destructive action and should only be set on initial synthesis.");
            }
            else {
                fs.copyFileSync(path.join(this.outdir, ".projenrc.ts"), ".projenrc.ts.bk"); // Make a backup of the existing .projenrc.ts just in case
                fs.copyFileSync(path.join(this.outdir, `node_modules/aws-prototyping-sdk/sample/nx_monorepo/nx-monorepo-sample-${options.sampleLanguage}.ts`), ".projenrc.ts");
            }
        }
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
        // Add workspaces for each subproject
        this.package.addField("workspaces", this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)));
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Check to see if a new subProject was added
        const newSubProject = this.subProjects.find((subProject) => !fs.existsSync(subProject.outdir));
        // Need to synth before generating the package.json otherwise the subdirectory won't exist
        newSubProject && super.synth();
        this.subProjects
            .filter((subProject) => !(subProject instanceof javascript_1.NodeProject))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
        super.synth();
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.pdk_projen.NxMonorepoProject", version: "0.0.19" };
//# sourceMappingURL=data:application/json;base64,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