import os
import posixpath
import shlex
from simpleeval import simple_eval
from typing import Optional, List, Iterable

from ddb.binary import Binary
from ddb.config import config


class DockerBinary(Binary):
    """
    Binary to run docker compose command.
    """

    def __init__(self,
                 name: str,
                 docker_compose_service: str,
                 workdir: Optional[str] = None,
                 options: Optional[str] = None,
                 options_condition: Optional[str] = None,
                 args: Optional[str] = None):
        self._name = name
        self.docker_compose_service = docker_compose_service
        self.workdir = workdir
        self.options = options
        self.options_condition = options_condition
        self.args = args

    @property
    def name(self) -> str:
        return self._name

    def command(self, *args) -> Iterable[str]:
        params = ["run"]

        if self.workdir:
            relpath = os.path.relpath(os.getcwd(), config.paths.project_home)
            container_workdir = posixpath.join(self.workdir, relpath)
            params.append("--workdir=%s" % (container_workdir,))

        self.add_options_to_params(params, self.options, self.options_condition, args)

        params.append(self.docker_compose_service)
        if self.args:
            params.extend(shlex.split(self.args))

        docker_compose_bin = config.data["docker.compose.bin"]
        docker_compose_args = config.data.get("docker.compose.args", [])
        command = [docker_compose_bin] + docker_compose_args + list(params)
        return command

    def add_options_to_params(self, params, options, condition, args=()):  # pylint: disable=no-self-use
        """
        Add options to params if condition is fulfilled
        :param params: the list of parameters
        :param options: options to add
        :param condition: the condition to fulfilled
        :param args: the list of args of binary call
        :return:
        """
        if condition is not None and options is not None:
            if simple_eval(condition, functions={}, names={'args': args}):
                params.extend(shlex.split(options))
        else:
            if options is not None:
                params.extend(shlex.split(options))
