"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitDependencyCheck = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cli_1 = require("./cli");
class CodeCommitDependencyCheck extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { schedule, computeType, repository, preCheckCommand = 'echo "No preCheckCommand!"', version = '5.3.2', projectName, failOnCVSS = 0, paths = ['.'], excludes, suppressions, enableExperimental, reportsBucket, } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        const dependencyCheck = `dependency-check-${version}-release`;
        const cli = new cli_1.Cli();
        this.checkProject = new aws_codebuild_1.Project(this, 'CheckProject', {
            cache: aws_codebuild_1.Cache.local(aws_codebuild_1.LocalCacheMode.CUSTOM),
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    install: {
                        commands: [
                            'echo "[===== Install OWASP Dependency Check =====]"',
                            'wget -O public-key.asc https://bintray.com/user/downloadSubjectPublicKey?username=jeremy-long',
                            'gpg --keyid-format long --list-options show-keyring public-key.asc',
                            'gpg --import public-key.asc',
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip`,
                            `wget https://dl.bintray.com/jeremy-long/owasp/${dependencyCheck}.zip.asc`,
                            `gpg --verify ${dependencyCheck}.zip.asc ${dependencyCheck}.zip`,
                            `unzip ${dependencyCheck}.zip -d /opt`,
                            'chmod +x /opt/dependency-check/bin/dependency-check.sh',
                            'export PATH="$PATH:/opt/dependency-check/bin"',
                        ],
                    },
                    pre_build: {
                        commands: [
                            `echo "[===== Clone repository: ${repositoryName} =====]"`,
                            `git clone "${repositoryCloneUrlHttp}"`,
                            `cd ${repositoryName}`,
                            `${preCheckCommand}`,
                            'SHA=$(git rev-parse HEAD)',
                            'cd ${CODEBUILD_SRC_DIR}',
                        ],
                    },
                    build: {
                        commands: [
                            `echo "[===== Scan repository: ${repositoryName} =====]"`,
                            'echo "[===== SHA: $SHA =====]"',
                            'mkdir reports',
                            cli.version(),
                            cli.scan({
                                projectName: projectName || repositoryName,
                                basedir: repositoryName,
                                paths,
                                failOnCVSS,
                                enableExperimental,
                                suppressions,
                                excludes,
                            }),
                        ],
                        finally: [
                            'echo "[===== Upload reports =====]"',
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            reportsBucket
                                ? `aws s3 cp reports/dependency-check-report.html s3://${reportsBucket.bucketName}/${repositoryName}/\${dt}_UTC/`
                                : 'echo "No reportsBuckets"',
                        ],
                    },
                },
                reports: {
                    dependencyCheckReport: {
                        files: ['reports/dependency-check-junit.xml'],
                    },
                },
                cache: {
                    paths: ['/opt/dependency-check/data/**/*'],
                },
            }),
        });
        this.checkProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        if (reportsBucket) {
            reportsBucket.grantWrite(this.checkProject);
        }
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.checkProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a check fails.
     */
    onCheckFailed(id, options) {
        return this.checkProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a check starts.
     */
    onCheckStarted(id, options) {
        return this.checkProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a check complets successfully.
     */
    onCheckSucceeded(id, options) {
        return this.checkProject.onBuildSucceeded(id, options);
    }
}
exports.CodeCommitDependencyCheck = CodeCommitDependencyCheck;
_a = JSII_RTTI_SYMBOL_1;
CodeCommitDependencyCheck[_a] = { fqn: "@cloudcomponents/cdk-dependency-check.CodeCommitDependencyCheck", version: "1.43.0" };
//# sourceMappingURL=data:application/json;base64,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