"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
const coreutils_1 = require("@phosphor/coreutils");
/**
 * A namespace for kernel messages.
 */
var KernelMessage;
(function (KernelMessage) {
    function createMessage(options) {
        // Backwards compatibility workaround for services 4.0 defining the wrong
        // comm_info_request content. This should be removed with the deprecated
        // `target` content option in services 5.0. See
        // https://github.com/jupyterlab/jupyterlab/issues/6947
        if (options.msgType === 'comm_info_request') {
            const content = options.content;
            if (content.target_name === undefined) {
                content.target_name = content.target;
            }
            delete content.target;
        }
        return {
            buffers: options.buffers || [],
            channel: options.channel,
            content: options.content,
            header: {
                date: new Date().toISOString(),
                msg_id: options.msgId || coreutils_1.UUID.uuid4(),
                msg_type: options.msgType,
                session: options.session,
                username: options.username || '',
                version: '5.2'
            },
            metadata: options.metadata || {},
            parent_header: options.parentHeader || {}
        };
    }
    KernelMessage.createMessage = createMessage;
    /**
     * Test whether a kernel message is a `'stream'` message.
     */
    function isStreamMsg(msg) {
        return msg.header.msg_type === 'stream';
    }
    KernelMessage.isStreamMsg = isStreamMsg;
    /**
     * Test whether a kernel message is an `'display_data'` message.
     */
    function isDisplayDataMsg(msg) {
        return msg.header.msg_type === 'display_data';
    }
    KernelMessage.isDisplayDataMsg = isDisplayDataMsg;
    /**
     * Test whether a kernel message is an `'update_display_data'` message.
     */
    function isUpdateDisplayDataMsg(msg) {
        return msg.header.msg_type === 'update_display_data';
    }
    KernelMessage.isUpdateDisplayDataMsg = isUpdateDisplayDataMsg;
    /**
     * Test whether a kernel message is an `'execute_input'` message.
     */
    function isExecuteInputMsg(msg) {
        return msg.header.msg_type === 'execute_input';
    }
    KernelMessage.isExecuteInputMsg = isExecuteInputMsg;
    /**
     * Test whether a kernel message is an `'execute_result'` message.
     */
    function isExecuteResultMsg(msg) {
        return msg.header.msg_type === 'execute_result';
    }
    KernelMessage.isExecuteResultMsg = isExecuteResultMsg;
    /**
     * Test whether a kernel message is an `'error'` message.
     */
    function isErrorMsg(msg) {
        return msg.header.msg_type === 'error';
    }
    KernelMessage.isErrorMsg = isErrorMsg;
    /**
     * Test whether a kernel message is a `'status'` message.
     */
    function isStatusMsg(msg) {
        return msg.header.msg_type === 'status';
    }
    KernelMessage.isStatusMsg = isStatusMsg;
    /**
     * Test whether a kernel message is a `'clear_output'` message.
     */
    function isClearOutputMsg(msg) {
        return msg.header.msg_type === 'clear_output';
    }
    KernelMessage.isClearOutputMsg = isClearOutputMsg;
    /**
     * Test whether a kernel message is an experimental `'debug_event'` message.
     *
     * @hidden
     *
     * #### Notes
     * Debug messages are experimental messages that are not in the official
     * kernel message specification. As such, this is *NOT* considered
     * part of the public API, and may change without notice.
     */
    function isDebugEventMsg(msg) {
        return msg.header.msg_type === 'debug_event';
    }
    KernelMessage.isDebugEventMsg = isDebugEventMsg;
    /**
     * Test whether a kernel message is a `'comm_open'` message.
     */
    function isCommOpenMsg(msg) {
        return msg.header.msg_type === 'comm_open';
    }
    KernelMessage.isCommOpenMsg = isCommOpenMsg;
    /**
     * Test whether a kernel message is a `'comm_close'` message.
     */
    function isCommCloseMsg(msg) {
        return msg.header.msg_type === 'comm_close';
    }
    KernelMessage.isCommCloseMsg = isCommCloseMsg;
    /**
     * Test whether a kernel message is a `'comm_msg'` message.
     */
    function isCommMsgMsg(msg) {
        return msg.header.msg_type === 'comm_msg';
    }
    KernelMessage.isCommMsgMsg = isCommMsgMsg;
    /**
     * Test whether a kernel message is a `'kernel_info_request'` message.
     */
    function isInfoRequestMsg(msg) {
        return msg.header.msg_type === 'kernel_info_request';
    }
    KernelMessage.isInfoRequestMsg = isInfoRequestMsg;
    /**
     * Test whether a kernel message is an `'execute_reply'` message.
     */
    function isExecuteReplyMsg(msg) {
        return msg.header.msg_type === 'execute_reply';
    }
    KernelMessage.isExecuteReplyMsg = isExecuteReplyMsg;
    /**
     * Test whether a kernel message is an experimental `'debug_request'` message.
     *
     * @hidden
     *
     * #### Notes
     * Debug messages are experimental messages that are not in the official
     * kernel message specification. As such, this is *NOT* considered
     * part of the public API, and may change without notice.
     */
    function isDebugRequestMsg(msg) {
        return msg.header.msg_type === 'debug_request';
    }
    KernelMessage.isDebugRequestMsg = isDebugRequestMsg;
    /**
     * Test whether a kernel message is an experimental `'debug_reply'` message.
     *
     * @hidden
     *
     * #### Notes
     * Debug messages are experimental messages that are not in the official
     * kernel message specification. As such, this is *NOT* considered
     * part of the public API, and may change without notice.
     */
    function isDebugReplyMsg(msg) {
        return msg.header.msg_type === 'debug_reply';
    }
    KernelMessage.isDebugReplyMsg = isDebugReplyMsg;
    /**
     * Test whether a kernel message is an `'input_request'` message.
     */
    function isInputRequestMsg(msg) {
        return msg.header.msg_type === 'input_request';
    }
    KernelMessage.isInputRequestMsg = isInputRequestMsg;
    /**
     * Test whether a kernel message is an `'input_reply'` message.
     */
    function isInputReplyMsg(msg) {
        return msg.header.msg_type === 'input_reply';
    }
    KernelMessage.isInputReplyMsg = isInputReplyMsg;
})(KernelMessage = exports.KernelMessage || (exports.KernelMessage = {}));
//# sourceMappingURL=messages.js.map