import mailbox
from pathlib import Path
from textwrap import dedent

from tqdm import tqdm

from mailanalysis.processors import Processor


def process_mbox(path: Path, processors: list[Processor]) -> str:
    """Process a mailbox file using the given processors.

    Parameters
    ----------
    path : Path
        Where to find the mailbox.
    processors : list of Processor
        Processors that will receive the messages and generate the
        corresponding report snippets.
    Returns
    -------
    str
        The reports generated by the processors after all the mails
        have been processed
    """

    for message in tqdm(mailbox.mbox(path)):
        # headers described at https://tools.ietf.org/html/rfc2822.html
        for processor in processors:
            processor.process(message)

    return "\n".join(
        f"<section>{dedent(p.report_snippet())}</section>" for p in processors
    )
