import json

from datetime import date, datetime
from hashlib import sha1
from pathlib import Path
from typing import Dict, List, Union, Optional
from urllib.parse import urlparse as url_parse, parse_qs as query_string_parse
from enum import Enum

from backports.datetime_fromisoformat import MonkeyPatch
from bas_metadata_library.standards.iso_19115_2_v1 import MetadataRecord
from dateutil.relativedelta import relativedelta
from markdown import markdown

from scar_add_metadata_toolbox.csw import (
    CSWGetRecordMode,
    CSWClient,
    RecordNotFoundException,
    RecordInsertConflictException,
)

from scar_add_metadata_toolbox.hazmat.metadata import (
    generate_xml_record_from_record_config_without_xml_declaration,
    dump_record_to_json,
    load_record_from_json,
    process_usage_constraints,
)

# Workaround for lack of `date(time).fromisoformat()` method in Python 3.6
MonkeyPatch.patch_fromisoformat()


class WellKnownExtents(Enum):
    """
    Represents a set of spatial bounding box extents for common, well-known, regions

    These are needed as bounding boxes for things like Antarctica are not simple to visualise using a corner coordinates
    due to the projections used at the poles. This enumeration holds these more complex representations.
    """

    ANTARCTICA = {
        "type": "Polygon",
        "coordinates": [
            [
                [-0.000005819109207, -3333134.027676715515554],
                [-232507.676298886770383, -3325014.680707213934511],
                [-463882.598564556566998, -3300696.196441804990172],
                [-692997.531439224374481, -3260297.052091349847615],
                [-918736.24995012988802, -3204014.068240646272898],
                [-1139998.977853273041546, -3132121.449904185254127],
                [-1355707.745328332995996, -3044969.450702776666731],
                [-1564811.640943090897053, -2942982.666416062042117],
                [-1766291.931537110125646, -2826657.966405434999615],
                [-1959167.025387293193489, -2696562.072916458826512],
                [-2142497.254424094222486, -2553328.800065048970282],
                [-2315389.452199890743941, -2397655.965958814136684],
                [-2477001.305306139867753, -2230301.992997379507869],
                [-2626545.457007425837219, -2052082.212955917464569],
                [-2763293.343295965809375, -1863864.894608139060438],
                [-2886578.742180570960045, -1666567.013848418835551],
                [-2995801.019590450450778, -1461149.78611955512315],
                [-3090428.055569021496922, -1248613.98350662435405],
                [-3169998.836708850227296, -1029995.059075984405354],
                [-3234125.702162629924715, -806358.102252019802108],
                [-3282496.232287866529077, -578792.649808516609482],
                [-3314874.770723965018988, -348407.377755039895419],
                [-3331103.572484511416405, -116324.700031127562397],
                [-3331103.572485930752009, 116324.699990476612584],
                [-3314874.770722748246044, 348407.377766617632005],
                [-3282496.232285845093429, 578792.649819981306791],
                [-3234125.702159813605249, 806358.102263315580785],
                [-3169998.836705251596868, 1029995.059087058994919],
                [-3090428.055564660578966, 1248613.983517418382689],
                [-2995801.01958534726873, 1461149.786130018532276],
                [-2886578.742174750193954, 1666567.013858501100913],
                [-2763293.343318711034954, 1863864.894574417034164],
                [-2626545.457032468169928, 2052082.21292386460118],
                [-2477001.305298350285739, 2230301.993006030563265],
                [-2315389.452191516757011, 2397655.965966900810599],
                [-2142497.254415175877512, 2553328.800072531681508],
                [-1959167.025377874495462, 2696562.072923301719129],
                [-1766291.931527236010879, 2826657.96641160454601],
                [-1564811.640932813985273, 2942982.666421526577324],
                [-1355707.7453176996205, 3044969.4507075115107],
                [-1139998.977891495916992, 3132121.449890273623168],
                [-918736.249989229603671, 3204014.068229434546083],
                [-692997.531427837326191, 3260297.052093770354986],
                [-463882.598553028190508, 3300696.196443425025791],
                [-232507.676287271577166, 3325014.680708026047796],
                [0.000005823307071, 3333134.027676715515554],
                [232507.676298889826285, 3325014.680707213934511],
                [463882.598564561398234, 3300696.196441804524511],
                [692997.531439226237126, 3260297.052091349381953],
                [918736.249950131517835, 3204014.068240645807236],
                [1139998.977853274671361, 3132121.449904184788465],
                [1355707.745328336255625, 3044969.450702775269747],
                [1564811.640943094389513, 2942982.666416060645133],
                [1766291.931537112686783, 2826657.96640543313697],
                [1959167.02538729691878, 2696562.072916456032544],
                [2142497.254424097947776, 2553328.800065045710653],
                [2315389.452199894469231, 2397655.965958810877055],
                [2477001.305306143593043, 2230301.992997375782579],
                [2626545.457007427234203, 2052082.212955916067585],
                [2763293.343295966740698, 1863864.894608137197793],
                [2886578.742180571891367, 1666567.013848417671397],
                [2995801.019590451382101, 1461149.786119553493336],
                [3090428.055569022428244, 1248613.983506622724235],
                [3169998.836708850692958, 1029995.059075982775539],
                [3234125.702162631321698, 806358.102252014563419],
                [3282496.2322878674604, 578792.649808511836454],
                [3314874.77072396595031, 348407.377755034365691],
                [3331103.572484511416405, 116324.70003112575796],
                [3331103.572485930286348, -116324.699990478446125],
                [3314874.770722747780383, -348407.377766619436443],
                [3282496.232285844627768, -578792.649819983053021],
                [3234125.702159813605249, -806358.102263317327015],
                [3169998.836705251131207, -1029995.059087060857564],
                [3090428.055564659647644, -1248613.983517420012504],
                [2995801.019585346337408, -1461149.786130020162091],
                [2886578.742174749262631, -1666567.013858502265066],
                [2763293.343318709172308, -1863864.894574420759454],
                [2626545.457032464910299, -2052082.212923869024962],
                [2477001.30529834702611, -2230301.993006034288555],
                [2315389.452191513031721, -2397655.96596690453589],
                [2142497.254415172617882, -2553328.800072534941137],
                [1959167.025377869838849, -2696562.072923305444419],
                [1766291.931527237640694, -2826657.966411604080349],
                [1564811.640932811424136, -2942982.666421527974308],
                [1355707.745317697525024, -3044969.450707511976361],
                [1139998.977891490561888, -3132121.449890275485814],
                [918736.249989224597812, -3204014.068229435943067],
                [692997.531427832553163, -3260297.052093770820647],
                [463882.598553023708519, -3300696.196443425957114],
                [232507.676287265872816, -3325014.680708026513457],
                [-0.000005829470669, -3333134.027676715515554],
            ]
        ],
    }
    SUB_ANTARCTICA = {
        "type": "Polygon",
        "coordinates": [
            [
                [-0.000000000554096, -4524537.706531359814107],
                [-78745.356075052462984, -4523852.409917974844575],
                [-157466.858254153805319, -4521796.727670939639211],
                [-236140.659867285052314, -4518371.282506729476154],
                [-314742.928694089583587, -4513577.112076532095671],
                [-393249.854183247080073, -4507415.668651928193867],
                [-471637.654665271984413, -4499888.818684957921505],
                [-549882.584556555142626, -4490998.84224272146821],
                [-627960.941552483243868, -4480748.432316701859236],
                [-705849.073807455832139, -4469140.694006983190775],
                [-783523.387099601677619, -4456179.143581642769277],
                [-860960.351978048565798, -4441867.707411589100957],
                [-938136.510890563833527, -4426210.720781167037785],
                [-1015028.485289431060664, -4409212.926574889570475],
                [-1091612.982713378500193, -4390879.473840708844364],
                [-1167866.803843435831368, -4371215.916230239905417],
                [-1243766.849530577659607, -4350228.210316424258053],
                [-1319290.127793019870296, -4327922.713789138011634],
                [-1394413.760781053453684, -4304306.183529292233288],
                [-1469114.991707309149206, -4279385.773562006652355],
                [-1543371.19174033543095, -4253169.032889485359192],
                [-1617159.866859437897801, -4225663.9032042324543],
                [-1690458.664668661309406, -4196878.716483322903514],
                [-1763245.381167881423607, -4166822.192464443389326],
                [-1835497.967478932347149, -4135503.436004468705505],
                [-1907194.536524760071188, -4102931.934321378357708],
                [-1978313.369659554911777, -4069117.554120342247188],
                [-2048832.923247853759676, -4034070.53860486112535],
                [-2118731.835190633777529, -3997801.504373846575618],
                [-2187988.931396424770355, -3960321.438205590005964],
                [-2256583.232195453252643, -3921641.693729601800442],
                [-2324493.958694895263761, -3881773.987987321801484],
                [-2391700.539073319174349, -3840730.397882733028382],
                [-2458182.61481238855049, -3798523.356523977592587],
                [-2523920.04686395637691, -3755165.649457064922899],
                [-2588892.921750684268773, -3710670.410792807117105],
                [-2653081.557598298415542, -3665051.119228194933385],
                [-2716466.51009774254635, -3618321.593963345978409],
                [-2779028.578395314980298, -3570495.99051534710452],
                [-2840748.810909078456461, -3521588.796430201269686],
                [-2901608.511069764383137, -3471614.826894188299775],
                [-2961589.242984419688582, -3420589.220245983917266],
                [-3020672.837021090555936, -3368527.433390889782459],
                [-3078841.395312839653343, -3315445.237118560355157],
                [-3136077.297179467044771, -3261358.711325632408261],
                [-3192363.204465223010629, -3206284.240144748706371],
                [-3247682.066790983546525, -3150238.506981384940445],
                [-3302017.126719202380627, -3093238.489460054785013],
                [-3355351.924830169882625, -3035301.454281359445304],
                [-3407670.304707962553948, -2976444.95199148543179],
                [-3458956.417834625579417, -2916686.811665714718401],
                [-3509194.728391063399613, -2856045.135507565457374],
                [-3558370.017963235266507, -2794538.293365181889385],
                [-3606467.390152174513787, -2732184.917166665196419],
                [-3653472.275086477398872, -2669003.895276005845517],
                [-3699370.433835872448981, -2605014.366771332919598],
                [-3744147.96272453898564, -2540235.715647219214588],
                [-3787791.297542876563966, -2474687.564942797645926],
                [-3830287.217656428925693, -2408389.770797457545996],
                [-3871622.850010741967708, -2341362.416435942519456],
                [-3911785.67303092777729, -2273625.806084639392793],
                [-3950763.520414754748344, -2205200.458820937667042],
                [-3988544.584818115923554, -2136107.102357497438788],
                [-4025117.421431767754257, -2066366.666763315908611],
                [-4060470.951448239851743, -1996000.278123499127105],
                [-4094594.46541787404567, -1925029.252139657037333],
                [-4127477.626492987852544, -1853475.087672847090289],
                [-4159110.473559148143977, -1781359.460231051314622],
                [-4189483.424252641387284, -1708704.215403127484024],
                [-4218587.277863210998476, -1635531.362241249764338],
                [-4246413.218121169134974, -1561863.066593829076737],
                [-4272952.815868061967194, -1487721.644390932517126],
                [-4298198.031610075384378, -1413129.554884240264073],
                [-4322141.217953386716545, -1338109.393843595171347],
                [-4344775.121920751407743, -1262683.886712179053575],
                [-4366092.887148605659604, -1186875.881722435820848],
                [-4386088.055964028462768, -1110708.342974764993414],
                [-4404754.571340925991535, -1034204.343481149757281],
                [-4422086.778734855353832, -957387.05817576427944],
                [-4438079.427795927040279, -880279.756894728168845],
                [-4452727.673959255218506, -802905.797327096108347],
                [-4466027.079912506975234, -725288.617939246352762],
                [-4477973.616940065287054, -647451.730874793836847],
                [-4488563.666143426671624, -569418.714832188910805],
                [-4497794.019537454470992, -491213.207922147470526],
                [-4505661.881022158078849, -412858.900507094978821],
                [-4512164.867229697294533, -334379.528024769446347],
                [-4517301.008246364071965, -255798.863798180304002],
                [-4521068.74820931814611, -177140.711834084766451],
                [-4523466.945777894929051, -98428.899612175489892],
                [-4524494.874479345045984, -19687.270867146842647],
                [-4524152.222928903065622, 59060.321634147119767],
                [-4522439.094924109987915, 137790.023318291612668],
                [-4519356.009413375519216, 216477.985031430260278],
                [-4514903.900338769890368, 295100.370263774646446],
                [-4509084.116353115998209, 373633.362370246672072],
                [-4501898.420411443337798, 452053.171785146230832],
                [-4493348.989236950874329, 530336.043228569207713],
                [-4483438.412661620415747, 608458.262902485905215],
                [-4472169.692841696552932, 686396.165674211457372],
                [-4459546.243348259478807, 764126.142245171242394],
                [-4445571.888133167289197, 841624.646302715991624],
                [-4430250.860370689071715, 918868.201652898802422],
                [-4413587.801175177097321, 995833.409331964678131],
                [-4395587.758195153437555, 1072496.954694493440911],
                [-4376256.184084258042276, 1148835.61447595548816],
                [-4355598.934849502518773, 1224826.263827625894919],
                [-4333622.268077346496284, 1300445.883321646600962],
                [-4310332.841038117185235, 1375671.565924195805565],
                [-4285737.708669364452362, 1450480.523934575263411],
                [-4259844.32143874373287, 1524850.095888164825737],
                [-4232660.523087086156011, 1598757.753421139670536],
                [-4204194.548252341337502, 1672181.108094859169796],
                [-4174455.019975100643933, 1745097.918177872430533],
                [-4143450.947086467407644, 1817486.095383486943319],
                [-4111191.721479065250605, 1889323.711560848867521],
                [-4077687.115261998958886, 1960589.005337512586266],
                [-4042947.277800644282252, 2031260.388711505802348],
                [-4006982.732642161659896, 2101316.453590847086161],
                [-3969804.374327647965401, 2170735.978278595488518],
                [-3931423.465091922320426, 2239497.933901409152895],
                [-3891851.631451909895986, 2307581.490779722575098],
                [-3851100.860684695187956, 2374966.024737544357777],
                [-3809183.497196274809539, 2441631.123350047506392],
                [-3766112.238782138563693, 2507556.592126974835992],
                [-3721900.132780792657286, 2572722.460630056913942],
                [-3676560.572121409699321, 2637108.98852253658697],
                [-3630107.2912667687051, 2700696.671549009624869],
                [-3582554.362052768934518, 2763466.247443730942905],
                [-3533916.189425707329065, 2825398.701765636447817],
                [-3484207.507078676484525, 2886475.273658282123506],
                [-3433443.372988375835121, 2946677.461532949004322],
                [-3381639.164853662718087, 3005987.028673258144408],
                [-3328810.575437294784933, 3064386.008759485092014],
                [-3274973.607812184374779, 3121856.711311027873307],
                [-3220144.570513693150133, 3178381.727045265492052],
                [-3164340.07259937049821, 3233943.9331512642093],
                [-3107577.018617664929479, 3288526.498476697131991],
                [-3049872.603487121406943, 3342112.888626406900585],
                [-2991244.30728761991486, 3394686.870971087366343],
                [-2931709.889965232927352, 3446232.519564531743526],
                [-2871287.385952302720398, 3496734.219968005083501],
                [-2809995.098704362288117, 3546176.673980234656483],
                [-2747851.595155591145158, 3594544.904271600302309],
                [-2684875.700094407889992, 3641824.258921155240387],
                [-2621086.49046101141721, 3688000.415855024475604],
                [-2556503.289568480569869, 3733059.387184939812869],
                [-2491145.661249288357794, 3776987.523445490282029],
                [-2425033.403928934596479, 3819771.517728894948959],
                [-2358186.544628511182964, 3861398.409715980291367],
                [-2290625.332898036111146, 3901855.589602185413241],
                [-2222370.234682358801365, 3941130.801917381118983],
                [-2153441.926121541298926, 3979212.149238339159638],
                [-2083861.287287526531145, 4016088.095792774111032],
                [-2013649.39585907314904, 4051747.47095379140228],
                [-1942827.520736800972372, 4086179.472623755224049],
                [-1871417.11560032190755, 4119373.670506504364312],
                [-1799439.81240939674899, 4151320.009266943205148],
                [-1726917.414851082954556, 4182008.811577052343637],
                [-1653871.891734865494072, 4211430.78104738611728],
                [-1580325.370337768224999, 4239577.005043174140155],
                [-1506300.129701450234279, 4266438.957384184934199],
                [-1431818.593883361667395, 4292008.500927501358092],
                [-1356903.325163914822042, 4316277.890032472088933],
                [-1281577.017211852362379, 4339239.772907049395144],
                [-1205862.488209748407826, 4360887.193834834732115],
                [-1129782.673941843444481, 4381213.595282119698822],
                [-1053360.620846225647256, 4400212.819884342141449],
                [-976619.4790335012367, 4417879.112311289645731],
                [-899582.495274059823714, 4434207.121010537259281],
                [-822273.005956058739685, 4449191.899828556925058],
                [-744714.430016261758283, 4462828.909509035758674],
                [-666930.261845860630274, 4475114.019067924469709],
                [-588944.064173463382758, 4486043.507044810801744],
                [-510779.460927357024048, 4495614.06263024546206],
                [-432460.13007924403064, 4503822.786668665707111],
                [-354009.796471605193801, 4510667.192536620423198],
                [-275452.224630863871425, 4516145.206896027550101],
                [-196811.211568529368378, 4520255.170322244986892],
                [-118110.57957250160689, 4522995.837806741707027],
                [-39374.168990706202749, 4524366.379134248942137],
                [39374.168990706202749, 4524366.379134248942137],
                [118110.579572499365895, 4522995.837806741707027],
                [196811.211568531609373, 4520255.170322244986892],
                [275452.224630863871425, 4516145.206896027550101],
                [354009.796471605193801, 4510667.192536620423198],
                [432460.130079241818748, 4503822.786668665707111],
                [510779.460927357024048, 4495614.06263024546206],
                [588944.064173463382758, 4486043.507044810801744],
                [666930.261845860630274, 4475114.019067924469709],
                [744714.430016259546392, 4462828.909509036689997],
                [822273.005956060951576, 4449191.899828556925058],
                [899582.495274062152021, 4434207.121010537259281],
                [976619.4790335012367, 4417879.112311289645731],
                [1053360.620846225647256, 4400212.819884342141449],
                [1129782.673941841349006, 4381213.595282119698822],
                [1205862.488209748407826, 4360887.193834834732115],
                [1281577.017211852362379, 4339239.772907049395144],
                [1356903.325163914822042, 4316277.890032472088933],
                [1431818.593883359339088, 4292008.500927501358092],
                [1506300.129701452562585, 4266438.957384184002876],
                [1580325.370337768224999, 4239577.005043174140155],
                [1653871.891734865494072, 4211430.78104738611728],
                [1726917.414851082954556, 4182008.811577052343637],
                [1799439.812409398844466, 4151320.009266942273825],
                [1871417.11560032190755, 4119373.670506504364312],
                [1942827.520736800972372, 4086179.472623755224049],
                [2013649.395859071286395, 4051747.470953793264925],
                [2083861.287287526531145, 4016088.095792774111032],
                [2153441.926121541298926, 3979212.149238339159638],
                [2222370.234682358801365, 3941130.801917381118983],
                [2290625.332898034714162, 3901855.589602186344564],
                [2358186.544628513045609, 3861398.409715978894383],
                [2425033.403928936459124, 3819771.517728893086314],
                [2491145.661249288357794, 3776987.523445490282029],
                [2556503.289568480569869, 3733059.387184939812869],
                [2621086.490461010020226, 3688000.415855025872588],
                [2684875.700094409286976, 3641824.258921153843403],
                [2747851.595155591145158, 3594544.904271600302309],
                [2809995.098704362288117, 3546176.673980234656483],
                [2871287.385952300857753, 3496734.219968006480485],
                [2931709.889965234789997, 3446232.519564529880881],
                [2991244.30728761991486, 3394686.870971087366343],
                [3049872.603487121406943, 3342112.888626406900585],
                [3107577.018617664929479, 3288526.498476697131991],
                [3164340.07259937049821, 3233943.9331512642093],
                [3220144.570513693150133, 3178381.727045265492052],
                [3274973.607812184374779, 3121856.711311027873307],
                [3328810.575437292456627, 3064386.008759486954659],
                [3381639.164853664115071, 3005987.028673256281763],
                [3433443.372988375835121, 2946677.461532949004322],
                [3484207.507078676484525, 2886475.273658282123506],
                [3533916.189425706397742, 2825398.701765638776124],
                [3582554.362052770331502, 2763466.24744372908026],
                [3630107.291266769636422, 2700696.671549008693546],
                [3676560.572121409699321, 2637108.98852253658697],
                [3721900.132780793122947, 2572722.460630056448281],
                [3766112.238782137166709, 2507556.592126976698637],
                [3809183.497196275275201, 2441631.123350047040731],
                [3851100.86068469658494, 2374966.024737543426454],
                [3891851.631451909895986, 2307581.490779722575098],
                [3931423.465091922320426, 2239497.933901410084218],
                [3969804.374327649828047, 2170735.978278592228889],
                [4006982.732642161659896, 2101316.453590847086161],
                [4042947.277800644282252, 2031260.388711505802348],
                [4077687.115261997561902, 1960589.005337514448911],
                [4111191.721479065250605, 1889323.711560847004876],
                [4143450.947086467407644, 1817486.095383486943319],
                [4174455.019975099246949, 1745097.918177874526009],
                [4204194.548252341337502, 1672181.108094860333949],
                [4232660.523087087087333, 1598757.753421138972044],
                [4259844.321438744664192, 1524850.095888164127246],
                [4285737.708669364452362, 1450480.523934575263411],
                [4310332.841038117185235, 1375671.565924196736887],
                [4333622.268077346496284, 1300445.883321644505486],
                [4355598.934849502518773, 1224826.263827624730766],
                [4376256.184084258042276, 1148835.61447595548816],
                [4395587.758195153437555, 1072496.954694494372234],
                [4413587.801175176165998, 995833.409331968636252],
                [4430250.860370689071715, 918868.201652898802422],
                [4445571.888133167289197, 841624.646302715060301],
                [4459546.243348259478807, 764126.142245171242394],
                [4472169.692841696552932, 686396.165674215415493],
                [4483438.412661620415747, 608458.262902485905215],
                [4493348.989236950874329, 530336.043228567112237],
                [4501898.420411443337798, 452053.171785145241302],
                [4509084.116353115998209, 373633.362370248651132],
                [4514903.900338769890368, 295100.370263773598708],
                [4519356.009413375519216, 216477.985031432268443],
                [4522439.094924109987915, 137790.023318290623138],
                [4524152.222928903065622, 59060.321634148131125],
                [4524494.874479345045984, -19687.270867148850812],
                [4523466.945777894929051, -98428.899612175489892],
                [4521068.74820931814611, -177140.711834082729183],
                [4517301.008246364071965, -255798.863798180304002],
                [4512164.867229696363211, -334379.528024772473145],
                [4505661.881022158078849, -412858.900507095910143],
                [4497794.019537454470992, -491213.207922146364581],
                [4488563.666143426671624, -569418.714832189842127],
                [4477973.616940066218376, -647451.730874792789109],
                [4466027.079912506975234, -725288.6179392474005],
                [4452727.673959255218506, -802905.797327095060609],
                [4438079.427795927040279, -880279.756894726189785],
                [4422086.778734855353832, -957387.05817576427944],
                [4404754.571340925060213, -1034204.343481151503511],
                [4386088.055964028462768, -1110708.342974764993414],
                [4366092.887148606590927, -1186875.881722433958203],
                [4344775.121920751407743, -1262683.886712181149051],
                [4322141.217953385785222, -1338109.393843598198146],
                [4298198.031610074453056, -1413129.554884243290871],
                [4272952.815868061967194, -1487721.644390932517126],
                [4246413.218121169134974, -1561863.066593826981261],
                [4218587.277863210998476, -1635531.362241249764338],
                [4189483.424252641387284, -1708704.215403129346669],
                [4159110.473559148143977, -1781359.460231051314622],
                [4127477.626492988783866, -1853475.087672846158966],
                [4094594.46541787404567, -1925029.252139658899978],
                [4060470.951448239851743, -1996000.278123499127105],
                [4025117.421431767754257, -2066366.666763315908611],
                [3988544.584818116854876, -2136107.102357495576143],
                [3950763.520414756145328, -2205200.458820934407413],
                [3911785.67303092777729, -2273625.806084639392793],
                [3871622.850010741502047, -2341362.416435944382101],
                [3830287.217656428925693, -2408389.770797457545996],
                [3787791.297542878892273, -2474687.564942794386297],
                [3744147.96272453898564, -2540235.715647219214588],
                [3699370.433835873380303, -2605014.366771331056952],
                [3653472.275086477398872, -2669003.895276005845517],
                [3606467.390152174513787, -2732184.917166665196419],
                [3558370.017963233869523, -2794538.293365183286369],
                [3509194.728391063399613, -2856045.135507565457374],
                [3458956.417834624182433, -2916686.811665716115385],
                [3407670.304707962553948, -2976444.95199148543179],
                [3355351.924830168485641, -3035301.454281360842288],
                [3302017.126719202380627, -3093238.489460054785013],
                [3247682.066790984943509, -3150238.506981383543462],
                [3192363.204465223010629, -3206284.240144748706371],
                [3136077.297179464250803, -3261358.711325634270906],
                [3078841.395312839653343, -3315445.237118560355157],
                [3020672.837021090555936, -3368527.433390889782459],
                [2961589.242984421085566, -3420589.220245982520282],
                [2901608.511069764383137, -3471614.826894188299775],
                [2840748.810909076128155, -3521588.79643020266667],
                [2779028.578395314980298, -3570495.99051534710452],
                [2716466.510097744408995, -3618321.593963344581425],
                [2653081.557598298415542, -3665051.119228194933385],
                [2588892.921750681009144, -3710670.410792809911072],
                [2523920.04686395637691, -3755165.649457064922899],
                [2458182.61481238855049, -3798523.356523977592587],
                [2391700.539073320571333, -3840730.397882731631398],
                [2324493.958694895263761, -3881773.987987321801484],
                [2256583.232195449993014, -3921641.693729603197426],
                [2187988.931396424770355, -3960321.438205590005964],
                [2118731.835190635640174, -3997801.504373845178634],
                [2048832.923247853759676, -4034070.53860486112535],
                [1978313.369659558404237, -4069117.554120340384543],
                [1907194.536524760071188, -4102931.934321378357708],
                [1835497.967478932347149, -4135503.436004468705505],
                [1763245.381167879560962, -4166822.192464444320649],
                [1690458.664668661309406, -4196878.716483322903514],
                [1617159.866859435802326, -4225663.9032042324543],
                [1543371.19174033543095, -4253169.032889485359192],
                [1469114.991707311244681, -4279385.773562005721033],
                [1394413.760781053453684, -4304306.183529292233288],
                [1319290.127793021500111, -4327922.713789138011634],
                [1243766.849530577659607, -4350228.210316424258053],
                [1167866.803843437926844, -4371215.916230238974094],
                [1091612.982713376404718, -4390879.473840709775686],
                [1015028.485289431060664, -4409212.926574889570475],
                [938136.510890561854467, -4426210.720781167037785],
                [860960.351978048565798, -4441867.707411589100957],
                [783523.387099599698558, -4456179.1435816437006],
                [705849.073807455832139, -4469140.694006983190775],
                [627960.941552485106513, -4480748.432316700927913],
                [549882.584556555142626, -4490998.84224272146821],
                [471637.654665269947145, -4499888.818684957921505],
                [393249.854183247080073, -4507415.668651928193867],
                [314742.928694089583587, -4513577.112076532095671],
                [236140.659867287031375, -4518371.282506728544831],
                [157466.858254153805319, -4521796.727670939639211],
                [78745.356075050440268, -4523852.409917974844575],
                [0.000000000554096, -4524537.706531359814107],
                [0.000000000540494, -4413472.474181111901999],
                [0.000000000526967, -4303012.154072260484099],
                [0.000000000513511, -4193138.147329211235046],
                [0.000000000500125, -4083832.214625354856253],
                [0.000000000486806, -3975076.462474949192256],
                [0.000000000473553, -3866853.329973086714745],
                [0.000000000460363, -3759145.575962432660162],
                [0.000000000447233, -3651936.266606318764389],
                [0.000000000434163, -3545208.763348782900721],
                [0.00000000042115, -3438946.711242982186377],
                [0.000000000408191, -3333134.027630276978016],
                [58010.08696926873381, -3332629.183686155360192],
                [116002.601263174103224, -3331114.804783378727734],
                [173959.975529548537452, -3328591.349664387758821],
                [231864.653060994140105, -3325059.58274550922215],
                [289699.093113247014116, -3320520.573885394725949],
                [347445.776218697894365, -3314975.698060940019786],
                [405087.209493462054525, -3308426.63495076354593],
                [462605.931936401640996, -3300875.368426394183189],
                [519984.519718498457223, -3292324.185951309744269],
                [577205.591460951720364, -3282775.677888004109263],
                [634251.813500426127575, -3272232.736713306047022],
                [691105.905139837530442, -3260698.556142176967114],
                [747750.643883106997237, -3248176.630160255357623],
                [804168.870652271667495, -3234670.751965442672372],
                [860343.494985389756039, -3220185.012818853370845],
                [916257.50021366099827, -3204723.800805467646569],
                [971893.948616191628389, -3188291.799504878930748],
                [1027235.986550843925215, -3170893.986572516616434],
                [1082266.849559620022774, -3152535.632231795229018],
                [1136969.867447020020336, -3133222.297677638009191],
                [1191328.469329860061407, -3112959.833391848485917],
                [1245326.188656993908808, -3091754.377370860427618],
                [1298946.668197437189519, -3069612.353266387246549],
                [1352173.664995366707444, -3046540.468439542688429],
                [1404991.055290517164394, -3022545.711929013952613],
                [1457382.839402462122962, -2997635.352333910763264],
                [1509333.146577303297818, -2971816.935611930675805],
                [1560826.239795311586931, -2945098.282793506514281],
                [1611846.520538060925901, -2917487.487612616736442],
                [1662378.533513597911224, -2888992.914055002387613],
                [1712406.971338224364445, -2859623.193824508227408],
                [1761916.679173486307263, -2829387.223728329874575],
                [1810892.65931693976745, -2798294.16298195021227],
                [1859320.075745322741568, -2766353.430434592999518],
                [1907184.258608756354079, -2733574.701716014649719],
                [1954470.708674584049731, -2699967.906305535696447],
                [2001165.101719568483531, -2665543.224524138029665],
                [2047253.29286903119646, -2630311.084450609516352],
                [2092721.320881684776396, -2594282.158762621693313],
                [2137555.41237884061411, -2557467.361503711435944],
                [2181741.986016697250307, -2519877.844777152873576],
                [2225267.656600465066731, -2481524.995367708615959],
                [2268119.239139061886817, -2442420.431292301975191],
                [2310283.752839185297489, -2402575.998280621133745],
                [2351748.425037491135299, -2362003.766186775639653],
                [2392500.695069768466055, -2320716.025333023630083],
                [2432528.218075851444155, -2278725.282786750234663],
                [2471818.868739196565002, -2236044.258571754675359],
                [2510360.744959927164018, -2192685.881815056316555],
                [2548142.171460277400911, -2148663.2868303344585],
                [2585151.703321310691535, -2103989.809139235876501],
                [2621378.129449878353626, -2058678.981431706110016],
                [2656810.475974724628031, -2012744.529466615524143],
                [2691438.009570743888617, -1966200.367913886439055],
                [2725250.240710367914289, -1919060.596139400498942],
                [2758236.926841100677848, -1871339.493933955207467],
                [2790388.07548823626712, -1823051.517187566729262],
                [2821693.947281827684492, -1774211.293510423507541],
                [2852145.058906975202262, -1724833.617801832268015],
                [2881732.185976560693234, -1674933.447768472833559],
                [2910446.365825536195189, -1624525.899393346626312],
                [2938278.900225935038179, -1573626.242356775095686],
                [2965221.358021778520197, -1522249.89541083923541],
                [2991265.57768308185041, -1470412.421708661830053],
                [3016403.669778174720705, -1418129.524089951068163],
                [3040628.019363612402231, -1365417.040324218571186],
                [3063931.288290918804705, -1312290.938313140068203],
                [3086306.41742950072512, -1258767.311253478284925],
                [3107746.628805024549365, -1204862.372762065846473],
                [3128245.427652632817626, -1150592.451964303618297],
                [3147796.604384366888553, -1095973.988547666696832],
                [3166394.236470205243677, -1041023.527781719923951],
                [3184032.690232140012085, -985757.71550615481101],
                [3200706.62255075853318, -930193.293088350328617],
                [3216410.982483801431954, -874347.092352012638003],
                [3231141.012796221766621, -818236.030478392378427],
                [3244892.251401263289154, -761877.104881667182781],
                [3257660.532712138723582, -705287.388060000259429],
                [3269441.988903884775937, -648484.022423869697377],
                [3280233.051085025537759, -591484.21510320622474],
                [3290030.450378673151135, -534305.232734937220812],
                [3298831.218912752810866, -476964.396232496714219],
                [3306632.690719043370336, -419479.075538899982348],
                [3313432.50254076346755, -361866.684364958258811],
                [3319228.594548459164798, -304144.674914244329557],
                [3324019.210963981691748, -246330.532596388249658],
                [3327802.900592348538339, -188441.770730321557494],
                [3330578.517261352855712, -130495.925239062838955],
                [3332345.220168759580702, -72510.549337660326273],
                [3333102.474137013312429, -14503.208215888156701],
                [3332850.049775348510593, 43508.526282672937668],
                [3331588.023549283389002, 101507.080983585998183],
                [3329316.77775745652616, 159474.886704891425325],
                [3326037.000415814109147, 217394.383579254994402],
                [3321749.685049199033529, 275248.02637326659169],
                [3316456.130390383768827, 333018.289802337938454],
                [3310157.939986654557288, 390687.673839522525668],
                [3302857.021714057773352, 448238.70901671925094],
                [3294555.587199456058443, 505653.961716587131377],
                [3285256.151150572579354, 562916.039453627774492],
                [3274961.530594226904213, 620007.596142780850641],
                [3263674.84402298508212, 676911.337353994022124],
                [3251399.510450496338308, 733610.025551112135872],
                [3238139.248375783674419, 790086.485313565586694],
                [3223898.074656827840954, 846323.608539210865274],
                [3208680.303293752949685, 902304.359626806573942],
                [3192490.544122020248324, 958011.780636499752291],
                [3175333.701415989082307, 1013428.996426816098392],
                [3157214.972403294872493, 1068539.219766543246806],
                [3138139.845690480433404, 1123325.756419995799661],
                [3118114.099600365385413, 1177772.010204110527411],
                [3097143.800421646796167, 1231861.48801583587192],
                [3075235.30057127494365, 1285577.804828292457387],
                [3052395.236670149024576, 1338904.688654206227511],
                [3028630.527532722335309, 1391825.985475085908547],
                [3003948.372071127407253, 1444325.664134669816121],
                [2978356.247114441823214, 1496387.821195163298398],
                [2951861.905143782496452, 1547996.685754769016057],
                [2924473.371943882200867, 1599136.624225088628009],
                [2896198.944171887822449, 1649792.145066909724846],
                [2867047.186844095122069, 1699947.903482984285802],
                [2837026.930741401389241, 1749588.706066329265013],
                [2806147.269734241534024, 1798699.515402695862576],
                [2774417.55802783370018, 1847265.454625757643953],
                [2741847.407328557223082, 1895271.811923688277602],
                [2708446.68393234256655, 1942704.044995719334111],
                [2674225.505735914222896, 1989547.785457369638607],
                [2639194.239171845838428, 2035788.843192969914526],
                [2603363.496068304404616, 2081413.210654205176979],
                [2566744.130434467457235, 2126407.067103342618793],
                [2529347.235172590240836, 2170756.782799861393869],
                [2491184.138717676047236, 2214448.923129251692444],
                [2452266.40160583332181, 2257470.252672669943422],
                [2412605.8129722927697, 2299807.739216288086027],
                [2372214.38698019599542, 2341448.557699058670551],
                [2331104.359181213658303, 2382380.094097744207829],
                [2289288.18280909024179, 2422589.949248011689633],
                [2246778.525007261428982, 2462065.942600446287543],
                [2203588.262991663068533, 2500796.115910334978253],
                [2159730.480149906594306, 2538768.736860111355782],
                [2115218.462078005075455, 2575972.302613363135606],
                [2070065.692555830581114, 2612395.543299326207489],
                [2024285.849462564336136, 2648027.425426793750376],
                [1977892.800633311970159, 2682857.155226447619498],
                [1930900.599658217513934, 2716874.181920533534139],
                [1883323.481625266838819, 2750068.200918972492218],
                [1835175.85880814678967, 2782429.156940857414156],
                [1786472.316300406819209, 2813947.247060451190919],
                [1737227.607597278896719, 2844612.923676728736609],
                [1687456.650126484455541, 2874416.897405582480133],
                [1637174.520729372976348, 2903350.139893806073815],
                [1586396.451093791984022, 2931403.886553992517292],
                [1535137.823140019318089, 2958569.639219559263438],
                [1483414.16436121519655, 2984839.168719036504626],
                [1431241.143119759391993, 3010204.517368890345097],
                [1378634.563900921493769, 3034658.001384100876749],
                [1325610.362525295000523, 3058192.21320576639846],
                [1272184.601321447873488, 3080800.023745036683977],
                [1218373.464260249631479, 3102474.584542685654014],
                [1164193.25205235206522, 3123209.329843678511679],
                [1109660.377210297621787, 3142997.978586103301495],
                [1054791.359076784225181, 3161834.536303850356489],
                [999602.818820527987555, 3179713.296942490618676],
                [944111.474401330924593, 3196628.844587774947286],
                [888334.135505776852369, 3212576.055106249637902],
                [832287.698455183766782, 3227550.097697478719056],
                [775989.141087286057882, 3241546.436357413418591],
                [719455.517613234580494, 3254560.831252460367978],
                [662703.953451457200572, 3266589.340003827121109],
                [605751.64003995177336, 3277628.318881770130247],
                [548615.829628578620031, 3287674.423909365199506],
                [491313.830052926961798, 3296724.611875482834876],
                [433862.99949135677889, 3304776.14125665044412],
                [376280.741206775302999, 3311826.573047524318099],
                [318584.49827476387145, 3317873.771499731112272],
                [260791.748299644241342, 3322915.904768829233944],
                [202919.998120083648246, 3326951.445469226688147],
                [144986.778505845140899, 3329979.171136860270053],
                [87009.638847289825208, 3331998.164599508978426],
                [29006.141839228832396, 3333007.81425463128835],
                [-29006.141839228832396, 3333007.81425463128835],
                [-87009.638847288166289, 3331998.164599508978426],
                [-144986.778505846799817, 3329979.171136860270053],
                [-202919.998120083648246, 3326951.445469226688147],
                [-260791.748299644241342, 3322915.904768829233944],
                [-318584.498274762241635, 3317873.771499731112272],
                [-376280.741206775302999, 3311826.573047524318099],
                [-433862.99949135677889, 3304776.14125665044412],
                [-491313.830052926961798, 3296724.611875482834876],
                [-548615.829628576873802, 3287674.423909366130829],
                [-605751.64003995328676, 3277628.318881770130247],
                [-662703.953451458946802, 3266589.340003827121109],
                [-719455.517613234580494, 3254560.831252460367978],
                [-775989.141087286057882, 3241546.436357413418591],
                [-832287.698455182136968, 3227550.097697478719056],
                [-888334.135505776852369, 3212576.055106249637902],
                [-944111.474401330924593, 3196628.844587774947286],
                [-999602.818820527987555, 3179713.296942490618676],
                [-1054791.359076782362536, 3161834.536303850822151],
                [-1109660.377210299251601, 3142997.978586102370173],
                [-1164193.25205235206522, 3123209.329843678511679],
                [-1218373.464260249631479, 3102474.584542685654014],
                [-1272184.601321447873488, 3080800.023745036683977],
                [-1325610.362525296397507, 3058192.21320576639846],
                [-1378634.563900921493769, 3034658.001384100876749],
                [-1431241.143119759391993, 3010204.517368890345097],
                [-1483414.164361213799566, 2984839.168719037435949],
                [-1535137.823140019318089, 2958569.639219559263438],
                [-1586396.451093791984022, 2931403.886553992517292],
                [-1637174.520729372976348, 2903350.139893806073815],
                [-1687456.650126483058557, 2874416.897405582945794],
                [-1737227.607597280759364, 2844612.923676727339625],
                [-1786472.316300408216193, 2813947.247060449793935],
                [-1835175.85880814678967, 2782429.156940857414156],
                [-1883323.481625266838819, 2750068.200918972492218],
                [-1930900.59965821611695, 2716874.181920533999801],
                [-1977892.800633313134313, 2682857.155226446222514],
                [-2024285.849462564336136, 2648027.425426793750376],
                [-2070065.692555830581114, 2612395.543299326207489],
                [-2115218.462078003678471, 2575972.302613364066929],
                [-2159730.480149908456951, 2538768.736860109493136],
                [-2203588.262991663068533, 2500796.115910334978253],
                [-2246778.525007261428982, 2462065.942600446287543],
                [-2289288.18280909024179, 2422589.949248011689633],
                [-2331104.359181213658303, 2382380.094097744207829],
                [-2372214.38698019599542, 2341448.557699058670551],
                [-2412605.8129722927697, 2299807.739216288086027],
                [-2452266.401605831924826, 2257470.252672671340406],
                [-2491184.138717676978558, 2214448.92312925029546],
                [-2529347.235172590240836, 2170756.782799861393869],
                [-2566744.130434467457235, 2126407.067103342618793],
                [-2603363.496068303473294, 2081413.210654206806794],
                [-2639194.239171847701073, 2035788.843192968750373],
                [-2674225.505735914688557, 1989547.785457368940115],
                [-2708446.68393234256655, 1942704.044995719334111],
                [-2741847.407328557688743, 1895271.81192368734628],
                [-2774417.558027831837535, 1847265.454625759040937],
                [-2806147.269734241999686, 1798699.515402695396915],
                [-2837026.930741402786225, 1749588.706066328566521],
                [-2867047.186844095122069, 1699947.903482984285802],
                [-2896198.944171887356788, 1649792.145066910423338],
                [-2924473.37194388313219, 1599136.624225086532533],
                [-2951861.905143782496452, 1547996.685754769016057],
                [-2978356.247114441823214, 1496387.821195163298398],
                [-3003948.37207112647593, 1444325.664134671213105],
                [-3028630.52753272280097, 1391825.985475084744394],
                [-3052395.236670149024576, 1338904.688654206227511],
                [-3075235.300571274477988, 1285577.804828294087201],
                [-3097143.800421646796167, 1231861.488015836570412],
                [-3118114.099600365851074, 1177772.01020411006175],
                [-3138139.845690481364727, 1123325.756419995333999],
                [-3157214.972403294872493, 1068539.219766543246806],
                [-3175333.701415989082307, 1013428.996426816913299],
                [-3192490.544122020713985, 958011.780636498355307],
                [-3208680.303293752949685, 902304.35962680587545],
                [-3223898.074656827840954, 846323.608539210865274],
                [-3238139.248375783674419, 790086.485313566285186],
                [-3251399.510450495872647, 733610.025551115046255],
                [-3263674.84402298508212, 676911.337353994022124],
                [-3274961.530594226904213, 620007.596142780152149],
                [-3285256.151150572579354, 562916.039453627774492],
                [-3294555.58719945512712, 505653.961716589983553],
                [-3302857.021714057773352, 448238.70901671925094],
                [-3310157.939986654557288, 390687.673839521012269],
                [-3316456.130390383768827, 333018.289802337181754],
                [-3321749.685049199033529, 275248.02637326810509],
                [-3326037.000415814109147, 217394.383579254237702],
                [-3329316.77775745652616, 159474.88670489290962],
                [-3331588.023549284320325, 101507.080983585241484],
                [-3332850.049775348510593, 43508.52628267367254],
                [-3333102.474137013312429, -14503.208215889637358],
                [-3332345.220168759580702, -72510.549337660326273],
                [-3330578.517261352855712, -130495.925239061354659],
                [-3327802.900592348538339, -188441.770730321557494],
                [-3324019.210963981691748, -246330.532596390461549],
                [-3319228.594548459164798, -304144.674914245028049],
                [-3313432.50254076346755, -361866.684364957502112],
                [-3306632.690719043370336, -419479.075538900739048],
                [-3298831.218912753276527, -476964.396232495899312],
                [-3290030.450378673151135, -534305.232734938035719],
                [-3280233.051085025537759, -591484.215103205526248],
                [-3269441.988903884775937, -648484.022423868183978],
                [-3257660.532712138723582, -705287.388060000259429],
                [-3244892.251401263289154, -761877.104881668463349],
                [-3231141.012796221766621, -818236.030478392378427],
                [-3216410.982483801431954, -874347.092352011124603],
                [-3200706.622550758067518, -930193.293088351842016],
                [-3184032.690232139080763, -985757.715506157139316],
                [-3166394.236470204312354, -1041023.527781722019427],
                [-3147796.604384366888553, -1095973.988547666696832],
                [-3128245.427652633283287, -1150592.451964302221313],
                [-3107746.628805024549365, -1204862.372762065846473],
                [-3086306.41742950072512, -1258767.311253479449078],
                [-3063931.288290918804705, -1312290.938313140068203],
                [-3040628.019363612867892, -1365417.040324217639863],
                [-3016403.669778174255043, -1418129.524089952232316],
                [-2991265.57768308185041, -1470412.421708661830053],
                [-2965221.358021778520197, -1522249.89541083923541],
                [-2938278.90022593550384, -1573626.242356773698702],
                [-2910446.365825537126511, -1624525.899393343832344],
                [-2881732.185976560693234, -1674933.447768472833559],
                [-2852145.058906974736601, -1724833.617801833664998],
                [-2821693.947281827684492, -1774211.293510423507541],
                [-2790388.075488237664104, -1823051.517187564168125],
                [-2758236.926841100677848, -1871339.493933955207467],
                [-2725250.240710369311273, -1919060.596139399567619],
                [-2691438.009570743888617, -1966200.367913886439055],
                [-2656810.475974724628031, -2012744.529466615524143],
                [-2621378.129449877422303, -2058678.981431707506999],
                [-2585151.703321310691535, -2103989.809139235876501],
                [-2548142.171460276003927, -2148663.286830335389823],
                [-2510360.744959927164018, -2192685.881815056316555],
                [-2471818.86873919563368, -2236044.258571756072342],
                [-2432528.218075851444155, -2278725.282786750234663],
                [-2392500.695069768931717, -2320716.02533302269876],
                [-2351748.425037491135299, -2362003.766186775639653],
                [-2310283.752839182969183, -2402575.99828062299639],
                [-2268119.239139061886817, -2442420.431292301975191],
                [-2225267.656600465066731, -2481524.995367708615959],
                [-2181741.986016698181629, -2519877.844777151476592],
                [-2137555.41237884061411, -2557467.361503711435944],
                [-2092721.320881683379412, -2594282.158762622624636],
                [-2047253.29286903119646, -2630311.084450609516352],
                [-2001165.101719569414854, -2665543.224524137564003],
                [-1954470.708674584049731, -2699967.906305535696447],
                [-1907184.258608753792942, -2733574.701716016512364],
                [-1859320.075745322741568, -2766353.430434592999518],
                [-1810892.65931693976745, -2798294.16298195021227],
                [-1761916.679173487471417, -2829387.223728328477591],
                [-1712406.971338224364445, -2859623.193824508227408],
                [-1662378.533513595117256, -2888992.914055003318936],
                [-1611846.520538060925901, -2917487.487612616736442],
                [-1560826.239795312983915, -2945098.282793505582958],
                [-1509333.146577303297818, -2971816.935611930675805],
                [-1457382.839402464684099, -2997635.35233390936628],
                [-1404991.055290517164394, -3022545.711929013952613],
                [-1352173.664995366707444, -3046540.468439542688429],
                [-1298946.668197435792536, -3069612.353266388177872],
                [-1245326.188656993908808, -3091754.377370860427618],
                [-1191328.469329858431593, -3112959.833391848951578],
                [-1136969.867447020020336, -3133222.297677638009191],
                [-1082266.849559621652588, -3152535.632231794763356],
                [-1027235.986550843925215, -3170893.986572516616434],
                [-971893.948616192908958, -3188291.799504878465086],
                [-916257.50021366099827, -3204723.800805467646569],
                [-860343.494985391385853, -3220185.012818852439523],
                [-804168.870652270037681, -3234670.751965442672372],
                [-747750.643883106997237, -3248176.630160255357623],
                [-691105.905139836017042, -3260698.556142176967114],
                [-634251.813500426127575, -3272232.736713306047022],
                [-577205.591460950206965, -3282775.677888004109263],
                [-519984.519718498457223, -3292324.185951309744269],
                [-462605.931936403154396, -3300875.368426393717527],
                [-405087.209493462054525, -3308426.63495076354593],
                [-347445.776218696439173, -3314975.698060940019786],
                [-289699.093113247014116, -3320520.573885394725949],
                [-231864.653060994140105, -3325059.58274550922215],
                [-173959.975529550021747, -3328591.349664387758821],
                [-116002.601263174103224, -3331114.804783378727734],
                [-58010.086969267249515, -3332629.183686155360192],
                [-0.000000000408191, -3333134.027630276978016],
                [-0.00000000042115, -3438946.711242982186377],
                [-0.000000000434163, -3545208.763348782900721],
                [-0.000000000447233, -3651936.266606318764389],
                [-0.000000000460363, -3759145.575962432660162],
                [-0.000000000473553, -3866853.329973086714745],
                [-0.000000000486806, -3975076.462474949192256],
                [-0.000000000500125, -4083832.214625354856253],
                [-0.000000000513511, -4193138.147329211235046],
                [-0.000000000526967, -4303012.154072260484099],
                [-0.000000000540494, -4413472.474181111901999],
                [-0.000000000554096, -4524537.706531359814107],
            ]
        ],
    }


class RecordRetractBeforeDeleteException(Exception):
    """
    Represents a situation whereby a record is deleted before it has been first been retracted

    This is illogical as published records must have an unpublished counterpart. If this unpublished counterpart is
    removed (deleted) this rule would be violated. Instead the published record must be removed (retracted) first.
    """

    pass


class CollectionInsertConflictException(Exception):
    """
    Represents a situation where a collection to be inserted already exists in a set of collections

    Collections must be unique. If a collection is inserted into a set with the same identifier as an existing
    collection, neither collection would be unique and this rule would be violated. Collections may be updated instead.
    """

    pass


class CollectionNotFoundException(Exception):
    """
    Represents a situation where a given collection does not exist
    """

    pass


class RecordSummary:
    """
    Records represent and describe a given resource, often in great detail using a conceptual model. These full
    representations (represented by the Record class) are inherently large and complex and so unwieldy in large numbers,
    such as indexes.

    Record summaries also represent and describe a resource but in far less detail. As RecordSummaries are simpler, they
    can be processed more easily when in greater numbers than Records.

    Record summaries are effectively subsets of Records, however to leverage inheritance, Records inherit from and
    extend RecordSummaries. Collections of record summaries are typically held in a repository (represented by the
    Repository class).

    Record summaries are created from a configuration dictionary. Properties are defined to access specific parts of
    this configuration. Record summaries are intended to be read-only objects.

    To ensure RecordSummaries remain lightweight, properties should be strictly limited, with anything non-essential
    added to the Record class instead.
    """

    def __init__(self, config: dict = None):
        """
        :type config dict
        :param config: Record configuration
        """
        self.config = {}
        if config is not None:
            self.config = config

    def __repr__(self):
        return f"<RecordSummary / {self.identifier} / {self.title}>"

    @property
    def identifier(self) -> str:
        return self.config["file_identifier"]

    @property
    def title(self) -> str:
        return self.config["resource"]["title"]["value"]


class Record(RecordSummary):
    """
    Records represent and describe a given resource, often in great detail using a conceptual model - currently assumed
    to be ISO 19115 (Geographic Information).

    As full representations are inherently large and complex, they are unwieldy in large numbers, such as indexes. In
    these circumstances, record summaries (represented by the RecordSummaries class), with an intentionally restricted
    set of properties can be processed and used more easily when in greater numbers compared to Records.

    Records are effectively supersets of RecordSummaries and so inherit from and extend them. Collections of records
    are typically held in a repository (represented by the Repository class).

    Records are created from a configuration dictionary. Properties are defined to access specific parts of this
    configuration, with additional processing performed as needed. Records are intended to be read-only.

    Properties currently track the ISO 19115 abstract model quite closely. As this catalogue evolves to cover other
    resources this may change.
    """

    def __repr__(self):
        return f"<Record / {self.identifier}>"

    def _process_resource_dates(self) -> Dict[str, Dict[str, Union[str, date, datetime]]]:
        """
        Processes resource dates into a dict, keyed by date type, with precision

        ISO allows multiple dates of the same type (e.g. multiple publication dates) however this doesn't make much
        sense and is harder to interact with. This method will restructure dates by their date type and stored with a
        default precision of 'day'. This precision will be overridden if less precise.

        E.g. This input:

        ```
        [
            {
                "date_type": "creation",
                "date": <date 2020-01-01>,
                "date_precision": "year"
            },
            {
                "date_type": "publication",
                "date": <datetime 2020-04-20T14:43:30>,
            }
        ]
        ```

        Will become:

        ```
        {
            'creation": {
                "date": <date 2020-01-01>,
                "date_precision": "year"
            },
            "publication": {
                "date": <datetime 2020-04-20T14:43:30>,
                "date_precision": "day"
            }
        ]
        ```

        :rtype dict
        :return: resource dates keyed by date type
        """
        resource_dates = {}
        for resource_date in self.config["resource"]["dates"]:
            _resource_date = {"value": resource_date["date"], "precision": "day"}
            try:
                _resource_date["precision"] = resource_date["date_precision"]
            except KeyError:
                pass
            resource_dates[resource_date["date_type"]] = _resource_date
        return resource_dates

    def _process_resource_contacts(self) -> Dict[str, List[dict]]:
        """
        Processes resource points of contact into a dict, key by role

        ISO allows multiple contacts to have the same role (e.g. multiple authors). The BAS Metadata Library config
        allows contacts to have multiple roles (e.g. publisher and distributor). This method will restructure contacts
        by their roles.

        E.g. This (simplified) input:

        ```
        [
            {
                "organisation_name": "MAGIC",
                "roles": [
                    "point of contact",
                    "distributor"
                ]
            },
            {
                "organisation_name": "Constance Watson",
                "roles": [
                    "author"
                ]
            },
            {
                "organisation_name": "John Cinnamon",
                "roles": [
                    "author"
                ]
            }
        ]
        ```

        Will become:

        ```
        {
            "author": [
                {
                    "organisation_name": "Constance Watson"
                },
                {
                    "organisation_name": "John Cinnamon"
                }
            ],
            "point of contact": [
                {
                    "name": "MAGIC"
                }
            ],
            "distributor": [
                {
                    "name": "MAGIC"
                }
            ]
        }
        ```

        :rtype dict
        :return: resource contacts keyed by role
        """
        resource_contacts = {}
        for resource_contact in self.config["resource"]["contacts"]:
            for resource_contact_role in resource_contact["role"]:
                if resource_contact_role not in resource_contacts.keys():
                    resource_contacts[resource_contact_role] = []
                resource_contacts[resource_contact_role].append(resource_contact)
        return resource_contacts

    def _filter_resource_keywords(self, keyword_type: str) -> List[dict]:
        """
        Filters resource descriptive keywords by keyword type

        ISO supports multiple keyword type for descriptive keywords (e.g. theme, place), as each type is typically used
        differently, this method filters keywords for a specified type (e.g. only theme keywords).

        Keyword types are defined by the relevant BAS Metadata Library record configuration schema.

        :type keyword_type str
        :param keyword_type: descriptive keyword type
        :rtype list
        :return: list of descriptive keywords of the specified keyword type
        """
        _keywords = []
        for keywords in self.config["resource"]["keywords"]:
            if keywords["type"] == keyword_type:
                _keywords.append(keywords)

        return _keywords

    @property
    def abstract(self) -> str:
        return self.config["resource"]["abstract"]

    @property
    def character_set(self) -> str:
        return self.config["resource"]["character_set"]

    @property
    def contacts(self) -> Dict[str, List[dict]]:
        return self._process_resource_contacts()

    @property
    def dates(self) -> Dict[str, Dict[str, Union[str, date, datetime]]]:
        return self._process_resource_dates()

    @property
    def edition(self) -> str:
        return self.config["resource"]["edition"]

    @property
    def geographic_extent(self) -> Dict:
        return self.config["resource"]["extent"]["geographic"]

    def hierarchy_level(self) -> str:
        return self.config["hierarchy_level"]

    @property
    def language(self) -> str:
        return self.config["resource"]["language"]

    @property
    def lineage(self) -> Optional[str]:
        try:
            return self.config["resource"]["lineage"]
        except KeyError:
            return None

    @property
    def location_keywords(self) -> List[dict]:
        return self._filter_resource_keywords(keyword_type="place")

    @property
    def maintenance_frequency(self) -> str:
        return self.config["resource"]["maintenance"]["maintenance_frequency"]

    @property
    def metadata_character_set(self) -> str:
        return self.config["character_set"]

    @property
    def metadata_language(self) -> str:
        return self.config["language"]

    @property
    def metadata_maintenance_frequency(self) -> str:
        return self.config["maintenance"]["maintenance_frequency"]

    @property
    def metadata_maintenance_progress(self) -> str:
        return self.config["maintenance"]["progress"]

    @property
    def metadata_standard_name(self) -> str:
        return self.config["metadata_standard"]["name"]

    @property
    def metadata_standard_version(self) -> str:
        return self.config["metadata_standard"]["version"]

    @property
    def metadata_updated(self) -> date:
        return self.config["date_stamp"]

    @property
    def spatial_reference_system(self) -> Optional[dict]:
        try:
            return self.config["reference_system_info"]
        except KeyError:  # pragma: no cover (will be addressed in #116)
            return None

    @property
    def spatial_representation_type(self) -> Optional[str]:
        try:
            return self.config["resource"]["spatial_representation_type"]
        except KeyError:  # pragma: no cover (will be addressed in #116)
            return None

    @property
    def theme_keywords(self) -> List[dict]:
        return self._filter_resource_keywords(keyword_type="theme")

    @property
    def temporal_extent(self) -> Dict[str, datetime]:
        return {
            "start": self.config["resource"]["extent"]["temporal"]["period"]["start"],
            "end": self.config["resource"]["extent"]["temporal"]["period"]["end"],
        }

    @property
    def transfer_options(self) -> Optional[List[dict]]:
        try:
            return self.config["resource"]["transfer_options"]
        except KeyError:
            return None

    @property
    def usage_constraints(self) -> Dict[str, dict]:
        return process_usage_constraints(constraints=self.config["resource"]["constraints"]["usage"])

    @property
    def topics(self) -> List[str]:
        return self.config["resource"]["topics"]

    def load(self, record_path: Path) -> None:
        """
        Loads a Record from a file encoded using JSON

        Specifically load a BAS Metadata Library record configuration for ISO 19115-2 that has been JSON encoded.

        :type record_path Path
        :param record_path: path to file containing JSON encoded record configuration
        """
        with open(str(record_path)) as record_file:
            _record_config = json.load(record_file)
            self.config = load_record_from_json(record=_record_config)

    def dump(self, record_path: Path, overwrite: bool = False) -> None:
        """
        Saves a Record to a file encoded using JSON

        Specifically saves a BAS Metadata Library record configuration for ISO 19115-2 using JSON encoding.

        :type record_path Path
        :param record_path: desired path of file that will contain JSON encoded record configuration
        :type overwrite: bool
        :param overwrite: if the desired file already exists, whether to replace its contents
        """
        _record_config = dump_record_to_json(record=self.config)
        try:
            with open(str(record_path), mode="x") as record_file:
                json.dump(_record_config, record_file, indent=4)
        except FileExistsError:
            if not overwrite:
                raise FileExistsError()

            with open(str(record_path), mode="w") as record_file:
                json.dump(_record_config, record_file, indent=4)

    def dumps(self, dump_format: str) -> str:
        """
        Encode a Record in a given format

        Specifically encodes a BAS Metadata Library record configuration for ISO 19115-2 using a specified encoding.

        Currently only the 'xml' format is supported for rendering a record configuration as ISO XML. Others may be
        added in the future as needs arise.

        :type dump_format str
        :param dump_format: format to encode record configuration in
        :rtype str
        :return: encoded record configuration
        """
        if dump_format == "xml":
            return generate_xml_record_from_record_config_without_xml_declaration(record_config=self.config)


class MirrorRecordSummary(Record):
    """
    Mirrored record summaries extend record summaries with a 'published' status, representing whether a record is
    *published* or *unpublished* based on the repositories a record appears within in a mirrored repository.
    """

    def __init__(self, config: dict, published: bool):
        super().__init__(config=config)
        self.published = published

    def __repr__(self):
        return f"<MirrorRecordSummary / {self.identifier} / {'Published' if self.published else 'Unpublished'}>"


class MirrorRecord(MirrorRecordSummary, Record):
    """
    Mirrored records extend mirrored record summaries and records.
    """

    def __init__(self, config: dict, published: bool):
        super().__init__(config=config, published=published)

    def __repr__(self):
        return f"<MirrorRecord / {self.identifier} / {'Published' if self.published else 'Unpublished'}>"


class Repository:
    """
    Represents a data store with an interface for creating, retrieving, updating and deleting Records

    Externally, repositories present an abstracted interface for interacting with records using the Record and
    RecordSummary classes. Internally, repositories are backed by an OGC Catalogue Services for the Web (CSW) catalogue.

    For example:

    * when creating a record - a Record class instance is converted into an ISO 19115-2 record encoded using XML and
      inserted into a CSW server using the CSW transactional profile
    * when retrieving a record - a CSW GetRecord request is made and the XML encoded ISO 19115-2 record is converted
      back into a Record class
    """

    def __init__(self, client_config: dict):
        """
        :type client_config dict
        :param client_config: configuration for the CSWClient class instance that backs this repository
        """
        self.csw_client = CSWClient(config=client_config)

    def retrieve_record(self, record_identifier: str) -> Record:
        """
        Retrieves a record from the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :type record_identifier str
        :param record_identifier: identifier of the record to retrieve
        :rtype Record
        :return: requested record
        """
        record_xml = self.csw_client.get_record(identifier=record_identifier, mode=CSWGetRecordMode.FULL)
        record_config = MetadataRecord(record=record_xml).make_config()
        record_config.validate()
        return Record(config=record_config.config)

    def retrieve_records(self) -> List[Record]:
        """
        Retrieves all records in the repository

        Note: Records are returned using a generator for use in iterators such as for loops. If an actual List of
        records is needed, for calculating a length for example, the return value can be wrapped, e.g.

        ```
        records_count = len(list(repository.retrieve_records()))
        ```

        :rtype list
        :return: all records
        """
        for record_xml in self.csw_client.get_records(mode=CSWGetRecordMode.FULL):
            record_config = MetadataRecord(record=record_xml).make_config()
            record_config.validate()
            yield Record(config=record_config.config)

    def list_record_identifiers(self) -> List[str]:
        """
        Retrieves identifiers for all records in the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :rtype list
        :return: all record identifiers
        """
        return list(self.list_records().keys())

    def list_records(self) -> Dict[str, RecordSummary]:
        """
        Retrieves summaries for all records in the repository

        Records are returned as a dictionary rather than a list to allow specific records to be easily selected.

        :rtype dict
        :return: all summarised records, keyed by record identifier
        """
        _record_summaries = {}
        for record_xml in self.csw_client.get_records(mode=CSWGetRecordMode.BRIEF):
            record_config = MetadataRecord(record=record_xml).make_config()
            record = RecordSummary(config=record_config.config)
            _record_summaries[record.identifier] = record
        return _record_summaries

    def insert_record(self, record: Record, update: bool = False) -> None:
        """
        Creates a new record, or updates an existing record, in the repository

        Records are assumed to be new records by default and will raise an exception if this causes a conflict. Records
        can be updated instead by setting the updated parameter to True.

        :type record Record
        :param record: record to be created or updated
        :type update bool
        :param update: whether an existing record can be overridden
        """
        try:
            record_xml = generate_xml_record_from_record_config_without_xml_declaration(record_config=record.config)
            self.csw_client.insert_record(record=record_xml)
        except RecordInsertConflictException:
            if not update:
                raise RecordInsertConflictException()

            # noinspection PyUnboundLocalVariable
            self.csw_client.update_record(record=record_xml)

    def delete_record(self, record_identifier: str) -> None:
        """
        Deletes a record from the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :type record_identifier str
        :param record_identifier: identifier of the record to delete
        """
        self.csw_client.delete_record(identifier=record_identifier)


class MirrorRepository:
    """
    Represents a composite data store with an interface for creating, retrieving, updating, deleting, publishing and
    retracting Records

    Externally, repositories present an abstracted interface for interacting with records using the MirrorRecord and
    MirrorRecordSummary classes. Internally, mirror repositories are backed by two Repository classes to represent
    'published' and 'unpublished' records.

    If a record exists in both repositories, it is considered published (all records must appear in the unpublished
    repository). If a published record is retracted, it is deleted from the published catalogue, and can then optionally
    also be deleted from the unpublished catalogue (fully deleting the record), or created in the published repository
    again to (re)publish it.
    """

    def __init__(self, unpublished_repository_config: dict, published_repository_config: dict):
        """
        :type unpublished_repository_config dict
        :param unpublished_repository_config: configuration for the unpublished Repository class instance
        :type published_repository_config dict
        :param published_repository_config: configuration for the published Repository class instance
        """
        self.published_repository = Repository(**published_repository_config)
        self.unpublished_repository = Repository(**unpublished_repository_config)

    def retrieve_record(self, record_identifier: str) -> MirrorRecord:
        """
        Retrieves a record from the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        If the record appears in both repositories it will be considered published.

        :type record_identifier str
        :param record_identifier: identifier of the record to retrieve
        :rtype MirrorRecord
        :return: requested record
        """
        try:
            record = self.published_repository.retrieve_record(record_identifier=record_identifier)
            return MirrorRecord(config=record.config, published=True)
        except RecordNotFoundException:
            record = self.unpublished_repository.retrieve_record(record_identifier=record_identifier)
            return MirrorRecord(config=record.config, published=False)

    # retrieve_records() method removed as part of #133/#134

    def retrieve_published_records(self) -> List[MirrorRecord]:
        """
        Retrieves all published records in the repository

        Note: Records are returned using a generator for use in iterators such as for loops. If an actual List of
        records is needed, for calculating a length for example, the return value can be wrapped, e.g.

        ```
        records_count = len(list(repository.retrieve_published_records()))
        ```

        :rtype list
        :return: all published records
        """
        for published_record in self.published_repository.retrieve_records():
            yield MirrorRecord(config=published_record.config, published=True)

    def list_record_identifiers(self) -> List[str]:
        """
        Retrieves identifiers for all records in the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        Note: As all records have to appear in the unpublished repository we can just return it's identifiers using the
        relevant method. The published catalogue's identifiers would only ever be a subset of those.

        :rtype list
        :return: all record identifiers
        """
        return self.unpublished_repository.list_record_identifiers()

    def list_unpublished_record_identifiers(self) -> List[str]:
        """
        Retrieves identifiers for all records in the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        Note: Use the `list_distinct_unpublished_record_identifiers()` method to only return unpublished, rather than
        all, records.

        :rtype list
        :return: all record identifiers
        """
        return self.unpublished_repository.list_record_identifiers()

    def list_published_record_identifiers(self) -> List[str]:
        """
        Retrieves identifiers for all published records in the repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :rtype list
        :return: published record identifiers
        """
        return self.published_repository.list_record_identifiers()

    def list_distinct_unpublished_record_identifiers(self) -> List[str]:
        """
        Retrieves identifiers for all unpublished records in the repository

        This method *only* returns identifiers for records that have not been published.

        I.e. this method returns identifiers for the subset of records that do not appear in the published repository.

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :rtype list
        :return: unpublished record identifiers
        """
        unpublished_record_identifiers = self.list_unpublished_record_identifiers()
        published_record_identifiers = self.list_published_record_identifiers()
        return list(set(unpublished_record_identifiers) - set(published_record_identifiers))

    def list_records(self) -> Dict[str, MirrorRecordSummary]:
        """
        Retrieves summaries for all records in the repository

        Records are returned as a dictionary rather than a list to allow specific records to be easily selected.

        :rtype dict
        :return: all summarised records, keyed by record identifier
        """
        _records = {}

        unpublished_records = self.unpublished_repository.list_records()
        published_record_identifiers = self.published_repository.list_record_identifiers()

        for record_identifier, unpublished_record in unpublished_records.items():
            _record_published = False
            if record_identifier in published_record_identifiers:
                _record_published = True
            _records[record_identifier] = MirrorRecordSummary(
                config=unpublished_record.config, published=_record_published
            )
        return _records

    def insert_record(self, record: Record, update: bool = False) -> None:
        """
        Creates a new, unpublished, record, or updates an existing record in the unpublished repository

        Records are assumed to be new by default and will raise an exception if this causes a conflict. Records can be
        updated instead by setting the updated parameter to True.

        To create or update a record in the published repository (publishing or republishing it) use the
        `publish_record()` method.

        :type record Record
        :param record: record to be created or updated
        :type update bool
        :param update: whether an existing record can be overridden
        """
        self.unpublished_repository.insert_record(record=record, update=update)

    def delete_record(self, record_identifier) -> None:
        """
        Deletes a record from the unpublished repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        Note: As all records must appear in the unpublished repository, if the record exists in the published repository
        an error will be raised and record won't be deleted from the unpublished repository. Use the `retract()` method
        to delete the record from the published repository first.

        :type record_identifier str
        :param record_identifier: identifier of the record to delete
        """
        if record_identifier in self.published_repository.list_record_identifiers():
            raise RecordRetractBeforeDeleteException()

        self.unpublished_repository.delete_record(record_identifier=record_identifier)

    def publish_record(self, record_identifier: str, republish: bool = False) -> None:
        """
        Creates a new, published, record, or updates an existing record in the published repository

        Records are assumed to be new by default and will raise an exception if this causes a conflict. Records can be
        updated (republished) instead by setting the updated parameter to True.

        :type record_identifier str
        :param record_identifier: identifier of the record to publish or republish
        :type republish bool
        :param republish: whether an existing record can be overridden
        """
        try:
            record = self.unpublished_repository.retrieve_record(record_identifier=record_identifier)
        except RecordNotFoundException:
            raise RecordNotFoundException()

        try:
            self.published_repository.insert_record(record=record, update=False)
        except RecordInsertConflictException:
            if not republish:
                raise RecordInsertConflictException()
            self.published_repository.insert_record(record=record, update=True)

    def retract_record(self, record_identifier: str) -> None:
        """
        Deletes (retracts) a record from the published repository

        Record identifiers are the same as ISO 19115-2 file identifiers.

        :type record_identifier str
        :param record_identifier: identifier of the record to delete/retract
        """
        self.published_repository.delete_record(record_identifier=record_identifier)


class Item:
    """
    Items are abstractions of records specific and tailored to the needs of this project.

    Items are read only and use a record internally for exposing information through properties. They are designed to
    provide final output to humans, rather than for onward use or interpretation by other services - use records for
    that.

    Various formatting, processing and filtering methods are used to transform some information to be more easily
    understood or to make more contextual sense.
    """

    def __init__(self, record: Record):
        """
        :type record Record
        :param record: Record item is based on
        """
        self.record = record

    def __repr__(self):
        return f"<Item / {self.identifier}>"

    @staticmethod
    def _format_date(date_datetime: Union[date, datetime], native_precision: str = "day") -> str:
        """
        Format a date for display

        Currently all dates use the default ISO 8601 representation upto their native precision.

        Note: Native precision currently only applies to dates and not times within datetimes - i.e. a datetime with
        seconds will be displayed with seconds even though the native precision may be day.

        For example:

        * a date `<date 2020-04-20 precision=day>` will be formatted as "2020-04-20"
        * a date `<date 2020-01-01 precision=year>` will be formatted as "2020"
        * a datetime `<datetime 2020-04-20T14:30:45 precision=day>` will be formatted as "2020-04-20T14:30:45"
        * a datetime `<datetime 2020-04-20T14:30 precision=day>` will be formatted as "2020-04-20T14:30"

        :type date_datetime date or datetime
        :param date_datetime: date or datetime to be formatted
        :type native_precision str
        :param native_precision: maximum precision of
        :rtype str
        :return: ISO 8601 date or datetime
        """
        if native_precision == "day":
            return date_datetime.isoformat()
        elif native_precision == "year":
            return str(date_datetime.year)

    @staticmethod
    def _format_language(language: str) -> str:
        """
        Format an ISO 19115 language code list value

        Note: It is currently assumed that where English is used this refers to the United Kingdom localisation.

        Note: Other code values can be added as needed in future.

        :type language str
        :param language: ISO 19115 language code list value
        :rtype str
        :return: formatted language name
        """
        if language == "eng":
            return "English (United Kingdom)"

    @staticmethod
    def _format_maintenance_frequency(maintenance_frequency: str) -> str:
        """
        Format an ISO 19115 maintenance frequency code list value

        Note: Other code values can be added as needed in future.

        :type maintenance_frequency str
        :param maintenance_frequency: ISO 19115 maintenance frequency code list value
        :rtype str
        :return: formatted maintenance frequency value
        """
        if maintenance_frequency == "biannually":
            return "Biannually (every 6 months)"
        elif maintenance_frequency == "asNeeded":
            return "As Needed"

    @staticmethod
    def _format_organisation_name(organisation_name: str) -> str:
        """
        Format an organisation name

        Typically this is used to remove redundant information from names, as items will be shown in a BAS branded page
        template it isn't necessary to include 'British Antarctic Survey' in organisation names for example, whereas
        records may be harvested and shown in non-BAS branded websites so that context is needed.

        Names may also be modified to include helpful, but informal, elements such as abbreviations.

        Note: Other organisation names can be added as needed in future

        :type organisation_name str
        :param organisation_name: organisation name
        :rtype str
        :return: formatted organisation name
        """
        if organisation_name == "Mapping and Geographic Information Centre, British Antarctic Survey":
            return "Mapping and Geographic Information Centre (MAGIC)"
        return organisation_name

    @staticmethod
    def _format_keyword_thesaurus_title(thesaurus_title: str) -> str:
        """
        Format the name of a keyword set

        In ISO 19115 keywords that are published by an authority can include a thesaurus that describes what the keyword
        set is, the authority behind them, etc.

        As keywords are shown in a relatively narrow part of the page and titles are often quite verbose, this method
        shortens them into something more suitable, whilst still being easily identifiable.

        Note: Other thesaurus titles can be added as needed in future.

        :type thesaurus_title str
        :param thesaurus_title: title of the keyword set as defined in the keyword thesaurus
        :rtype str
        :return: formatted thesaurus title
        """
        if thesaurus_title == "General Multilingual Environmental Thesaurus - INSPIRE themes":
            return "INSPIRE themes"
        elif thesaurus_title == "Global Change Master Directory (GCMD) Science Keywords":
            return "GCMD Science Keywords"
        elif thesaurus_title == "Global Change Master Directory (GCMD) Location Keywords":
            return "GCMD Location Keywords"

    @staticmethod
    def _format_spatial_reference_system(spatial_reference_system_code: Dict[str, str]) -> str:
        """
        Format a spatial reference system identifier

        Formal identifiers for spatial reference systems (or coordinate/spatial reference systems) are readily
        accessible to those not very familiar with them. This method expands identifiers to include information people
        will understand, if only at a high level (e.g. that it relates to Antarctic rather than the Arctic).

        Wherever possible URIs are used to match identifiers to avoid ambiguity with how they are referenced as codes.

        Note: Other reference systems can be added as needed in the future.

        :type spatial_reference_system_code dict
        :param spatial_reference_system_code: spatial reference system containing a href property with an identifier URI
        :rtype str
        :return: formatted spatial reference system identifier, including markdown links
        """
        if spatial_reference_system_code["href"] == "http://www.opengis.net/def/crs/EPSG/0/3031":
            return "WGS 84 / Antarctic Polar Stereographic ([EPSG:3031](https://spatialreference.org/ref/epsg/3031/))"

    @staticmethod
    def _process_bounding_box_geojson(bounding_box: Dict[str, str]) -> Dict:
        """
        Constructs a GeoJSON bounding box for an items spatial extent

        By default this method uses a top-left and bottom-right pair of coordinates to define a box indicating the
        extent of an item.

        However, where a dataset covers all of Antarctica, a bounding box and corners no longer make sense (either to
        humans or to mapping libraries). To workaround this, the WellKnownExtents class is used for regions that should
        be treated differently. For Antarctica for example a polygon is used to effectively define a bounding radius
        from the South pole that covers the Antarctic continent.

        This method, and the WellKnownExtents class can be expanded as needed to accommodate other areas where a
        bounding box feature is unsuitable.

        :type bounding_box dict
        :param bounding_box: bounding box as a set of maximum/minimum latitude/longitude using WGS 84
        :rtype dict
        :return: GeoJSON feature representing a bounding area (box, circle, polygon, etc.)
        """
        bounding_polygon = {
            "type": "FeatureCollection",
            "name": "bounding_polygon",
            "crs": {"type": "name", "properties": {"name": "urn:ogc:def:crs:EPSG::4326"}},
            "features": [
                {
                    "type": "Feature",
                    "properties": {"gid": 1, "label": "Bounding polygon"},
                    "geometry": {
                        "type": "Polygon",
                        "coordinates": [
                            [
                                [bounding_box["west_longitude"], bounding_box["south_latitude"]],
                                [bounding_box["east_longitude"], bounding_box["south_latitude"]],
                                [bounding_box["east_longitude"], bounding_box["north_latitude"]],
                                [bounding_box["west_longitude"], bounding_box["north_latitude"]],
                                [bounding_box["west_longitude"], bounding_box["south_latitude"]],
                            ],
                        ],
                    },
                }
            ],
        }

        if (
            bounding_box["east_longitude"] == 180
            and bounding_box["north_latitude"] == -60
            and bounding_box["south_latitude"] == -90
            and bounding_box["west_longitude"] == -180
        ):
            bounding_polygon["crs"]["properties"][
                "name"
            ] = "urn:ogc:def:crs:EPSG::3031"  # pragma: no cover (will be addressed in #116)
            bounding_polygon["features"][0]["geometry"] = WellKnownExtents.ANTARCTICA.value
        elif (  # pragma: no cover (will be addressed in #116)
            bounding_box["east_longitude"] == 180
            and bounding_box["north_latitude"] == -50
            and bounding_box["south_latitude"] == -60
            and bounding_box["west_longitude"] == -180
        ):
            bounding_polygon["crs"]["properties"][
                "name"
            ] = "urn:ogc:def:crs:EPSG::3031"  # pragma: no cover (will be addressed in #116)
            bounding_polygon["features"][0][
                "geometry"
            ] = WellKnownExtents.SUB_ANTARCTICA.value  # pragma: no cover (will be addressed in #116)

        return bounding_polygon

    @staticmethod
    def _process_status(maintenance_frequency: str, released_date: Union[date, datetime]):
        """
        Determines whether an item is current or outdated/superseded etc

        Supported maintenance frequencies are:

        * biannual (periodic)
        * as-needed (non-periodic)

        Possible statuses are:

        * current
        * outdated

        Maintenance frequencies are based on ISO 19115 maintenance frequency code list values.

        Where an item has a periodic maintenance frequency (e.g. every month), an overdue date is calculated by adding
        the maintenance frequency to the items release date. If the current date is less or equal to this overdue date
        it's considered current, otherwise it's deemed to be outdated.

        Note: Other status values can be added as needed in future.

        Note: This method includes code paths that are not currently used and so are exempt from code coverage. These
        paths are known to be needed in future and will therefore be tested and included in coverage in the future.

        :type maintenance_frequency str
        :param maintenance_frequency: maintenance frequency code list value
        :type released_date date or datetime
        :param released_date: item release date
        :rtype str
        :return: item status
        """
        if maintenance_frequency == "asNeeded":  # pragma: no cover (added for future use)
            return "current"

        _now = datetime.today()
        _overdue = released_date
        if type(released_date) == date:  # pragma: no cover (added for future use)
            _now = _now.date()
        if maintenance_frequency == "biannually":
            _overdue += relativedelta(months=+6)

        if _now <= _overdue:
            return "current"  # pragma: no cover (will be addressed in #116)
        return "outdated"  # pragma: no cover (added for future use)

    @staticmethod
    def _process_download(transfer_option: dict) -> Dict[str, str]:
        """
        Generate an abstracted dataset download option

        Transforms a ISO 19115 transfer option into an abstracted download option intended specifically for the item
        page template. It remaps elements such as size sizes and file names which are currently used to infer the data
        format using conventional names. Special support is included for OGC WMS URIs to extract elements such as base
        endpoint and layer.

        In future, ISO transfer options will be associated to an distribution format removing the need to infer data
        formats from conventional file names.

        :type transfer_option dict
        :param transfer_option: ISO transfer option
        :rtype dict
        :return: item download option
        """
        download = {
            # Exempting Bandit security issue (weak hash method), not used for security/cryptography
            "id": sha1(transfer_option["online_resource"]["href"].encode("utf-8")).hexdigest(),  # nosec
            "format": None,
            "format_title": None,
            "format_description": None,
            "format_version": None,
            "size": None,
            "url": transfer_option["online_resource"]["href"],
        }

        if transfer_option["online_resource"]["title"] == "GeoPackage":
            download["format"] = "gpkg"
            download["format_title"] = "GeoPackage"
            download["format_description"] = "OGC GeoPackage"
            download["format_version"] = "1.2"
        elif transfer_option["online_resource"]["title"] == "Shapefile":
            download["format"] = "shp"
            download["format_title"] = "Shapefile"
            download["format_description"] = "ESRI Shapefile"
            download["format_version"] = "1"
        elif transfer_option["online_resource"]["title"] == "Web Map Service (WMS)":
            download["format"] = "wms"
            download["format_title"] = "Web Map Service (WMS)"
            download["format_description"] = "OGC Web Map Service"
        elif transfer_option["online_resource"]["title"] == "PNG":  # pragma: no cover (will be addressed in #116)
            download["format"] = "png"
            download["format_title"] = "PNG"
            download["format_description"] = "PNG image"
            download["format_version"] = "1"
        elif transfer_option["online_resource"]["title"] == "JPEG":  # pragma: no cover (will be addressed in #116)
            download["format"] = "jpeg"
            download["format_title"] = "JPEG"
            download["format_description"] = "JPEG image"
            download["format_version"] = "1"
        elif transfer_option["online_resource"]["title"] == "PDF":  # pragma: no cover (will be addressed in #116)
            download["format"] = "pdf"
            download["format_title"] = "PDF"
            download["format_description"] = "Adobe PDF"
            download["format_version"] = "1.6"

        if "size" in transfer_option.keys():
            download["size"] = f"{transfer_option['size']['magnitude']} {transfer_option['size']['unit']}"

        if download["format"] == "wms":
            url_parsed = url_parse(download["url"])
            url_query_string: Dict[str, List[str]] = query_string_parse(url_parsed.query)
            download["endpoint"] = f"{url_parsed.scheme}://{url_parsed.netloc}{url_parsed.path}"
            download["layer"] = url_query_string["layer"][0]

        return download

    @staticmethod
    def _filter_keyword_terms(keyword_sets: List[dict], keyword_set_url: str) -> List[dict]:
        """
        Filter a specific keyword set from a collection of keyword sets based on the keyword set's URI

        :type keyword_sets dict
        :param keyword_sets: collection of keyword sets to filter
        :type keyword_set_url str
        :param keyword_set_url: URI of keyword set to filter out of the collection of keyword sets
        :rtype dict
        :return: filtered keyword set
        """
        for keyword_set in keyword_sets:
            if keyword_set["thesaurus"]["title"]["href"] == keyword_set_url:
                return keyword_set["terms"]

    @property
    def abstract(self) -> str:
        return self.record.abstract

    @property
    def abstract_markdown(self) -> str:
        return markdown(self.abstract, output_format="html5")

    @property
    def authors(self) -> List[dict]:
        return self.record.contacts["author"]

    @property
    def character_set(self) -> str:
        return str(self.record.character_set).upper()

    @property
    def citation(self) -> str:  # pragma: no cover (will be addressed in #116)
        return self.record.usage_constraints["required_citation"]["statement"]

    @property
    def citation_markdown(self) -> str:
        return markdown(self.record.usage_constraints["required_citation"]["statement"], output_format="html5")

    @property
    def collections(self) -> Optional[List[str]]:
        """
        Item's Collections

        Collections are implemented as a descriptive keyword set using the NERC Vocabulary Service (T02).

        Currently only collection names to show as free text values are returned. In future, collection IDs will be
        returned to allow linking items to their collections.

        :rtype list
        :return: Collection names
        """
        collection_terms = self._filter_keyword_terms(
            keyword_sets=self.record.theme_keywords, keyword_set_url="http://vocab.nerc.ac.uk/collection/T02/1/"
        )
        if collection_terms is None:  # pragma: no cover (will be addressed in #116)
            return []

        # return a list of just term values
        return [term["term"] for term in collection_terms]

    @property
    def created(self) -> str:
        _date = self.record.dates["creation"]
        return self._format_date(date_datetime=_date["value"], native_precision=_date["precision"])

    @property
    def data_type(self) -> str:
        return self.record.spatial_representation_type

    @property
    def downloads(self) -> List[Dict[str, str]]:
        downloads = []

        if self.record.transfer_options is None:  # pragma: no cover (will be addressed in #116)
            return downloads

        for transfer_option in self.record.transfer_options:
            downloads.append(self._process_download(transfer_option=transfer_option))
        return downloads

    @property
    def edition(self) -> str:
        return self.record.edition

    @property
    def geographic_extent(self) -> Dict:
        geographic_extent = self.record.geographic_extent
        geographic_extent["bounding_box_geojson"] = self._process_bounding_box_geojson(
            bounding_box=geographic_extent["bounding_box"]
        )
        return geographic_extent

    @property
    def identifier(self) -> str:
        return self.record.identifier

    @property
    def item_type(self) -> str:
        return self.record.hierarchy_level()

    @property
    def language(self) -> str:
        return self._format_language(language=self.record.language)

    @property
    def licence(self) -> dict:
        return self.record.usage_constraints["copyright_licence"]

    @property
    def lineage(self) -> str:
        return self.record.lineage

    @property
    def lineage_markdown(self) -> str:
        return markdown(self.lineage, output_format="html5")

    @property
    def location_keywords(self) -> List[dict]:
        location_keywords = self.record.location_keywords
        for location_keyword in location_keywords:
            location_keyword["thesaurus"]["title"]["value"] = self._format_keyword_thesaurus_title(
                thesaurus_title=location_keyword["thesaurus"]["title"]["value"]
            )
        return location_keywords

    @property
    def maintenance_frequency(self) -> str:
        return self._format_maintenance_frequency(maintenance_frequency=self.record.maintenance_frequency)

    @property
    def metadata_maintenance_progress(self) -> str:
        return str(self.record.metadata_maintenance_progress).capitalize()

    @property
    def metadata_character_set(self) -> str:
        return str(self.record.metadata_character_set).upper()

    @property
    def metadata_language(self) -> str:
        return self._format_language(language=self.record.metadata_language)

    @property
    def metadata_maintenance_frequency(self) -> str:
        return self._format_maintenance_frequency(maintenance_frequency=self.record.metadata_maintenance_frequency)

    @property
    def metadata_standard_name(self) -> str:
        return self.record.metadata_standard_name

    @property
    def metadata_standard_version(self) -> str:
        return self.record.metadata_standard_version

    @property
    def metadata_updated(self) -> str:
        return self._format_date(date_datetime=self.record.metadata_updated, native_precision="day")

    @property
    def released(self) -> str:
        _date = self.record.dates["released"]
        return self._format_date(date_datetime=_date["value"], native_precision=_date["precision"])

    @property
    def point_of_contact(self) -> str:
        points_of_contact = self.record.contacts["pointOfContact"]
        point_of_contact = points_of_contact[0]
        return self._format_organisation_name(organisation_name=point_of_contact["organisation"]["name"])

    @property
    def point_of_contact_details(self) -> dict:
        points_of_contact = self.record.contacts["pointOfContact"]
        point_of_contact = points_of_contact[0]
        point_of_contact["organisation"]["name"] = self._format_organisation_name(
            organisation_name=point_of_contact["organisation"]["name"]
        )
        return point_of_contact

    @property
    def published(self) -> str:
        _date = self.record.dates["publication"]
        return self._format_date(date_datetime=_date["value"], native_precision=_date["precision"])

    @property
    def temporal_extent(self) -> Dict[str, str]:
        return {
            "start": self._format_date(date_datetime=self.record.temporal_extent["start"], native_precision="day"),
            "end": self._format_date(date_datetime=self.record.temporal_extent["end"], native_precision="day"),
        }

    @property
    def spatial_reference_system(self) -> Optional[str]:
        if self.record.spatial_reference_system is None:
            return None  # pragma: no cover (will be addressed in #116)

        return self._format_spatial_reference_system(
            spatial_reference_system_code=self.record.spatial_reference_system["code"]
        )

    @property
    def spatial_reference_system_markdown(self) -> Optional[str]:
        if self.spatial_reference_system is None:
            return None  # pragma: no cover (will be addressed in #116)

        return markdown(self.spatial_reference_system, output_format="html5")

    @property
    def status(self) -> str:
        return self._process_status(
            maintenance_frequency=self.record.maintenance_frequency,
            released_date=self.record.dates["released"]["value"],
        )

    @property
    def theme_keywords(self) -> List[dict]:
        """
        Theme keywords (filtered)

        Theme keywords are currently used for two keyword sets that the catalogue treats differently:

        * BAS research topics - http://vocab.nerc.ac.uk/collection/T01/1/
        * Data catalogue collections - http://vocab.nerc.ac.uk/collection/T02/1/

        As these keyword sets are exposed through other Item properties (collections and topics respectively), they are
        filtered out from other theme keyword sets. Additionally, ISO Topics are filtered in as a theme keyword set.

        :rtype list
        :return: theme keyword sets, exc. BAS research topics and data catalogue collections, inc. ISO topics
        """
        theme_keywords = []
        excluded_keyword_sets = [
            "http://vocab.nerc.ac.uk/collection/T01/1/",
            "http://vocab.nerc.ac.uk/collection/T02/1/",
        ]

        _iso_topics_keyword_terms = []
        for iso_topic in self.record.topics:
            _iso_topics_keyword_terms.append({"term": iso_topic})
        _iso_topics_keyword_set = {"terms": _iso_topics_keyword_terms, "thesaurus": {"title": {"value": "ISO Topics"}}}
        theme_keywords.append(_iso_topics_keyword_set)

        _theme_keywords = self.record.theme_keywords
        for theme_keyword in _theme_keywords:
            if theme_keyword["thesaurus"]["title"]["href"] in excluded_keyword_sets:
                continue

            theme_keyword["thesaurus"]["title"]["value"] = self._format_keyword_thesaurus_title(
                thesaurus_title=theme_keyword["thesaurus"]["title"]["value"]
            )
            theme_keywords.append(theme_keyword)

        return theme_keywords

    @property
    def title(self) -> str:
        return self.record.title

    @property
    def title_markdown(self) -> str:
        return markdown(self.title, output_format="html5")

    @property
    def topics(self) -> List[str]:
        """
        Item's research topics

        Research topics are implemented as a descriptive keyword set using the NERC Vocabulary Service (T01).

        Currently only topic names to show as free text values are returned. In future, topic IDs will be returned to
        allow linking items to their collections.

        :rtype list
        :return: Topic names
        """
        topic_terms = self._filter_keyword_terms(
            keyword_sets=self.record.theme_keywords, keyword_set_url="http://vocab.nerc.ac.uk/collection/T01/1/"
        )
        # return a list of just term values
        return [term["term"] for term in topic_terms]

    @property
    def updated(self) -> Optional[str]:
        try:
            _date = self.record.dates["revision"]
            return self._format_date(date_datetime=_date["value"], native_precision=_date["precision"])
        except KeyError:  # pragma: no cover (will be addressed in #116)
            return None


class Collection:
    """
    Collections represent an unstructured set of Items that are somehow related. Collections are independent of other
    grouping mechanisms, such as descriptive keywords, dataset series and aggregations, publishers and/or any other
    common properties.

    Collections only exist within the data catalogue and can be used to relate any set of items together by including
    the relevant collection identifier as a descriptive keyword in Records (that underpin Items). Currently items in
    collections also need to be defined directly in the collection definitions file (`collections.json`).

    See the project README for Collection configurations properties.
    """

    def __init__(self, config: dict = None):
        self.config = {}
        if self.config is not None:
            self.config = config

    def __repr__(self):
        return f"<Collection / {self.identifier}>"

    @property
    def identifier(self) -> str:
        return self.config["identifier"]

    @property
    def title(self) -> str:
        return self.config["title"]

    @property
    def title_markdown(self) -> str:
        return markdown(self.title, output_format="html5")

    @property
    def topics(self) -> List[str]:
        return self.config["topics"]

    @property
    def summary(self) -> str:
        return self.config["summary"]

    @property
    def summary_markdown(self) -> str:
        # noinspection PyTypeChecker
        return markdown(self.summary, output_format="html5")

    @property
    def item_identifiers(self) -> List[str]:
        return self.config["item_identifiers"]

    def load(self, collection_path: Path) -> None:
        """
        Loads a Collection from a file encoded using JSON

        :type collection_path Path
        :param collection_path: path to file containing JSON encoded record configuration
        """
        with open(str(collection_path)) as collection_file:
            self.config = json.load(collection_file)

    def dump(self, collection_path: Path, overwrite: bool = False) -> None:
        """
        Saves a Collection to a file encoded using JSON

        :type collection_path Path
        :param collection_path: desired path of file that will contain JSON encoded record configuration
        :type overwrite: bool
        :param overwrite: if the desired file already exists, whether to replace its contents
        """
        try:
            with open(str(collection_path), mode="x") as collection_file:
                json.dump(self.config, collection_file, indent=4)
        except FileExistsError:
            if not overwrite:
                raise FileExistsError()

            with open(str(collection_path), mode="w") as collection_file:
                json.dump(self.config, collection_file, indent=4)

    def dumps(self) -> Dict:
        """
        Return a collections configuration

        :rtype dict
        :return: collection configuration
        """
        return self.config


class Collections:
    """
    Represents a data store with an interface for creating, retrieving, updating and deleting Collections

    Collections use a file to encode a series of Collection configurations with methods to interact with this file and
    the configurations it contains.
    """

    def __init__(self, config: dict):
        """
        :type config dict
        :param config: Collections configuration, consisting of a single parameter for the path to a collections file
        """
        self.collections = {}
        self.collections_path = None
        if "collections_path" in config.keys():
            self.collections_path = config["collections_path"]

        try:
            with open(str(self.collections_path), mode="r") as collections_file:  # pragma: no cover
                collections_data = json.load(collections_file)
                self.collections = collections_data
        except FileNotFoundError:  # pragma: no cover
            # Ignore because the collections file hasn't been setup yet
            pass

    def get_all(self) -> List[Collection]:
        """
        Retrieve all Collections

        :rtype list
        :return: all Collections
        """
        _collections = []
        for collection_identifier in self.collections.keys():
            _collections.append(self.get(collection_identifier=collection_identifier))
        return _collections

    def get(self, collection_identifier: str) -> Collection:
        """
        Retrieve specific Collection specified by its identifier

        :rtype Collection
        :return: specified Collection
        """
        try:
            return Collection(config=self.collections[collection_identifier])
        except KeyError:
            raise CollectionNotFoundException()

    def add(self, collection: Collection) -> None:
        """
        Create a new Collections

        :type collection Collection
        :param collection: Collection to create
        """
        if collection.identifier in self.collections.keys():
            raise CollectionInsertConflictException

        self.update(collection=collection)

    def update(self, collection: Collection) -> None:  # pragma: no cover (method mocked in testing)
        """
        Update an existing Collection

        :type collection Collection
        :param collection: Collection to update
        """
        self.collections[collection.identifier] = collection.dumps()

        with open(str(self.collections_path), mode="w") as collections_file:
            json.dump(self.collections, collections_file, indent=4)

    def delete(self, collection_identifier: str) -> None:  # pragma: no cover (method mocked in testing)
        """
        Delete a specific Collection specified by its identifier

        :rtype str
        :return: Collection identifier
        """
        del self.collections[collection_identifier]

        with open(str(self.collections_path), mode="w") as collections_file:
            json.dump(self.collections, collections_file, indent=4)
