from odoo.addons.base_rest import restapi
from odoo.addons.component.core import Component
from odoo.exceptions import UserError

from ..pydantic_models.bank_account import BankAccountOut
from ..pydantic_models.info import Country, PersonType, State
from ..pydantic_models.user import UserIn, UserOut, UserShort
from ..pydantic_models import false_to_none


class UserService(Component):
    _inherit = 'base.rest.service'
    _name = 'user.service'
    _usage = 'user'
    _collection = 'photovoltaic_api.services'


    @restapi.method(
        [(['/'], 'GET')],
        output_param=restapi.PydanticModel(UserOut)
    )
    def get(self):
        return self._to_pydantic(self.env.user.partner_id)

    @restapi.method(
        [(['/'], 'PUT')],
        input_param=restapi.PydanticModel(UserIn),
        output_param=restapi.PydanticModel(UserOut)
    )
    def update(self, user_in):
        if user_in.vat and self.env['res.partner'].search([('vat', '=', user_in.vat)]) != self.env.user.partner_id:
            raise UserError('vat already exists')

        user_dict = user_in.dict(exclude_unset=True, exclude={'representative', 'interests'})

        partner = self.env.user.partner_id
        is_login_vat = (partner.vat == self.env.user.login)
        partner.write(user_dict)

        if (partner.company_type == 'company' and partner.child_ids and user_in.representative):
            representative = partner.child_ids[0]
            representative.write(user_in.representative.dict(exclude_unset=True))

        if (user_in.interests):
            interest_ids = [self.env['res.partner.interest'].name_search(i)[0][0] for i in user_in.interests]
            partner.write({'interest_ids': [(6, 0, interest_ids)]})

        if is_login_vat:
            self.env.user.sudo().write({'login': partner.vat})

        return self._to_pydantic(partner)

    #Private methods
    def _to_pydantic(self, user):

        representative = None
        if (user.company_type == 'company' and user.child_ids):
            representative = UserShort.from_orm(user.child_ids[0])
        
        return UserOut.parse_obj({
            'id': user.id,
            'person_type': user.company_type,
            'firstname': user.firstname,
            'lastname': user.lastname,
            'street': user.street,
            'additional_street': false_to_none(user, 'street2'),
            'zip': user.zip,
            'city': user.city,
            'state': State.from_orm(user.state_id),
            'country': Country.from_orm(user.country_id),
            'email': user.email,
            'phone': user.phone,
            'alias': false_to_none(user, 'alias'),
            'vat': user.vat,
            'gender': false_to_none(user, 'gender_partner'),
            'birthday': false_to_none(user, 'birthday'),
            # Omit dummy accounts with acc_number 'CRECE SOLAR' since they are for internal use
            'bank_accounts': [BankAccountOut.from_orm(a) for a in user.bank_ids if 'CRECE SOLAR' not in a.acc_number] if user.bank_ids else [],
            'representative': representative,
            'about_us': false_to_none(user, 'about_us'),
            'interests': user.interest_ids.mapped(lambda i: i.name) if user.interest_ids else []
        })
